package ru.yandex.solomon.gateway.api.v2.dto;

import java.time.Instant;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.core.db.model.Agent;
import ru.yandex.solomon.core.exceptions.BadRequestException;

/**
 * @author msherbakov
 */

@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class AgentDto {
    private static final int DEFAULT_INTERVAL_SECONDS = 15;
    private static final int MIN_INTERVAL_SECONDS = 5;
    private static final int MAX_INTERVAL_SECONDS = 300;

    @JsonProperty
    public String provider;
    @JsonProperty
    public String hostname;
    @JsonProperty
    public int dataPort;
    @JsonProperty
    public int managementPort;
    @JsonProperty
    public String version;
    @JsonProperty
    public String description;
    @JsonProperty
    public Instant lastSeen;
    @JsonProperty
    public int pullIntervalSeconds;

    @JsonIgnore
    public void validate() {
        if (dataPort > 0xffff || dataPort <= 0) {
            throw new BadRequestException("data port is out of range: " + dataPort);
        }

        if (managementPort > 0xffff || managementPort < 0) {
            throw new BadRequestException("management port is out of range: " + managementPort);
        }

        if (StringUtils.isBlank(provider)) {
            throw new BadRequestException("provider cannot be blank");
        }

        if (StringUtils.isBlank(hostname)) {
            throw new BadRequestException("hostname cannot be blank");
        }

        if (pullIntervalSeconds == 0) {
            pullIntervalSeconds = DEFAULT_INTERVAL_SECONDS;
        } else if (pullIntervalSeconds < MIN_INTERVAL_SECONDS || pullIntervalSeconds > MAX_INTERVAL_SECONDS) {
            throw new BadRequestException("Interval must be in range [" + MIN_INTERVAL_SECONDS + ", " + MAX_INTERVAL_SECONDS +  "], but is " + pullIntervalSeconds);
        }
    }

    public static Agent toModel(@Nonnull AgentDto dto) {
        return Agent.newBuilder()
                .setProvider(dto.provider)
                .setHostname(dto.hostname)
                .setDataPort(dto.dataPort)
                .setManagementPort(dto.managementPort)
                .setVersion(dto.version)
                .setLastSeen(dto.lastSeen)
                .setDescription(dto.description)
                .setPullIntervalSeconds(dto.pullIntervalSeconds)
                .build();
    }

    public static AgentDto fromModel(@Nonnull Agent agent) {
        AgentDto dto = new AgentDto();
        dto.hostname = agent.getHostname();
        dto.provider = agent.getProvider();
        dto.dataPort = agent.getDataPort();
        dto.managementPort = agent.getManagementPort();
        dto.version = agent.getVersion();
        dto.lastSeen = agent.getLastSeen();
        dto.description = agent.getDescription();
        dto.pullIntervalSeconds = agent.getIntervalSeconds();
        return dto;
    }
}
