package ru.yandex.solomon.gateway.api.v2.dto;

import java.time.Instant;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.validators.IdValidator;
import ru.yandex.solomon.gateway.api.v2.dto.ClusterHosts.CloudDnsDto;
import ru.yandex.solomon.gateway.api.v2.dto.ClusterHosts.ConductorGroupDto;
import ru.yandex.solomon.gateway.api.v2.dto.ClusterHosts.ConductorTagDto;
import ru.yandex.solomon.gateway.api.v2.dto.ClusterHosts.InstanceGroupDto;
import ru.yandex.solomon.gateway.api.v2.dto.ClusterHosts.NannyGroupDto;
import ru.yandex.solomon.gateway.api.v2.dto.ClusterHosts.NetworkDto;
import ru.yandex.solomon.gateway.api.v2.dto.ClusterHosts.PatternDto;
import ru.yandex.solomon.gateway.api.v2.dto.ClusterHosts.QloudGroupDto;
import ru.yandex.solomon.gateway.api.v2.dto.ClusterHosts.UrlDto;
import ru.yandex.solomon.gateway.api.v2.dto.ClusterHosts.YpDto;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.util.net.NetworkValidator;

import static ru.yandex.solomon.core.db.model.ShardSettings.AggregationSettings.EMPTY;

/**
 * @author Sergey Polovko
 */
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown=true)
public class ClusterDto {

    private String id;
    private String name;
    private String projectId;

    private PatternDto[] hosts;
    private UrlDto[] hostUrls;
    private ConductorGroupDto[] conductorGroups;
    private ConductorTagDto[] conductorTags;
    private NannyGroupDto[] nannyGroups;
    private QloudGroupDto[] qloudGroups;
    private NetworkDto[] networks;
    private YpDto[] ypClusters;
    private InstanceGroupDto[] instanceGroups;
    private CloudDnsDto[] cloudDns;

    private Instant deleteAfter;
    private Integer sensorsTtlDays;
    private Integer port;
    private Boolean useFqdn;
    private String tvmDestId;

    private Integer version;

    private Instant createdAt;
    private Instant updatedAt;
    private String createdBy;
    private String updatedBy;

    private Integer hostGroupsCount;

    public Integer getHostGroupsCount() {
        return hostGroupsCount;
    }

    public void setHostGroupsCount(Integer hostGroupsCount) {
        this.hostGroupsCount = hostGroupsCount;
    }

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getProjectId() {
        return projectId;
    }

    public void setProjectId(String projectId) {
        this.projectId = projectId;
    }

    public PatternDto[] getHosts() {
        return hosts;
    }

    public void setHosts(PatternDto[] hosts) {
        this.hosts = hosts;
    }

    public UrlDto[] getHostUrls() {
        return hostUrls;
    }

    public void setHostUrls(UrlDto[] hostUrls) {
        this.hostUrls = hostUrls;
    }

    public ConductorGroupDto[] getConductorGroups() {
        return conductorGroups;
    }

    public ConductorTagDto[] getConductorTags() {
        return conductorTags;
    }

    public void setConductorGroups(ConductorGroupDto[] conductorGroups) {
        this.conductorGroups = conductorGroups;
    }

    public void setConductorTags(ConductorTagDto[] conductorTags) {
        this.conductorTags = conductorTags;
    }

    public NannyGroupDto[] getNannyGroups() {
        return nannyGroups;
    }

    public void setNannyGroups(NannyGroupDto[] nannyGroups) {
        this.nannyGroups = nannyGroups;
    }

    public QloudGroupDto[] getQloudGroups() {
        return qloudGroups;
    }

    public void setQloudGroups(QloudGroupDto[] qloudGroups) {
        this.qloudGroups = qloudGroups;
    }

    public NetworkDto[] getNetworks() {
        return networks;
    }

    public void setNetworks(NetworkDto[] networks) {
        this.networks = networks;
    }

    public void setYpClusters(YpDto[] ypClusters) {
        this.ypClusters = ypClusters;
    }

    public YpDto[] getYpClusters() {
        return ypClusters;
    }

    public InstanceGroupDto[] getInstanceGroups() {
        return instanceGroups;
    }

    public void setInstanceGroups(InstanceGroupDto[] instanceGroups) {
        this.instanceGroups = instanceGroups;
    }

    public CloudDnsDto[] getCloudDns() {
        return cloudDns;
    }

    public void setCloudDns(CloudDnsDto[] cloudDns) {
        this.cloudDns = cloudDns;
    }

    public Instant getDeleteAfter() {
        return deleteAfter;
    }

    public void setDeleteAfter(Instant deleteAfter) {
        this.deleteAfter = deleteAfter;
    }

    public Integer getSensorsTtlDays() {
        return sensorsTtlDays;
    }

    public void setSensorsTtlDays(Integer sensorsTtlDays) {
        this.sensorsTtlDays = sensorsTtlDays;
    }

    public Integer getPort() {
        return port;
    }

    public void setPort(Integer port) {
        this.port = port;
    }

    public void setUseFqdn(Boolean useFqdn) {
        this.useFqdn = useFqdn;
    }

    public Boolean getUseFqdn() {
        return useFqdn;
    }

    public String getTvmDestId() { return tvmDestId; }

    public Integer getVersion() {
        return version;
    }

    public void setVersion(Integer version) {
        this.version = version;
    }

    public Instant getCreatedAt() {
        return createdAt;
    }

    public void setCreatedAt(Instant createdAt) {
        this.createdAt = createdAt;
    }

    public Instant getUpdatedAt() {
        return updatedAt;
    }

    public void setUpdatedAt(Instant updatedAt) {
        this.updatedAt = updatedAt;
    }

    public String getCreatedBy() {
        return createdBy;
    }

    public void setCreatedBy(String createdBy) {
        this.createdBy = createdBy;
    }

    public String getUpdatedBy() {
        return updatedBy;
    }

    public void setUpdatedBy(String updatedBy) {
        this.updatedBy = updatedBy;
    }

    public void validate(NetworkValidator validator) {
        IdValidator.ensureClusterIdValid(id, "cluster");
        if (StringUtils.isBlank(name)) {
            throw new BadRequestException("name cannot be blank");
        }
        if (StringUtils.isBlank(projectId)) {
            throw new BadRequestException("projectId cannot be blank");
        }
        if (hosts != null) {
            for (PatternDto host : hosts) {
                host.validate();
            }
        }
        if (hostUrls != null) {
            for (UrlDto hostUrl : hostUrls) {
                hostUrl.validate();
            }
        }
        if (conductorGroups != null) {
            for (ConductorGroupDto group : conductorGroups) {
                group.validate();
            }
        }
        if (conductorTags != null) {
            for (ConductorTagDto tag : conductorTags) {
                tag.validate();
            }
        }
        if (nannyGroups != null) {
            for (NannyGroupDto group : nannyGroups) {
                group.validate();
            }
        }
        if (qloudGroups != null) {
            for (QloudGroupDto group : qloudGroups) {
                group.validate();
            }
        }
        if (networks != null) {
            try {
                for (NetworkDto network : networks) {
                    network.validate(validator);
                }
            } catch (BadRequestException e) {
                throw e;
            } catch (Exception e) {
                throw new BadRequestException(e.getMessage());
            }
        }
        if (ypClusters != null) {
            for (YpDto yp : ypClusters) {
                yp.validate();
            }
        }
        if (instanceGroups != null) {
            for (InstanceGroupDto group : instanceGroups) {
                group.validate();
            }
        }
        if (cloudDns != null) {
            for (CloudDnsDto dto : cloudDns) {
                dto.validate();
            }
        }
        if (port != null && (port < 0 || port > 0xffff)) {
            throw new BadRequestException("port is out of range: " + port);
        }
        if (StringUtils.isNotEmpty(tvmDestId) && !NumberUtils.isParsable(tvmDestId)) {
            throw new BadRequestException("tvmDestId must be a valid integer number, but got: '" + tvmDestId + '\'');
        }
        if (deleteAfter != null && !deleteAfter.equals(Instant.EPOCH) && Instant.now().isAfter(deleteAfter)) {
            throw new BadRequestException("deleteAfter must be in our bright future");
        }
        if (version != null && version < 0) {
            throw new BadRequestException("version cannot be negative");
        }
    }

    @Nonnull
    public static Cluster toModel(@Nonnull ClusterDto dto) {
        return toModel(dto, "");
    }

    @Nonnull
    public static Cluster toModel(@Nonnull ClusterDto dto, @Nonnull String folderId) {
        var settings = getShardSettings(dto);
        return new Cluster(
            dto.id,
            dto.name,
            dto.projectId,
            folderId,
            "",
            toModelList(dto.hosts, ClusterHosts.EMPTY_HOSTS, PatternDto::toModel),
            toModelList(dto.hostUrls, ClusterHosts.EMPTY_HOST_URLS, UrlDto::toModel),
            toModelList(dto.conductorGroups, ClusterHosts.EMPTY_CONDUCTOR_GROUPS, ConductorGroupDto::toModel),
            toModelList(dto.conductorTags, ClusterHosts.EMPTY_CONDUCTOR_TAGS, ConductorTagDto::toModel),
            toModelList(dto.nannyGroups, ClusterHosts.EMPTY_NANNY_GROUPS, NannyGroupDto::toModel),
            toModelList(dto.qloudGroups, ClusterHosts.EMPTY_QLOUD_GROUPS, QloudGroupDto::toModel),
            toModelList(dto.networks, ClusterHosts.EMPTY_NETWORK, NetworkDto::toModel),
            toModelList(dto.ypClusters, ClusterHosts.EMPTY_YP, YpDto::toModel),
            toModelList(dto.instanceGroups, ClusterHosts.EMPTY_INSTANCE_GROUPS, InstanceGroupDto::toModel),
            toModelList(dto.cloudDns, ClusterHosts.EMPTY_CLOUD_DNS, CloudDnsDto::toModel),
            settings,
            Nullables.orZero(dto.version),
            Nullables.orEpoch(dto.createdAt),
            Nullables.orEpoch(dto.updatedAt),
            Nullables.orEmpty(dto.createdBy),
            Nullables.orEmpty(dto.updatedBy),
            Map.of()
        );
    }

    private static <T, U> List<U> toModelList(@Nullable T[] array, @Nonnull T[] dflt, @Nonnull Function<T, U> mapper) {
        return Arrays.stream(Nullables.orDefault(array, dflt))
            .map(mapper)
            .collect(Collectors.toList());
    }

    @Nonnull
    public static ClusterDto fromModel(@Nonnull Cluster cluster) {
        ClusterDto dto = new ClusterDto();
        dto.setId(cluster.getId());
        dto.setName(cluster.getName());
        dto.setProjectId(cluster.getProjectId());

        if (!cluster.getHosts().isEmpty()) {
            dto.setHosts(cluster.getHosts().stream()
                .map(PatternDto::fromModel)
                .toArray(PatternDto[]::new));
        }

        if (!cluster.getHostUrls().isEmpty()) {
            dto.setHostUrls(cluster.getHostUrls().stream()
                .map(UrlDto::fromModel)
                .toArray(UrlDto[]::new));
        }

        if (!cluster.getConductorGroups().isEmpty()) {
            dto.setConductorGroups(cluster.getConductorGroups().stream()
                .map(ConductorGroupDto::fromModel)
                .toArray(ConductorGroupDto[]::new));
        }

        if (!cluster.getConductorTags().isEmpty()) {
            dto.setConductorTags(cluster.getConductorTags().stream()
                .map(ConductorTagDto::fromModel)
                .toArray(ConductorTagDto[]::new));
        }

        if (!cluster.getNannyGroups().isEmpty()) {
            dto.setNannyGroups(cluster.getNannyGroups().stream()
                .map(NannyGroupDto::fromModel)
                .toArray(NannyGroupDto[]::new));
        }

        if (!cluster.getQloudGroups().isEmpty()) {
            dto.setQloudGroups(cluster.getQloudGroups().stream()
                .map(QloudGroupDto::fromModel)
                .toArray(QloudGroupDto[]::new));
        }
        if (!cluster.getNetworks().isEmpty()) {
            dto.setNetworks(cluster.getNetworks().stream()
            .map(NetworkDto::fromModel)
            .toArray(NetworkDto[]::new));
        }

        if (!cluster.getYpClusters().isEmpty()) {
            dto.setYpClusters(cluster.getYpClusters().stream()
                .map(YpDto::fromModel)
                .toArray(YpDto[]::new));
        }

        if (!cluster.getInstanceGroups().isEmpty()) {
            dto.setInstanceGroups(cluster.getInstanceGroups().stream()
                .map(InstanceGroupDto::fromModel)
                .toArray(InstanceGroupDto[]::new));
        }

        if (!cluster.getCloudDns().isEmpty()) {
            dto.setCloudDns(cluster.getCloudDns().stream()
                .map(CloudDnsDto::fromModel)
                .toArray(CloudDnsDto[]::new));
        }

        if (cluster.getShardSettings().getMetricsTtl() > 0) {
            dto.setSensorsTtlDays(cluster.getShardSettings().getMetricsTtl());
        }

        var port = ShardSettings.getPort(cluster.getShardSettings(), 0);
        if (port != 0) {
            dto.setPort(port);
        }

        var pullSettings = cluster.getShardSettings().getPullSettings();
        dto.setUseFqdn(pullSettings != null && pullSettings.getHostLabelPolicy() == ShardSettings.HostLabelPolicy.FULL_HOSTNAME);

        dto.setVersion(cluster.getVersion());
        dto.setCreatedAt(cluster.getCreatedAt());
        dto.setUpdatedAt(cluster.getUpdatedAt());
        dto.setCreatedBy(cluster.getCreatedBy());
        dto.setUpdatedBy(cluster.getUpdatedBy());
        dto.setTvmDestId(pullSettings != null ? pullSettings.getTvmDestinationId() : "");
        dto.setHostGroupsCount(cluster.getHostGroupsCount());

        return dto;
    }

    private static ShardSettings getShardSettings(@Nonnull ClusterDto cluster) {
        final ShardSettings.Type type;
        final ShardSettings.PullSettings pullSettings;
        boolean isPull = (cluster.hosts != null && cluster.hosts.length > 0) ||
                (cluster.hostUrls != null && cluster.hostUrls.length > 0) ||
                (cluster.conductorGroups != null && cluster.conductorGroups.length > 0) ||
                (cluster.conductorTags != null && cluster.conductorTags.length > 0) ||
                (cluster.nannyGroups != null && cluster.nannyGroups.length > 0) ||
                (cluster.qloudGroups != null && cluster.qloudGroups.length > 0) ||
                (cluster.networks != null && cluster.networks.length > 0) ||
                (cluster.ypClusters != null && cluster.ypClusters.length > 0) ||
                (cluster.instanceGroups != null && cluster.instanceGroups.length > 0) ||
                (cluster.cloudDns != null && cluster.cloudDns.length > 0) ||
                Boolean.TRUE.equals(cluster.getUseFqdn()) ||
                (cluster.getPort() != null && cluster.getPort() > 0) ||
                !Nullables.orEmpty(cluster.getTvmDestId()).isEmpty();
        if (!isPull) {
            type = ShardSettings.Type.UNSPECIFIED;
            pullSettings = null;
        } else {
            type = ShardSettings.Type.PULL;
            pullSettings = ShardSettings.PullSettings.newBuilder()
                    .setPort(Nullables.orZero(cluster.getPort()))
                    .setTvmDestinationId(Nullables.orEmpty(cluster.getTvmDestId()))
                    .setHostLabelPolicy(Nullables.orFalse(cluster.getUseFqdn())
                            ? ShardSettings.HostLabelPolicy.FULL_HOSTNAME
                            : ShardSettings.HostLabelPolicy.SHORT_HOSTNAME)
                    .build();
        }
        return ShardSettings.of(
                type,
                pullSettings,
                0,
                Nullables.orDefault(cluster.getSensorsTtlDays(), 0),
                DecimPolicy.UNDEFINED,
                EMPTY,
                0
        );
    }

    public void setTvmDestId(String tvmDestId) {
        this.tvmDestId = tvmDestId;
    }
}
