package ru.yandex.solomon.gateway.api.v2.dto;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.core.db.model.CloudEnv;
import ru.yandex.solomon.core.db.model.ClusterCloudDnsConf;
import ru.yandex.solomon.core.db.model.ClusterConductorGroupConf;
import ru.yandex.solomon.core.db.model.ClusterConductorTagConf;
import ru.yandex.solomon.core.db.model.ClusterHostListConf;
import ru.yandex.solomon.core.db.model.ClusterHostUrlConf;
import ru.yandex.solomon.core.db.model.ClusterInstanceGroupConf;
import ru.yandex.solomon.core.db.model.ClusterNannyGroupConf;
import ru.yandex.solomon.core.db.model.ClusterNetworkConf;
import ru.yandex.solomon.core.db.model.ClusterQloudGroupConf;
import ru.yandex.solomon.core.db.model.ClusterYpConf;
import ru.yandex.solomon.core.db.model.NannyEnv;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.util.net.NetworkValidator;

/**
 * @author Sergey Polovko
 */
interface ClusterHosts {

    PatternDto[] EMPTY_HOSTS = new PatternDto[0];
    UrlDto[] EMPTY_HOST_URLS = new UrlDto[0];
    ConductorGroupDto[] EMPTY_CONDUCTOR_GROUPS = new ConductorGroupDto[0];
    ConductorTagDto[] EMPTY_CONDUCTOR_TAGS = new ConductorTagDto[0];
    NannyGroupDto[] EMPTY_NANNY_GROUPS = new NannyGroupDto[0];
    QloudGroupDto[] EMPTY_QLOUD_GROUPS = new QloudGroupDto[0];
    NetworkDto[] EMPTY_NETWORK = new NetworkDto[0];
    YpDto[] EMPTY_YP = new YpDto[0];
    InstanceGroupDto[] EMPTY_INSTANCE_GROUPS = new InstanceGroupDto[0];
    CloudDnsDto[] EMPTY_CLOUD_DNS = new CloudDnsDto[0];

    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    class PatternDto {
        @JsonProperty
        String urlPattern;
        @JsonProperty
        String ranges;
        @JsonProperty
        String dc;
        @JsonProperty
        String[] labels;

        void validate() {
            validateUrl(urlPattern, "urlPattern", "hosts");
            if (!StringUtils.isEmpty(ranges) && StringUtils.isBlank(ranges)) {
                throw new BadRequestException("ranges in hosts cannot be blank");
            }
            if (!StringUtils.isEmpty(dc) && StringUtils.isBlank(dc)) {
                throw new BadRequestException("dc in hosts cannot be blank");
            }
            noBlankInList(labels, "label", "hosts");
        }

        @Nonnull
        public static ClusterHostListConf toModel(@Nonnull PatternDto dto) {
            return ClusterHostListConf.of(dto.urlPattern, dto.ranges, dto.dc, dto.labels);
        }

        @Nonnull
        public static PatternDto fromModel(@Nonnull ClusterHostListConf model) {
            PatternDto dto = new PatternDto();
            dto.urlPattern = model.getUrlPattern();
            dto.ranges = model.getRanges();
            dto.dc = model.getDc();
            dto.labels = model.getLabels();
            return dto;
        }
    }

    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    class UrlDto {
        @JsonProperty
        String url;
        @JsonProperty
        String[] labels;
        @JsonProperty
        boolean ignorePorts;

        void validate() {
            validateUrl(url, "url", "hostUrls");
            noBlankInList(labels, "label", "hostUrls");
        }

        @Nonnull
        public static ClusterHostUrlConf toModel(@Nonnull UrlDto dto) {
            return ClusterHostUrlConf.of(dto.url, dto.labels, dto.ignorePorts);
        }

        @Nonnull
        public static UrlDto fromModel(@Nonnull ClusterHostUrlConf model) {
            UrlDto dto = new UrlDto();
            dto.url = model.getUrl();
            dto.labels = model.getLabels();
            dto.ignorePorts = model.isIgnorePorts();
            return dto;
        }
    }

    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    class ConductorGroupDto {
        @JsonProperty
        String group;
        @JsonProperty
        String[] labels;

        void validate() {
            if (StringUtils.isBlank(group)) {
                throw new BadRequestException("group in conductorGroups cannot be blank");
            }
            noBlankInList(labels, "label", "conductorGroups");
        }

        @Nonnull
        public static ClusterConductorGroupConf toModel(@Nonnull ConductorGroupDto dto) {
            return ClusterConductorGroupConf.of(dto.group, dto.labels);
        }

        @Nonnull
        public static ConductorGroupDto fromModel(@Nonnull ClusterConductorGroupConf model) {
            ConductorGroupDto dto = new ConductorGroupDto();
            dto.group = model.getGroup();
            dto.labels = model.getLabels();
            return dto;
        }
    }

    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    class ConductorTagDto {
        @JsonProperty
        String name;
        @JsonProperty
        String[] labels;

        void validate() {
            if (StringUtils.isBlank(name)) {
                throw new BadRequestException("name in conductorTags cannot be blank");
            }
            noBlankInList(labels, "label", "conductorGroups");
        }

        @Nonnull
        public static ClusterConductorTagConf toModel(@Nonnull ConductorTagDto dto) {
            return ClusterConductorTagConf.of(dto.name, dto.labels);
        }

        @Nonnull
        public static ConductorTagDto fromModel(@Nonnull ClusterConductorTagConf model) {
            ConductorTagDto dto = new ConductorTagDto();
            dto.name = model.getName();
            dto.labels = model.getLabels();
            return dto;
        }
    }

    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    class NannyGroupDto {
        @JsonProperty
        String service;
        @JsonProperty
        boolean useFetchedPort;
        @JsonProperty
        NannyEnv env = NannyEnv.PRODUCTION;
        @JsonProperty
        int portShift;
        @JsonProperty
        String[] cfgGroup;
        @JsonProperty
        String[] labels;

        void validate() {
            if (StringUtils.isBlank(service)) {
                throw new BadRequestException("service in nannyGroups cannot be blank");
            }
            noBlankInList(cfgGroup, "cfgGroup", "nannyGroups");
            noBlankInList(labels, "label", "nannyGroups");
        }

        @Nonnull
        public static ClusterNannyGroupConf toModel(@Nonnull NannyGroupDto dto) {
            return ClusterNannyGroupConf.of(dto.service, dto.labels, dto.useFetchedPort, dto.env, dto.portShift, dto.cfgGroup);
        }

        @Nonnull
        public static NannyGroupDto fromModel(@Nonnull ClusterNannyGroupConf model) {
            NannyGroupDto dto = new NannyGroupDto();
            dto.service = model.getService();
            dto.useFetchedPort = model.isUseFetchedPort();
            dto.env = model.getEnv();
            dto.portShift = model.getPortShift();
            dto.cfgGroup = model.getCfgGroup();
            dto.labels = model.getLabels();
            return dto;
        }
    }

    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    class QloudGroupDto {
        @JsonProperty
        String component;
        @JsonProperty
        String environment;
        @JsonProperty
        String application;
        @JsonProperty
        String project;
        @JsonProperty
        String deployment;
        @JsonProperty
        String[] labels;

        void validate() {
            if (!StringUtils.isEmpty(component) && StringUtils.isBlank(component)) {
                throw new BadRequestException("component in qloudGroups cannot be blank");
            }
            if (!StringUtils.isEmpty(environment) && StringUtils.isBlank(environment)) {
                throw new BadRequestException("environment in qloudGroups cannot be blank");
            }
            if (!StringUtils.isEmpty(application) && StringUtils.isBlank(application)) {
                throw new BadRequestException("application in qloudGroups cannot be blank");
            }
            if (!StringUtils.isEmpty(project) && StringUtils.isBlank(project)) {
                throw new BadRequestException("project in qloudGroups cannot be blank");
            }
            if (!StringUtils.isEmpty(deployment) && StringUtils.isBlank(deployment)) {
                throw new BadRequestException("deployment in qloudGroups cannot be blank");
            }
            if (StringUtils.isEmpty(component) && StringUtils.isEmpty(environment) &&
                StringUtils.isEmpty(application) && StringUtils.isEmpty(project) &&
                StringUtils.isEmpty(deployment))
            {
                throw new BadRequestException("qloudGroups must have at least one field filled");
            }
            noBlankInList(labels, "label", "qloudGroups");
        }

        @Nonnull
        public static ClusterQloudGroupConf toModel(@Nonnull QloudGroupDto dto) {
            return ClusterQloudGroupConf.of(
                dto.component, dto.environment, dto.application, dto.project, dto.deployment,
                dto.labels);
        }

        @Nonnull
        public static QloudGroupDto fromModel(@Nonnull ClusterQloudGroupConf model) {
            QloudGroupDto dto = new QloudGroupDto();
            dto.component = model.getComponent();
            dto.environment = model.getEnvironment();
            dto.application = model.getApplication();
            dto.project = model.getProject();
            dto.deployment = model.getDeployment();
            dto.labels = model.getLabels();
            return dto;
        }
    }

    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    class NetworkDto {

        @JsonProperty
        String network;
        @JsonProperty
        Integer port;
        @JsonProperty
        String[] labels;

        void validate(NetworkValidator validator) {
            if (StringUtils.isBlank(network)) {
                throw new BadRequestException("network cannot be blank");
            }
            if (port != null && (port < 1 || port > 65535)) {
                throw new BadRequestException("port is not in area 1-65535");
            }
            noBlankInList(labels, "label", "network");

            validator.validateNetwork(network);
        }

        @Nonnull
        public static ClusterNetworkConf toModel(@Nonnull NetworkDto dto) {
            return ClusterNetworkConf.of(dto.network, dto.port, dto.labels);
        }

        @Nonnull
        public static NetworkDto fromModel(@Nonnull ClusterNetworkConf model) {
            NetworkDto dto = new NetworkDto();
            dto.network = model.getNetwork();
            dto.port = model.getPort();
            dto.labels = model.getLabels();
            return dto;
        }
    }

    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    class YpDto {
        @JsonProperty
        private String podSetId;
        @JsonProperty
        private String endpointSetId;
        @JsonProperty
        private String cluster;
        @JsonProperty
        private String[] labels;
        @JsonProperty
        private String ypLabel;
        @JsonProperty
        private String tvmLabel;

        void validate() {
            var isPodSetIdPresent = StringUtils.isNotBlank(podSetId);
            var isEndpointSetIdPresent = StringUtils.isNotBlank(endpointSetId);
            var isYpLabelPresent = StringUtils.isNotBlank(ypLabel);

            if (cluster.isEmpty()) {
                throw new BadRequestException("Cluster cannot be empty");
            }

            if (isPodSetIdPresent && isYpLabelPresent) {
                throw new BadRequestException("podSetId and ypLabel cannot be set at the same time");
            } else if (isEndpointSetIdPresent && isPodSetIdPresent) {
                throw new BadRequestException("podSetId and endpointSetId cannot be set at the same time");
            } else if (isEndpointSetIdPresent && isYpLabelPresent) {
                throw new BadRequestException("ypLabel and endpointSetId cannot be set at the same time");
            } else if (!(isPodSetIdPresent || isYpLabelPresent || isEndpointSetIdPresent)) {
                throw new BadRequestException("On of podSetId, ypLabel, endpointSetId must be set");
            }

            noBlankInList(labels, "label", "ypClusters");
        }

        @Nonnull
        public static ClusterYpConf toModel(@Nonnull YpDto dto) {
            return ClusterYpConf.of(dto.podSetId, dto.endpointSetId, dto.cluster, dto.labels, dto.tvmLabel, dto.ypLabel);
        }

        @Nonnull
        public static YpDto fromModel(@Nonnull ClusterYpConf model) {
            YpDto dto = new YpDto();
            dto.podSetId = model.getPodSetId();
            dto.endpointSetId = model.getEndpointSetId();
            dto.cluster = model.getCluster();
            dto.labels = model.getLabels();
            dto.tvmLabel = model.getTvmLabel();
            dto.ypLabel = model.getYpLabel();

            return dto;
        }
    }

    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    class InstanceGroupDto {
        @JsonProperty
        private String instanceGroupId;
        @JsonProperty
        private String folderId;
        @JsonProperty
        private String[] labels;

        void validate() {
            var isGroupPresent = !StringUtils.isBlank(instanceGroupId);
            var isFolderPresent = !StringUtils.isBlank(folderId);
            if (!isGroupPresent && !isFolderPresent) {
                throw new BadRequestException("either instanceGroupId or folderId must be non-empty");
            } else if (isGroupPresent && isFolderPresent) {
                throw new BadRequestException("only one of instanceGroupId and folderId must be present");
            }

            noBlankInList(labels, "label", "instanceGroups");
        }

        @Nonnull
        public static ClusterInstanceGroupConf toModel(@Nonnull InstanceGroupDto dto) {
            return ClusterInstanceGroupConf.of(dto.instanceGroupId, dto.folderId, dto.labels);
        }

        @Nonnull
        public static InstanceGroupDto fromModel(@Nonnull ClusterInstanceGroupConf model) {
            InstanceGroupDto dto = new InstanceGroupDto();
            dto.instanceGroupId = model.getInstanceGroupId();
            dto.folderId = model.getFolderId();
            dto.labels = model.getLabels();
            return dto;
        }
    }

    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    class CloudDnsDto {
        @JsonProperty
        CloudEnv env = CloudEnv.UNKNOWN;
        @JsonProperty
        String name;
        @JsonProperty
        private String[] labels;

        void validate() {
            if (StringUtils.isBlank(name)) {
                throw new BadRequestException("name cannot be blank");
            }
            noBlankInList(labels, "label", "cloudDns");
        }

        @Nonnull
        public static ClusterCloudDnsConf toModel(@Nonnull CloudDnsDto dto) {
            return ClusterCloudDnsConf.of(dto.env, dto.name, dto.labels);
        }

        @Nonnull
        public static CloudDnsDto fromModel(@Nonnull ClusterCloudDnsConf model) {
            CloudDnsDto dto = new CloudDnsDto();
            dto.env = model.getEnv();
            dto.name = model.getName();
            dto.labels = model.getLabels();
            return dto;
        }
    }

    static void noBlankInList(String[] list, String itemName, String fieldName) {
        if (list == null) {
            return;
        }
        for (String item : list) {
            if (StringUtils.isBlank(item)) {
                throw new BadRequestException(itemName + " in " + fieldName + " cannot be blank");
            }
        }
    }

    private static void validateUrl(String url, String itemName, String fieldName) {
        if (StringUtils.isBlank(url)) {
            throw new BadRequestException(itemName + " in " + fieldName + " must not be blank");
        }
        if (url.startsWith("http://")) {
            throw new BadRequestException(itemName + " in " + fieldName + " must not start with \"http://\"");
        }
        if (url.startsWith("https://")) {
            throw new BadRequestException(itemName + " in " + fieldName + " must not start with \"https://\"");
        }
    }
}
