package ru.yandex.solomon.gateway.api.v2.dto;

import java.time.Instant;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.core.db.model.Dashboard;
import ru.yandex.solomon.core.db.model.DashboardRow;
import ru.yandex.solomon.core.db.model.Selector;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.validators.IdValidator;
import ru.yandex.solomon.util.collection.Nullables;


/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class DashboardDto {

    @JsonProperty
    public String id;
    @JsonProperty
    public String projectId;
    @JsonProperty
    public String name;
    @JsonProperty
    public String description;
    @JsonProperty
    public double heightMultiplier;
    @JsonProperty
    public SelectorListDto parameters;
    @JsonProperty
    public DashboardRowDto[] rows;
    @JsonProperty
    public String generatedId;
    @JsonProperty
    public Instant createdAt;
    @JsonProperty
    public Instant updatedAt;
    @JsonProperty
    public String createdBy;
    @JsonProperty
    public String updatedBy;
    @JsonProperty
    public int version;

    public void validate() {
        IdValidator.ensureValid(id, "dashboard");

        if (StringUtils.isBlank(name)) {
            throw new BadRequestException("name cannot be blank");
        }

        if (StringUtils.isBlank(projectId)) {
            throw new BadRequestException("projectId cannot be blank");
        }

        if (heightMultiplier < 1) {
            throw new BadRequestException("height multiplier cannot be less than 1");
        }

        if (heightMultiplier > 10) {
            throw new BadRequestException("height multiplier cannot be greater than 10");
        }

        if (parameters != null) {
            parameters.validate();
        }

        if (rows == null) {
            throw new BadRequestException("rows cannot me empty");
        }
        DashboardRowDto.validateDtos(rows);

        if (version < 0) {
            throw new BadRequestException("version cannot be negative");
        }
    }

    @Nonnull
    public Dashboard toModel() {
        Selector[] parameterModels = this.parameters == null
            ? new Selector[0]
            : SelectorListDto.toModel(this.parameters);
        DashboardRow[] rowModels = this.rows == null
            ? new DashboardRow[0]
            : DashboardRowDto.toModels(this.rows);

        return new Dashboard(
            id,
            projectId,
            "",
            name,
            Nullables.orEmpty(description),
            heightMultiplier,
            parameterModels,
            rowModels,
            Nullables.orEmpty(generatedId),
            Nullables.orEpoch(createdAt),
            Nullables.orEpoch(updatedAt),
            Nullables.orEmpty(createdBy),
            Nullables.orEmpty(updatedBy),
            version
        );
    }

    @Nonnull
    public static DashboardDto fromModel(@Nonnull Dashboard model) {
        DashboardDto dto = new DashboardDto();

        dto.id = model.getId();
        dto.projectId = model.getProjectId();
        dto.name = model.getName();
        dto.description = model.getDescription();
        dto.heightMultiplier = model.getHeightMultiplier();
        dto.parameters = SelectorListDto.fromModel(model.getParameters());
        dto.rows = DashboardRowDto.fromModels(model.getRows());
        dto.generatedId = model.getGeneratedId();
        dto.createdAt = model.getCreatedAt();
        dto.updatedAt = model.getUpdatedAt();
        dto.createdBy = model.getCreatedBy();
        dto.updatedBy = model.getUpdatedBy();
        dto.version = model.getVersion();

        return dto;
    }
}
