package ru.yandex.solomon.gateway.api.v2.dto;

import java.util.Arrays;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.solomon.core.db.model.DashboardPanel;
import ru.yandex.solomon.core.db.model.DashboardRow;
import ru.yandex.solomon.core.exceptions.BadRequestException;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class DashboardRowDto {

    @JsonProperty
    private DashboardPanelDto[] panels;

    public void validate() {
        if (panels == null) {
            throw new BadRequestException("panels cannot be empty");
        }
        for (DashboardPanelDto panelDto : panels) {
            if (panelDto == null) {
                throw new BadRequestException("panel cannot be null");
            }
            panelDto.validate();
        }
    }

    public static void validateDtos(@Nonnull DashboardRowDto[] rows) {
        for (DashboardRowDto rowDto : rows) {
            if (rowDto == null) {
                throw new BadRequestException("row cannot be empty");
            }
            rowDto.validate();
        }
    }

    @Nonnull
    public DashboardRow toModel() {
        DashboardPanel[] panelModels =
            Arrays.stream(this.panels)
                .map(DashboardPanelDto::toModel)
                .toArray(DashboardPanel[]::new);
        return new DashboardRow(panelModels);
    }

    @Nonnull
    public static DashboardRow[] toModels(@Nonnull DashboardRowDto[] dtos) {
        return Arrays.stream(dtos)
            .map(DashboardRowDto::toModel)
            .toArray(DashboardRow[]::new);
    }

    @Nonnull
    public static DashboardRowDto fromModel(@Nonnull DashboardRow model) {
        DashboardRowDto dto = new DashboardRowDto();
        dto.panels =
            Arrays.stream(model.getPanels())
            .map(DashboardPanelDto::fromModel)
            .toArray(DashboardPanelDto[]::new);
        return dto;
    }

    @Nonnull
    public static DashboardRowDto[] fromModels(@Nonnull DashboardRow[] models) {
        return Arrays.stream(models)
            .map(DashboardRowDto::fromModel)
            .toArray(DashboardRowDto[]::new);
    }
}
