package ru.yandex.solomon.gateway.api.v2.dto;

import java.time.Instant;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Strings;
import com.google.protobuf.util.Timestamps;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.monitoring.api.v3.DeleteMetricsOperation;

/**
 * @author Stanislav Kashirin
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
@ParametersAreNullableByDefault
public class DeleteMetricsOperationDto {

    @ApiModelProperty(
        value = "Operation ID.",
        required = true,
        position = 0)
    @JsonProperty
    public String id;

    @ApiModelProperty(
        value = "Project ID.",
        required = true,
        position = 1)
    @JsonProperty
    public String projectId;

    @ApiModelProperty(
        value = "Operation description (reason for launch, etc).",
        required = false,
        position = 2)
    @JsonProperty
    public String description;

    @ApiModelProperty(
        value = "Creation timestamp.",
        required = true,
        position = 3)
    @JsonProperty
    public Instant createdAt;

    @ApiModelProperty(
        value = "ID of the user who created the operation.",
        required = true,
        position = 4)
    @JsonProperty
    public String createdBy;

    @ApiModelProperty(
        value = "The time when the operation state was last updated.",
        required = true,
        position = 5)
    @JsonProperty
    public Instant updatedAt;

    @ApiModelProperty(
        value = "Cancel attempt timestamp.",
        required = false,
        position = 6)
    @JsonProperty
    public Instant cancelledAt;

    @ApiModelProperty(
        value = "ID of the user who has been performed the attempt to cancel this operation.",
        required = false,
        position = 7)
    @JsonProperty
    public String cancelledBy;

    @ApiModelProperty(
        value = """
                Status of operation. Possible values:
                - DELETING
                Deleting metrics.

                - WAITING_FOR_PERMANENT_DELETION
                Waiting for effective date to start permanent deletion.

                - DELETING_PERMANENTLY
                Deleting metrics data permanently. Cancel is not possible starting from this state.

                - COMPLETED
                Terminal status, data deleted permanently.

                - CANCELLING
                Cancelling the operation, already deleted metrics are restoring.

                - CANCELLED
                Terminal status, all previously deleted metrics have been restored.

                - FAILED
                Terminal status, operation has been failed. See statusMessage for an additional information.

                - METRICS_HAVE_RECENT_WRITES
                Terminal status, deletion rejected due to validation (e.g. metrics are still used and have recent writes).
                See statusMessage for an additional information.
                """,
        required = true,
        position = 9)
    @JsonProperty
    public String status;

    @ApiModelProperty(
        value = "Detailed description of status.",
        required = false,
        position = 10)
    @JsonProperty
    public String statusMessage;

    @ApiModelProperty(
        value = "Selectors to match metrics for deletion.",
        required = true,
        position = 11)
    @JsonProperty
    public String selectors;

    @ApiModelProperty(
        value = "Timestamp of expected permanent deletion launch. " +
            "Cancel is not possible starting from this time.",
        required = true,
        position = 12)
    @JsonProperty
    public Instant permanentDeletionAt;

    @ApiModelProperty(
        value = "Estimated number of metrics selected for deletion.",
        required = true,
        position = 13)
    @JsonProperty
    public int estimatedMetricsCount;

    @ApiModelProperty(
        value = "Representation of operation progress in range [0, 100].",
        required = true,
        position = 14)
    @JsonProperty
    public double progressPercentage;

    @ApiModelProperty(
        value = "Precise number of permanently deleted metrics after operation completion.",
        required = true,
        position = 15)
    @JsonProperty
    public int permanentlyDeletedMetricsCount;

    @Nonnull
    public static DeleteMetricsOperationDto fromProto(@Nonnull DeleteMetricsOperation proto) {
        var dto = new DeleteMetricsOperationDto();

        dto.id = proto.getId();
        dto.projectId = proto.getProjectId();
        dto.description = proto.getDescription();
        dto.createdAt = Instant.ofEpochMilli(Timestamps.toMillis(proto.getCreatedAt()));
        dto.createdBy = proto.getCreatedBy();
        dto.updatedAt = Instant.ofEpochMilli(Timestamps.toMillis(proto.getModifiedAt()));
        if (!Timestamps.EPOCH.equals(proto.getCancelledAt())) {
            dto.cancelledAt = Instant.ofEpochMilli(Timestamps.toMillis(proto.getCancelledAt()));
        }
        dto.cancelledBy = Strings.emptyToNull(proto.getCancelledBy());
        dto.status = proto.getStatus().name();
        dto.statusMessage = proto.getStatusMessage();
        dto.selectors = proto.getSelectors();
        dto.permanentDeletionAt = Instant.ofEpochMilli(Timestamps.toMillis(proto.getPermanentDeletionAt()));
        dto.estimatedMetricsCount = proto.getEstimatedMetricsCount();
        dto.progressPercentage = proto.getProgressPercentage();
        dto.permanentlyDeletedMetricsCount = proto.getPermanentlyDeletedMetricsCount();

        return dto;
    }

}
