package ru.yandex.solomon.gateway.api.v2.dto;

import java.time.Instant;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.core.db.model.graph.AggregationType;
import ru.yandex.solomon.core.db.model.graph.ColorSchemeType;
import ru.yandex.solomon.core.db.model.graph.DownsamplingAggregationType;
import ru.yandex.solomon.core.db.model.graph.DownsamplingFillType;
import ru.yandex.solomon.core.db.model.graph.DownsamplingMode;
import ru.yandex.solomon.core.db.model.graph.FilterFunction;
import ru.yandex.solomon.core.db.model.graph.FilterOrder;
import ru.yandex.solomon.core.db.model.graph.Graph;
import ru.yandex.solomon.core.db.model.graph.GraphMode;
import ru.yandex.solomon.core.db.model.graph.GraphTransform;
import ru.yandex.solomon.core.db.model.graph.InterpolationType;
import ru.yandex.solomon.core.db.model.graph.OverLinesTransform;
import ru.yandex.solomon.core.db.model.graph.ScaleType;
import ru.yandex.solomon.core.db.model.graph.SecondaryGraphMode;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.validators.IdValidator;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class GraphDto {

    private static final int MAX_POINTS_UPPER_LIMIT = 10000;

    @JsonProperty
    public String id;
    @JsonProperty
    public String projectId;
    @JsonProperty
    public String name;

    @JsonProperty
    public String description;
    @JsonProperty
    public SelectorListDto parameters;
    @JsonProperty
    public GraphElementListDto elements;
    @JsonProperty
    public GraphMode graphMode;
    @JsonProperty
    public SecondaryGraphMode secondaryGraphMode;
    @JsonProperty
    public String min;
    @JsonProperty
    public String max;
    @JsonProperty
    public Boolean normalize;
    @JsonProperty
    public ColorSchemeType colorScheme;
    @JsonProperty
    public Boolean dropNans;
    @JsonProperty
    public Boolean stack;
    @JsonProperty
    public AggregationType aggr;
    @JsonProperty
    public InterpolationType interpolate;
    @JsonProperty
    public ScaleType scale;
    @JsonProperty
    public String numberFormat;
    @JsonProperty
    public String green;
    @JsonProperty
    public String yellow;
    @JsonProperty
    public String red;
    @JsonProperty
    public String violet;
    @JsonProperty
    public String greenValue;
    @JsonProperty
    public String yellowValue;
    @JsonProperty
    public String redValue;
    @JsonProperty
    public String violetValue;
    @JsonProperty
    public String limit;
    @JsonProperty
    public OverLinesTransform overLinesTransform;
    @JsonProperty
    public String percentiles;
    @JsonProperty
    public String bucketLabel;
    @JsonProperty
    public boolean ignoreInf;
    @JsonProperty
    public FilterOrder filter;
    @JsonProperty
    public FilterFunction filterBy;
    @JsonProperty
    public String filterLimit;
    @JsonProperty
    public GraphTransform transform;
    @JsonProperty
    public String movingWindow;
    @JsonProperty
    public String movingPercentile;
    @JsonProperty
    public DownsamplingMode downsampling;
    @JsonProperty
    public DownsamplingAggregationType downsamplingAggr;
    @JsonProperty
    public DownsamplingFillType downsamplingFill;
    @JsonProperty
    public Boolean ignoreMinStepMillis;
    @JsonProperty
    public String grid;
    @JsonProperty
    public Integer maxPoints;
    @JsonProperty
    public Boolean hideNoData;
    @JsonProperty
    public Double threshold;
    @JsonProperty
    public String generatedId;

    @JsonProperty
    public Instant createdAt;
    @JsonProperty
    public Instant updatedAt;
    @JsonProperty
    public String createdBy;
    @JsonProperty
    public String updatedBy;
    @JsonProperty
    public Integer version;

    public void setId(String id) {
        this.id = id;
    }

    public void setCreatedAt(Instant createdAt) {
        this.createdAt = createdAt;
    }

    public void setUpdatedAt(Instant updatedAt) {
        this.updatedAt = updatedAt;
    }

    public void setCreatedBy(String createdBy) {
        this.createdBy = createdBy;
    }

    public void setUpdatedBy(String updatedBy) {
        this.updatedBy = updatedBy;
    }

    public void setProjectId(String projectId) {
        this.projectId = projectId;
    }

    public void setGeneratedId(String generatedId) {
        this.generatedId = generatedId;
    }

    public Integer getVersion() {
        return version;
    }

    public void validate() {
        IdValidator.ensureValid(id, "graph");

        if ("auto".equals(id)) {
            throw new BadRequestException("graph id cannot be \"auto\"");
        }

        if (StringUtils.isBlank(name)) {
            throw new BadRequestException("name cannot be blank");
        }

        if (StringUtils.isBlank(projectId)) {
            throw new BadRequestException("projectId cannot be blank");
        }

        if (parameters != null) {
            parameters.validate();
        }

        if (elements == null || elements.getElements() == null || elements.getElements().length == 0) {
            throw new BadRequestException("elements cannot be empty");
        }

        elements.validate();

        if (maxPoints != null && !(maxPoints >= 0 && maxPoints <= MAX_POINTS_UPPER_LIMIT)) {
            throw new BadRequestException("max points must be in [0, " + MAX_POINTS_UPPER_LIMIT + "]");
        }

        if (version != null && version < 0) {
            throw new BadRequestException("version cannot be negative");
        }
    }

    @Nonnull
    public Graph toModel() {
        return new Graph(
            id,
            projectId,
            "",
            name,
            description,
            SelectorListDto.toModel(parameters),
            GraphElementListDto.toModel(elements),
            graphMode,
            secondaryGraphMode,
            min,
            max,
            normalize,
            colorScheme,
            dropNans,
            stack,
            aggr,
            interpolate,
            scale,
            numberFormat,
            green,
            yellow,
            red,
            violet,
            greenValue,
            yellowValue,
            redValue,
            violetValue,
            "",
            false,
            limit,
            overLinesTransform,
            percentiles,
            bucketLabel,
            ignoreInf,
            filter,
            filterBy,
            filterLimit,
            transform,
            movingWindow,
            movingPercentile,
            downsampling,
            downsamplingAggr,
            downsamplingFill,
            ignoreMinStepMillis,
            grid,
            maxPoints,
            hideNoData,
            threshold,
            generatedId,
            createdAt,
            updatedAt,
            createdBy,
            updatedBy,
            version
        );
    }

    @Nonnull
    public static GraphDto fromModel(@Nonnull Graph model) {
        GraphDto dto = new GraphDto();
        dto.id = model.getId();
        dto.projectId = model.getProjectId();
        dto.name = model.getName();
        dto.description = model.getDescription();
        dto.parameters = SelectorListDto.fromModel(model.getParameters());
        dto.elements = GraphElementListDto.fromModel(model.getElements());
        dto.graphMode = model.getGraphMode();
        dto.secondaryGraphMode = model.getSecondaryGraphMode();
        dto.min = model.getMin();
        dto.max = model.getMax();
        dto.normalize = model.isNormalize();
        dto.colorScheme = model.getColorScheme();
        dto.dropNans = model.isDropNans();
        dto.stack = model.getStack();
        dto.aggr = model.getAggr();
        dto.interpolate = model.getInterpolate();
        dto.scale = model.getScale();
        dto.numberFormat = model.getNumberFormat();
        dto.green = model.getGreen();
        dto.yellow = model.getYellow();
        dto.red = model.getRed();
        dto.violet = model.getViolet();
        dto.greenValue = model.getGreenValue();
        dto.yellowValue = model.getYellowValue();
        dto.redValue = model.getRedValue();
        dto.violetValue = model.getVioletValue();
        dto.limit = model.getLimit();
        dto.overLinesTransform = model.getOverLinesTransform();
        dto.percentiles = model.getPercentiles();
        dto.bucketLabel = model.getBucketLabel();
        dto.ignoreInf = model.isIgnoreInf();
        dto.filter = model.getFilter();
        dto.filterBy = model.getFilterBy();
        dto.filterLimit = model.getFilterLimit();
        dto.transform = model.getTransform();
        dto.movingWindow = model.getMovingWindow();
        dto.movingPercentile = model.getMovingPercentile();
        dto.downsampling = model.getDownsampling();
        dto.downsamplingAggr = model.getDownsamplingAggr();
        dto.downsamplingFill = model.getDownsamplingFill();
        dto.ignoreMinStepMillis = model.isIgnoreMinStepMillis();
        dto.grid = model.getGrid();
        dto.maxPoints = model.getMaxPoints();
        dto.hideNoData = model.isHideNoData();
        dto.threshold = model.getThreshold();
        dto.generatedId = model.getGeneratedId();
        dto.createdAt = model.getCreatedAt();
        dto.updatedAt = model.getUpdatedAt();
        dto.createdBy = model.getCreatedBy();
        dto.updatedBy = model.getUpdatedBy();
        dto.version = model.getVersion();
        return dto;
    }
}
