package ru.yandex.solomon.gateway.api.v2.dto;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.core.db.model.Selector;
import ru.yandex.solomon.core.db.model.graph.ElementTransform;
import ru.yandex.solomon.core.db.model.graph.GraphElement;
import ru.yandex.solomon.core.db.model.graph.GraphElementType;
import ru.yandex.solomon.core.db.model.graph.YaxisPosition;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class GraphElementDto {

    @JsonProperty
    private String title;
    @JsonProperty
    private GraphElementType type;
    @JsonProperty
    private SelectorListDto selectors;
    @JsonProperty
    private String expression;
    @JsonProperty
    private String link;
    @JsonProperty
    private Boolean area;
    @JsonProperty
    private String stack;
    @JsonProperty
    private Boolean down;
    @JsonProperty
    private String color;
    @JsonProperty
    private YaxisPosition yaxis;
    @JsonProperty
    private ElementTransform transform;

    public void validate() {
        if (StringUtils.isBlank(expression) && (selectors == null || selectors.getSelectors().isEmpty())) {
            throw new BadRequestException("graph element must contain any selector or expression");
        }

        if (selectors != null) {
            selectors.validate();
        }
    }

    @Nonnull
    public GraphElement toModel() {
        final Selector[] selectors;
        final String expression;
        final String link;

        GraphElementType type =
            Nullables.orDefault(this.type, GraphElementType.SELECTORS);

        switch (type) {
            case SELECTORS:
                selectors = SelectorListDto.toModel(this.selectors);
                link = this.link;
                expression = "";
                break;
            case EXPRESSION:
                expression = this.expression;
                selectors = new Selector[0];
                link = "";
                break;
            default:
                throw new BadRequestException("unknown element type: " + this.type);
        }

        return new GraphElement(
            title,
            type,
            selectors,
            expression,
            link,
            area,
            stack,
            down,
            color,
            yaxis,
            transform
        );
    }

    @Nonnull
    public static GraphElementDto fromModel(@Nonnull GraphElement model) {
        GraphElementDto dto = new GraphElementDto();
        dto.title = model.getTitle();
        dto.type = model.getType();

        switch (model.getType()) {
            case SELECTORS:
                dto.selectors = SelectorListDto.fromModel(model.getSelectors());
                dto.expression = "";
                dto.link = model.getLink();
                break;
            case EXPRESSION:
                dto.expression = model.getExpression();
                dto.selectors = SelectorListDto.fromModel(new Selector[0]);
                break;
            default:
                throw new IllegalStateException("unknown element type: " + model.getType());
        }

        dto.area = model.getArea();
        dto.stack = model.getStack();
        dto.down = model.getDown();
        dto.color = model.getColor();
        dto.yaxis = model.getYaxis();
        dto.transform = model.getTransform();
        return dto;
    }
}
