package ru.yandex.solomon.gateway.api.v2.dto;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.core.db.model.MenuItem;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.labels.FormatLabel;
import ru.yandex.solomon.labels.selector.LabelSelectorSet;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class MenuItemDto extends DefaultObject {

    private String title;
    private String url;
    private MenuItemDto[] children;
    private String selectors;

    public String getTitle() {
        return title;
    }

    public void setTitle(String title) {
        this.title = title;
    }

    public String getUrl() {
        return url;
    }

    public void setUrl(String url) {
        this.url = url;
    }

    public MenuItemDto[] getChildren() {
        return children;
    }

    public void setChildren(MenuItemDto[] children) {
        this.children = children;
    }

    public String getSelectors() {
        return selectors;
    }

    public void setSelectors(String selectors) {
        this.selectors = selectors;
    }

    public void validate(@Nonnull Map<String, String> parentRestrictions) {
        if (StringUtils.isBlank(title)) {
            throw new BadRequestException("menu item title cannot be blank");
        }

        Map<String, String> restrictions = new HashMap<>(parentRestrictions);
        if (!StringUtils.isBlank(selectors)) {
            LabelSelectorSet labelSelectorSet = parseSelectors(selectors);
            restrictions.putAll(labelSelectorSet.toLinkedHashMap());
        }

        FormatLabel.formatLabel(title, a -> restrictions.computeIfAbsent(a, key -> {
            throw new BadRequestException("bad menu item: title references to non-existing selector: " + key);
        }));

        if (children == null || children.length == 0) {
            if (StringUtils.isBlank(url)) {
                throw new BadRequestException("menu item url cannot be blank");
            }
            FormatLabel.formatLabel(url, a -> restrictions.computeIfAbsent(a, key -> {
                throw new BadRequestException("menu item url references to non-existing selector: " + key);
            }));
        } else {
            validateMenuItems(children, restrictions);
        }
    }

    @Nonnull
    private static LabelSelectorSet parseSelectors(@Nonnull String selectors) {
        try {
            return LabelSelectorSet.parseEscaped(selectors);
        } catch (Exception ignored) {
            throw new BadRequestException("failed to parse selector \"" + selectors + '"');
        }
    }

    static void validateMenuItems(
        @Nonnull MenuItemDto[] menuItems,
        @Nonnull Map<String, String> parentRestrictions)
    {
        for (MenuItemDto subMenuItemDto : menuItems) {
            if (subMenuItemDto == null) {
                throw new BadRequestException("menu tree must contain sub items");
            }
            subMenuItemDto.validate(parentRestrictions);
        }
    }

    @Nonnull
    public static MenuItem toModel(@Nonnull MenuItemDto dto) {
        return new MenuItem(
            dto.title,
            dto.url,
            toModelList(dto.children),
            dto.selectors
        );
    }

    @Nonnull
    static MenuItem[] toModelList(@Nullable MenuItemDto[] dtos) {
        if (dtos == null) {
            return new MenuItem[0];
        }

        return Arrays.stream(dtos)
            .filter(Objects::nonNull)
            .map(MenuItemDto::toModel)
            .toArray(MenuItem[]::new);
    }

    @Nonnull
    public static MenuItemDto fromModel(@Nonnull MenuItem model) {
        MenuItemDto dto = new MenuItemDto();
        dto.setTitle(model.getTitle());
        dto.setUrl(model.getUrl());
        dto.setChildren(fromModelList(model.getChildren()));
        dto.setSelectors(model.getSelectors());
        return dto;
    }

    @Nonnull
    private static MenuItemDto[] fromModelList(@Nonnull MenuItem[] models) {
        return Arrays.stream(models)
            .map(MenuItemDto::fromModel)
            .toArray(MenuItemDto[]::new);
    }
}
