package ru.yandex.solomon.gateway.api.v2.dto;

import java.time.Instant;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.validators.IdValidator;
import ru.yandex.solomon.util.collection.Nullables;


/**
 * @author Sergey Polovko
 */
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class ProjectDto {

    private String id;
    private String name;
    private String description;
    private String owner;
    private AclDto acl;
    private String abcService;
    private Boolean onlyAuthRead;
    private Boolean onlyAuthPush;
    private Boolean onlySensorNameShards;
    private Boolean onlyNewFormatWrites;
    private Boolean onlyNewFormatReads;
    private String metricNameLabel;
    private Instant createdAt;
    private Instant updatedAt;
    private String createdBy;
    private String updatedBy;
    private int version;

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public String getOwner() {
        return owner;
    }

    public void setOwner(String owner) {
        this.owner = owner;
    }

    public AclDto getAcl() {
        return acl;
    }

    public void setAcl(AclDto acl) {
        this.acl = acl;
    }

    public String getAbcService() {
        return abcService;
    }

    public void setAbcService(String abcService) {
        this.abcService = abcService;
    }

    public Boolean getOnlyAuthRead() {
        return onlyAuthRead;
    }

    public void setOnlyAuthRead(Boolean onlyAuthRead) {
        this.onlyAuthRead = onlyAuthRead;
    }

    public Boolean getOnlyAuthPush() {
        return onlyAuthPush;
    }

    public void setOnlyAuthPush(Boolean onlyAuthPush) {
        this.onlyAuthPush = onlyAuthPush;
    }

    public Boolean getOnlySensorNameShards() {
        return this.onlySensorNameShards;
    }

    public void setOnlySensorNameShards(Boolean onlySensorNameShards) {
        this.onlySensorNameShards = onlySensorNameShards;
    }

    public Boolean getOnlyNewFormatWrites() {
        return onlyNewFormatWrites;
    }

    public void setOnlyNewFormatWrites(Boolean onlyNewFormatWrites) {
        this.onlyNewFormatWrites = onlyNewFormatWrites;
    }

    public Boolean getOnlyNewFormatReads() {
        return onlyNewFormatReads;
    }

    public void setOnlyNewFormatReads(Boolean onlyNewFormatReads) {
        this.onlyNewFormatReads = onlyNewFormatReads;
    }

    public String getMetricNameLabel() {
        return metricNameLabel;
    }

    public void setMetricNameLabel(String metricNameLabel) {
        this.metricNameLabel = metricNameLabel;
    }

    public Instant getCreatedAt() {
        return createdAt;
    }

    public void setCreatedAt(Instant createdAt) {
        this.createdAt = createdAt;
    }

    public Instant getUpdatedAt() {
        return updatedAt;
    }

    public void setUpdatedAt(Instant updatedAt) {
        this.updatedAt = updatedAt;
    }

    public String getCreatedBy() {
        return createdBy;
    }

    public void setCreatedBy(String createdBy) {
        this.createdBy = createdBy;
    }

    public String getUpdatedBy() {
        return updatedBy;
    }

    public void setUpdatedBy(String updatedBy) {
        this.updatedBy = updatedBy;
    }

    public int getVersion() {
        return version;
    }

    public void setVersion(int version) {
        this.version = version;
    }

    @JsonIgnore
    public boolean isAclEmpty() {
        return acl == null || acl.isEmpty();
    }

    /**
     * @deprecated Use standalone validators instead
     * @see ru.yandex.solomon.gateway.api.v2.validation.AbcServiceFieldValidator
     * @see ru.yandex.solomon.gateway.api.v2.validation.ProjectAclValidator
     */
    @Deprecated
    public void validate() {
        IdValidator.ensureProjectIdValid(id);
        ValidationUtils.validateLogin("owner login", owner);
        if (StringUtils.isBlank(name)) {
            throw new BadRequestException("name cannot be blank");
        }
        validateAbcService(abcService);
        if (version < 0) {
            throw new BadRequestException("version cannot be negative");
        }

        if (acl != null) {
            acl.validate();
        }
    }

    private void validateAbcService(String abcService) {
        if (StringUtils.isBlank(abcService)) {
            throw new BadRequestException("abcService cannot be blank");
        }
    }

    @Nonnull
    public static ProjectDto fromModel(@Nonnull Project project) {
        ProjectDto dto = new ProjectDto();
        dto.setId(project.getId());
        dto.setName(project.getName());
        dto.setDescription(project.getDescription());
        dto.setOwner(project.getOwner());
        dto.setAcl(AclDto.fromModel(project.getAcl()));
        dto.setAbcService(project.getAbcService());
        dto.setOnlyAuthRead(project.isOnlyAuthRead());
        dto.setOnlyAuthPush(project.isOnlyAuthPush());
        dto.setOnlySensorNameShards(project.isOnlyMetricNameShards());
        dto.setOnlyNewFormatWrites(project.isOnlyNewFormatWrites());
        dto.setOnlyNewFormatReads(project.isOnlyNewFormatReads());
        dto.setMetricNameLabel(project.getMetricNameLabel());
        dto.setCreatedAt(project.getCreatedAt());
        dto.setUpdatedAt(project.getUpdatedAt());
        dto.setCreatedBy(project.getCreatedBy());
        dto.setUpdatedBy(project.getUpdatedBy());
        dto.setVersion(project.getVersion());
        return dto;
    }

    @Nonnull
    public static Project toModel(@Nonnull ProjectDto dto) {
        return new Project(
            dto.id,
            dto.name,
            Nullables.orEmpty(dto.description),
            dto.owner,
            AclDto.toModel(dto.acl),
            Nullables.orEmpty(dto.abcService),
            Nullables.orFalse(dto.onlyAuthRead),
            Nullables.orFalse(dto.onlyAuthPush),
            Nullables.orFalse(dto.onlySensorNameShards),
            Nullables.orFalse(dto.onlyNewFormatWrites),
            Nullables.orFalse(dto.onlyNewFormatReads),
            Nullables.orEmpty(dto.metricNameLabel),
            Nullables.orEpoch(dto.createdAt),
            Nullables.orEpoch(dto.updatedAt),
            Nullables.orEmpty(dto.createdBy),
            Nullables.orEmpty(dto.updatedBy),
            dto.version,
            Map.of()
        );
    }
}
