package ru.yandex.solomon.gateway.api.v2.dto;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.core.db.model.ProjectMenu;
import ru.yandex.solomon.core.exceptions.BadRequestException;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class ProjectMenuDto {

    private String id;
    private MenuItemDto[] items;
    private int version;
    private Instant createdAt;
    private Instant updatedAt;
    private String createdBy;
    private String updatedBy;

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public MenuItemDto[] getItems() {
        return items;
    }

    public void setItems(MenuItemDto[] items) {
        this.items = items;
    }

    public int getVersion() {
        return version;
    }

    public void setVersion(int version) {
        this.version = version;
    }

    public Instant getCreatedAt() {
        return createdAt;
    }

    public void setCreatedAt(Instant createdAt) {
        this.createdAt = createdAt;
    }

    public Instant getUpdatedAt() {
        return updatedAt;
    }
    public void setUpdatedAt(Instant updatedAt) {
        this.updatedAt = updatedAt;
    }

    public String getCreatedBy() {
        return createdBy;
    }

    public void setCreatedBy(String createdBy) {
        this.createdBy = createdBy;
    }

    public String getUpdatedBy() {
        return updatedBy;
    }

    public void setUpdatedBy(String updatedBy) {
        this.updatedBy = updatedBy;
    }

    public void validate() {
        if (StringUtils.isBlank(id)) {
            throw new BadRequestException("project id must not be blank");
        }
        if (items != null) {
            MenuItemDto.validateMenuItems(items, Collections.emptyMap());
        }
    }

    @Nonnull
    public static ProjectMenu toModel(@Nonnull ProjectMenuDto dto) {
        return new ProjectMenu(
            dto.id,
            MenuItemDto.toModelList(dto.items),
            dto.getVersion(),
            dto.getCreatedAt(),
            dto.getUpdatedAt(),
            dto.getCreatedBy(),
            dto.getUpdatedBy()
        );
    }

    @Nonnull
    public static ProjectMenuDto fromModel(@Nonnull ProjectMenu projectMenu) {
        MenuItemDto[] menuItems =
            Arrays.stream(projectMenu.getItems())
                .map(MenuItemDto::fromModel)
                .toArray(MenuItemDto[]::new);

        ProjectMenuDto dto = new ProjectMenuDto();
        dto.setId(projectMenu.getId());
        dto.setItems(menuItems);
        dto.setVersion(projectMenu.getVersion());
        dto.setCreatedAt(projectMenu.getCreatedAt());
        dto.setUpdatedAt(projectMenu.getUpdatedAt());
        dto.setCreatedBy(projectMenu.getCreatedBy());
        dto.setUpdatedBy(projectMenu.getUpdatedBy());
        return dto;
    }
}
