package ru.yandex.solomon.gateway.api.v2.dto;

import java.time.Instant;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.monitoring.api.v3.ShardSettings.PullProtocol;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.validators.IdValidator;
import ru.yandex.solomon.util.ServicePathEncoder;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.util.time.InstantUtils;

import static ru.yandex.solomon.gateway.api.v3.intranet.dto.ShardSettingsDtoConverter.fromPullProtocolModel;
import static ru.yandex.solomon.gateway.api.v3.intranet.dto.ShardSettingsDtoConverter.toPullProtocolModel;

/**
 * @author Sergey Polovko
 */
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class ServiceDto {

    private String id;
    private String name;
    private String projectId;

    private int port;
    private String path;
    private boolean addTsArgs;
    private int interval;     // in seconds
    private int gridSec;
    private PullProtocol protocol;

    private ServiceMetricConfDto sensorConf;

    private Integer sensorsTtlDays;
    private Instant deleteAfter;

    private String sensorNameLabel;
    private String tvmDestId;

    private Integer version;

    private Instant createdAt;
    private Instant updatedAt;
    private String createdBy;
    private String updatedBy;
    private String monitoringModel;

    public String getMonitoringModel() {
        return monitoringModel;
    }

    public void setMonitoringModel(String monitoringModel) {
        this.monitoringModel = monitoringModel;
    }

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getProjectId() {
        return projectId;
    }

    public void setProjectId(String projectId) {
        this.projectId = projectId;
    }

    public int getPort() {
        return port;
    }

    public void setPort(int port) {
        this.port = port;
    }

    public String getPath() {
        return path;
    }

    public void setPath(String path) {
        this.path = path;
    }

    public boolean isAddTsArgs() {
        return addTsArgs;
    }

    public void setAddTsArgs(boolean addTsArgs) {
        this.addTsArgs = addTsArgs;
    }

    public int getInterval() {
        return interval;
    }

    public void setInterval(int interval) {
        this.interval = interval;
    }

    public int getGridSec() {
        return gridSec;
    }

    public void setGridSec(int gridSec) {
        this.gridSec = gridSec;
    }

    public PullProtocol getProtocol() {
        return protocol;
    }

    public void setProtocol(PullProtocol protocol) {
        this.protocol = protocol;
    }

    public ServiceMetricConfDto getSensorConf() {
        return sensorConf;
    }

    public void setSensorConf(ServiceMetricConfDto sensorConf) {
        this.sensorConf = sensorConf;
    }

    public Integer getSensorsTtlDays() {
        return sensorsTtlDays;
    }

    public void setSensorsTtlDays(Integer sensorsTtlDays) {
        this.sensorsTtlDays = sensorsTtlDays;
    }

    public Instant getDeleteAfter() {
        return deleteAfter;
    }

    public void setDeleteAfter(Instant deleteAfter) {
        this.deleteAfter = deleteAfter;
    }

    public String getSensorNameLabel() {
        return sensorNameLabel;
    }

    public void setSensorNameLabel(String sensorNameLabel) {
        this.sensorNameLabel = sensorNameLabel;
    }

    public void setTvmDestId(String tvmDestId) {
        this.tvmDestId = tvmDestId;
    }

    public String getTvmDestId() {
        return tvmDestId;
    }

    public Integer getVersion() {
        return version;
    }

    public void setVersion(Integer version) {
        this.version = version;
    }

    public Instant getCreatedAt() {
        return createdAt;
    }

    public void setCreatedAt(Instant createdAt) {
        this.createdAt = createdAt;
    }

    public Instant getUpdatedAt() {
        return updatedAt;
    }

    public void setUpdatedAt(Instant updatedAt) {
        this.updatedAt = updatedAt;
    }

    public String getCreatedBy() {
        return createdBy;
    }

    public void setCreatedBy(String createdBy) {
        this.createdBy = createdBy;
    }

    public String getUpdatedBy() {
        return updatedBy;
    }

    public void setUpdatedBy(String updatedBy) {
        this.updatedBy = updatedBy;
    }

    public void validate() {
        IdValidator.ensureServiceIdValid(id, "service");
        if (StringUtils.isBlank(name)) {
            throw new BadRequestException("name cannot be blank");
        }
        if (StringUtils.isBlank(projectId)) {
            throw new BadRequestException("projectId cannot be blank");
        }
        if (port != 0 || !StringUtils.isEmpty(path)) {
            if (port == 0) {
                throw new BadRequestException("port must be defined");
            }
            if (port < 0 || port > 0xffff) {
                throw new BadRequestException("port out of range: " + port);
            }
            if (StringUtils.isBlank(path)) {
                throw new BadRequestException("path cannot be blank");
            }

            if (!path.startsWith("/")) {
                throw new BadRequestException("path must start with \"/\"");
            }

            try {
                String urlEncodedPath = ServicePathEncoder.encode(path);
            } catch (Exception e) {
                throw new BadRequestException("invalid path \" " + path + "\": " + e.getMessage());
            }
        }
        if (interval == 0) {
            interval = Service.DEFAULT_INTERVAL;
        } else {
            if (interval < 10 || interval > 3600) {
                throw new BadRequestException("interval must be in range [10, 3600] seconds");
            }
        }
        if (gridSec != Service.GRID_ABSENT && gridSec != Service.GRID_UNKNOWN) {
            if (gridSec < -1) {
                throw new BadRequestException("gridSec can't be less than -1");
            }
            long decimGridSec = TimeUnit.MINUTES.toSeconds(5);
            if (!InstantUtils.checkGridMultiplicity(decimGridSec, gridSec)) {
                throw new BadRequestException("gridSec(" + gridSec + ") must be multiple of 5m");
            }
        }

        if (deleteAfter != null && !deleteAfter.equals(Instant.EPOCH) && Instant.now().isAfter(deleteAfter)) {
            throw new BadRequestException("deleteAfter must be in our bright future");
        }
        if (sensorNameLabel != null) {
            ValidationUtils.validateMetricNameLabel(sensorNameLabel);
        }
        if (version != null && version < 0) {
            throw new BadRequestException("version cannot be negative");
        }
        if (sensorConf != null) {
            sensorConf.validate();
        }
    }

    @Nonnull
    public static Service toModel(@Nonnull ServiceDto dto) {
        var settings = getShardSettings(dto);
        return Service.newBuilder()
                .setId(dto.id)
                .setName(dto.name)
                .setProjectId(dto.projectId)
                .setAddTsArgs(dto.addTsArgs)
                .setMetricConf(ServiceMetricConfDto.toModel(dto.sensorConf))
                .setMetricNameLabel(dto.sensorNameLabel)
                .setVersion(Nullables.orZero(dto.version))
                .setCreatedAt(dto.createdAt)
                .setUpdatedAt(dto.updatedAt)
                .setCreatedBy(dto.createdBy)
                .setUpdatedBy(dto.updatedBy)
                .setShardSettings(settings)
                .build();
    }

    @Nonnull
    public static ServiceDto fromModel(@Nonnull Service service) {
        ServiceDto dto = new ServiceDto();
        dto.setId(service.getId());
        dto.setName(service.getName());
        dto.setProjectId(service.getProjectId());
        dto.setPort(ShardSettings.getPort(service.getShardSettings(), 0));
        dto.setPath(ShardSettings.getPath(service.getShardSettings(), ""));
        dto.setAddTsArgs(service.isAddTsArgs());
        dto.setInterval(ShardSettings.getInterval(service.getShardSettings(), 0));
        dto.setGridSec(Nullables.orZero(service.getShardSettings().getGrid()));
        dto.setProtocol(fromPullProtocolModel(
            ShardSettings.getProtocol(service.getShardSettings(), ShardSettings.PullProtocol.UNKNOWN)));

        if (!service.getMetricConf().isEmpty()) {
            dto.setSensorConf(ServiceMetricConfDto.fromModel(service.getMetricConf()));
        }

        if (service.getShardSettings().getMetricsTtl() > 0) {
            dto.setSensorsTtlDays(service.getShardSettings().getMetricsTtl());
        }

        if (!service.getMetricNameLabel().isEmpty()) {
            dto.setSensorNameLabel(service.getMetricNameLabel());
        }

        var pullSettings = service.getShardSettings().getPullSettings();
        dto.setTvmDestId(pullSettings == null ? "" : pullSettings.getTvmDestinationId());

        dto.setVersion(service.getVersion());
        dto.setCreatedAt(service.getCreatedAt());
        dto.setUpdatedAt(service.getUpdatedAt());
        dto.setCreatedBy(service.getCreatedBy());
        dto.setUpdatedBy(service.getUpdatedBy());
        dto.monitoringModel = service.getShardSettings().getType().name();
        return dto;
    }

    private static ShardSettings getShardSettings(@Nonnull ServiceDto service) {
        final ShardSettings.Type type;
        final ShardSettings.PullSettings pullSettings;
        if (Nullables.orEmpty(service.getPath()).isEmpty()) {
            type = ShardSettings.Type.PUSH;
            pullSettings = null;
        } else {
            type = ShardSettings.Type.PULL;
            pullSettings = ShardSettings.PullSettings.newBuilder()
                    .setPath(service.getPath())
                    .setPort(Nullables.orZero(service.getPort()))
                    .setAddTsArgs(service.isAddTsArgs())
                    .setProtocol(toPullProtocolModel(Nullables.orDefault(service.getProtocol(), PullProtocol.HTTP)))
                    .setTvmDestinationId(Nullables.orEmpty(service.tvmDestId))
                    .build();
        }
        ServiceMetricConf serviceMetricConf = ServiceMetricConfDto.toModel(service.sensorConf);
        return ShardSettings.of(
                type,
                pullSettings,
                service.getGridSec(),
                Nullables.orZero(service.getSensorsTtlDays()),
                DecimPolicy.UNDEFINED,
                ShardSettings.AggregationSettings.of(true, serviceMetricConf.getAggrRules(),serviceMetricConf.isRawDataMemOnly()),
                Nullables.orZero(service.getInterval())
        );
    }
}
