package ru.yandex.solomon.gateway.api.v2.dto;

import java.util.Arrays;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.core.conf.aggr.LabelValueExpr;
import ru.yandex.solomon.core.db.model.MetricAggregation;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.labels.LabelValidator;
import ru.yandex.solomon.util.collection.Nullables;



/**
 * @author Sergey Polovko
 */
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class ServiceMetricConfDto {

    private static final int MAX_AGGR_RULES_COUNT = 10;

    private static final AggrRuleDto[] EMPTY_AGGR_RULES = new AggrRuleDto[0];
    private static final PriorityRuleDto[] EMPTY_PRIORITY_RULES = new PriorityRuleDto[0];

    @JsonProperty
    private AggrRuleDto[] aggrRules;
    @JsonProperty
    private PriorityRuleDto[] priorityRules;
    @JsonProperty
    private boolean rawDataMemOnly;

    void validate() {
        if (aggrRules != null) {
            if (aggrRules.length > MAX_AGGR_RULES_COUNT) {
                throw new BadRequestException("too many aggregation rules, max count is " + MAX_AGGR_RULES_COUNT);
            }

            for (AggrRuleDto rule : aggrRules) {
                String[] cond = Nullables.orEmpty(rule.cond);
                String[] target = Nullables.orEmpty(rule.target);
                if (cond.length == 0 && target.length == 0) {
                    continue;
                }
                checkNonBlankStrings(cond, "aggrRule", "cond");
                checkNonBlankStrings(cond, "aggrRule", "cond");
                for (String c : cond) {
                    checkLabel(c, "cond in aggregation rule");
                }
                for (String t : target) {
                    checkLabel(t, "target in aggregation rule");
                }
            }
        }
    }

    @Nonnull
    public static ServiceMetricConf toModel(ServiceMetricConfDto dto) {
        if (dto == null) {
            return ServiceMetricConf.empty();
        }

        ServiceMetricConf.AggrRule[] aggrRules =
            Arrays.stream(dto.aggrRules == null ? EMPTY_AGGR_RULES : dto.aggrRules)
                .filter(r -> Nullables.orEmpty(r.cond).length != 0 && Nullables.orEmpty(r.target).length != 0)
                .map(r -> new ServiceMetricConf.AggrRule(r.cond, r.target, r.function))
                .toArray(ServiceMetricConf.AggrRule[]::new);

        return ServiceMetricConf.of(aggrRules, dto.rawDataMemOnly);
    }

    @Nonnull
    public static ServiceMetricConfDto fromModel(@Nonnull ServiceMetricConf conf) {
        ServiceMetricConfDto dto = new ServiceMetricConfDto();
        dto.aggrRules = Arrays.stream(conf.getAggrRules())
            .map(r -> {
                AggrRuleDto ruleDto = new AggrRuleDto();
                ruleDto.cond = r.getCond();
                ruleDto.target = r.getTarget();
                ruleDto.function = r.getAggr();
                return ruleDto;
            })
            .toArray(AggrRuleDto[]::new);
        dto.priorityRules = EMPTY_PRIORITY_RULES;
        dto.rawDataMemOnly = conf.isRawDataMemOnly();
        return dto;
    }

    /**
     * AGGREGATION RULE DTO
     */
    static class AggrRuleDto {
        @JsonProperty
        private String[] cond;
        @JsonProperty
        private String[] target;
        @Nullable
        @JsonProperty
        @JsonInclude(JsonInclude.Include.NON_NULL)
        private MetricAggregation function;
    }

    /**
     * PRIORITY RULE DTO
     */
    static class PriorityRuleDto {
        @JsonProperty
        private String target;
        @JsonProperty
        private int priority;
    }

    private static void checkNonBlankStrings(@Nonnull String[] strs, String ruleType, String fieldName) {
        if (strs.length == 0) {
            throw new BadRequestException(fieldName + " in " + ruleType + " cannot be an empty");
        }
        for (String str : strs) {
            if (StringUtils.isBlank(str)) {
                throw new BadRequestException(fieldName + " in " + ruleType + " cannot have blank element");
            }
        }
    }

    private static void checkLabel(@Nonnull String labelStr, String message) {
        String[] nameValue = StringUtils.split(labelStr, '=');
        if (nameValue.length != 2) {
            throw new BadRequestException(message + " must be in 'name=value' format, but got: '" + labelStr + '\'');
        }
        if (!LabelValidator.isValidNameChars(nameValue[0])) {
            throw new BadRequestException("invalid label name for " + message + ", got: '" + nameValue[0] + '\'');
        }
        if (!LabelValueExpr.isValid(nameValue[1])) {
            throw new BadRequestException("invalid label value expression for " + message + ", got: '" + nameValue[1] + '\'');
        }
    }
}
