package ru.yandex.solomon.gateway.api.v2.dto;

import java.text.DecimalFormat;
import java.time.Instant;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.core.conf.ShardsManager;
import ru.yandex.solomon.core.db.model.ClusterServiceNames;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.core.db.model.ValidationMode;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.validators.IdValidator;
import ru.yandex.solomon.coremon.client.CoremonClient;
import ru.yandex.solomon.util.UnknownShardLocation;
import ru.yandex.solomon.util.collection.Nullables;

import static ru.yandex.solomon.core.db.model.Shard.PARTITIONS_HARD_LIMIT;
import static ru.yandex.solomon.core.db.model.ShardSettings.AggregationSettings.EMPTY;

/**
 * @author Sergey Polovko
 */
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class ShardDto {

    private static final DecimalFormat DOUBLE_FORMATTER = new DecimalFormat("#.##");

    private String id;
    private int numId;

    private String projectId;
    private String clusterId;
    private String serviceId;

    private QuotasDto quotas;
    private CurrentQuotasDto estimatedQuotasCurrent;
    private Instant deleteAfter;
    private Integer sensorsTtlDays;
    private String sensorsTtlDaysDefinedIn;
    private List<String> hosts = List.of();
    private ValidationMode validationMode;
    private DecimPolicy decimPolicy;
    private String sensorNameLabel;
    private String sensorNameLabelDefinedIn;

    private Instant createdAt;
    private Instant updatedAt;
    private String createdBy;
    private String updatedBy;

    private ShardState state;
    private Integer version;
    private String clusterName;
    private String serviceName;

    private String monitoringModel;
    private String typeDefinedIn;
    private Integer targetErrors;
    private String quotasStatus;

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public long getNumId() {
        return numId;
    }

    public void setNumId(int numId) {
        this.numId = numId;
    }

    public String getProjectId() {
        return projectId;
    }

    public void setProjectId(String projectId) {
        this.projectId = projectId;
    }

    public String getClusterId() {
        return clusterId;
    }

    public void setClusterId(String clusterId) {
        this.clusterId = clusterId;
    }

    public String getServiceId() {
        return serviceId;
    }

    public void setServiceId(String serviceId) {
        this.serviceId = serviceId;
    }

    public String getClusterName() {
        return clusterName;
    }

    public void setClusterName(String clusterName) {
        this.clusterName = clusterName;
    }

    public String getServiceName() {
        return serviceName;
    }

    public void setServiceName(String serviceName) {
        this.serviceName = serviceName;
    }

    public QuotasDto getQuotas() {
        return quotas;
    }

    public void setQuotas(QuotasDto quotas) {
        this.quotas = quotas;
    }

    public Instant getDeleteAfter() {
        return deleteAfter;
    }

    public void setDeleteAfter(Instant deleteAfter) {
        this.deleteAfter = deleteAfter;
    }

    public Integer getSensorsTtlDays() {
        return sensorsTtlDays;
    }

    public void setSensorsTtlDays(Integer sensorsTtlDays) {
        this.sensorsTtlDays = sensorsTtlDays;
    }

    public List<String> getHosts() {
        return hosts;
    }

    public void setHosts(List<String> hosts) {
        this.hosts = hosts;
    }

    public Instant getCreatedAt() {
        return createdAt;
    }

    public void setCreatedAt(Instant createdAt) {
        this.createdAt = createdAt;
    }

    public Instant getUpdatedAt() {
        return updatedAt;
    }

    public void setUpdatedAt(Instant updatedAt) {
        this.updatedAt = updatedAt;
    }

    public String getCreatedBy() {
        return createdBy;
    }

    public void setCreatedBy(String createdBy) {
        this.createdBy = createdBy;
    }

    public String getUpdatedBy() {
        return updatedBy;
    }

    public void setUpdatedBy(String updatedBy) {
        this.updatedBy = updatedBy;
    }

    public ShardState getState() {
        return state;
    }

    public void setState(ShardState state) {
        this.state = state;
    }

    public Integer getVersion() {
        return version;
    }

    public void setVersion(Integer version) {
        this.version = version;
    }

    public void setValidationMode(ValidationMode validationMode) {
        this.validationMode = validationMode;
    }

    public ValidationMode getValidationMode() {
        return validationMode;
    }

    public void setSensorNameLabel(String sensorNameLabel) {
        this.sensorNameLabel = sensorNameLabel;
    }

    public String getSensorNameLabel() {
        return sensorNameLabel;
    }

    public void setDecimPolicy(DecimPolicy decimPolicy) {
        this.decimPolicy = decimPolicy;
    }

    public DecimPolicy getDecimPolicy() {
        return decimPolicy;
    }

    public CurrentQuotasDto getEstimatedQuotasCurrent() {
        return estimatedQuotasCurrent;
    }

    public void setEstimatedQuotasCurrent(CurrentQuotasDto estimatedQuotasCurrent) {
        this.estimatedQuotasCurrent = estimatedQuotasCurrent;
    }

    public String getSensorsTtlDaysDefinedIn() {
        return sensorsTtlDaysDefinedIn;
    }

    public void setSensorsTtlDaysDefinedIn(String sensorsTtlDaysDefinedIn) {
        this.sensorsTtlDaysDefinedIn = sensorsTtlDaysDefinedIn;
    }

    public String getMonitoringModel() {
        return monitoringModel;
    }

    public void setMonitoringModel(String monitoringModel) {
        this.monitoringModel = monitoringModel;
    }

    public String getTypeDefinedIn() {
        return typeDefinedIn;
    }

    public void setTypeDefinedIn(String typeDefinedIn) {
        this.typeDefinedIn = typeDefinedIn;
    }

    public Integer getTargetErrors() {
        return targetErrors;
    }

    public void setTargetErrors(Integer targetErrors) {
        this.targetErrors = targetErrors;
    }

    public String getQuotasStatus() {
        return quotasStatus;
    }

    public void setQuotasStatus(String quotasStatus) {
        this.quotasStatus = quotasStatus;
    }

    public String getSensorNameLabelDefinedIn() {
        return sensorNameLabelDefinedIn;
    }

    public void setSensorNameLabelDefinedIn(String sensorNameLabelDefinedIn) {
        this.sensorNameLabelDefinedIn = sensorNameLabelDefinedIn;
    }

    public void validate() {
        IdValidator.ensureShardIdValid(id, "shard");
        if (StringUtils.isBlank(projectId)) {
            throw new BadRequestException("project id cannot be blank");
        }
        if (StringUtils.isBlank(clusterId)) {
            throw new BadRequestException("cluster id cannot be blank");
        }
        if (StringUtils.isBlank(serviceId)) {
            throw new BadRequestException("service id cannot be blank");
        }
        if (deleteAfter != null && !deleteAfter.equals(Instant.EPOCH) && Instant.now().isAfter(deleteAfter)) {
            throw new BadRequestException("deleteAfter must be in our bright future");
        }
        ValidationUtils.validateMetricNameLabel(sensorNameLabel);
        if (version != null && version < 0) {
            throw new BadRequestException("version cannot be negative");
        }

        if (quotas != null && (quotas.getNumPartitions() <= 0 || quotas.getNumPartitions() > PARTITIONS_HARD_LIMIT)) {
            throw new BadRequestException(String.format("number of partitions should between [1, %d]", PARTITIONS_HARD_LIMIT));
        }
    }

    public static ShardDto fromModel(@Nonnull ShardsManager.ShardExtended shardExtended, @Nonnull CoremonClient coremonClient) {
        var shard = shardExtended.shard();
        var cluster = shardExtended.cluster();
        var service = shardExtended.service();
        ShardDto dto = new ShardDto();
        dto.setId(shard.getId());
        dto.setNumId(shard.getNumId());

        dto.setProjectId(shard.getProjectId());
        dto.setClusterId(shard.getClusterId());
        dto.setServiceId(shard.getServiceId());

        dto.setClusterName(shard.getClusterName());
        dto.setServiceName(shard.getServiceName());

        {
            QuotasDto quotasDto = new QuotasDto();
            quotasDto.setMaxSensorsPerUrl(shard.getMaxMetricsPerUrl());
            quotasDto.setMaxFileSensors(shard.getMaxFileMetrics());
            quotasDto.setMaxMemSensors(shard.getMaxMemMetrics());
            quotasDto.setMaxResponseSizeMb(shard.getMaxResponseSizeBytes() / (1 << 20));
            quotasDto.setNumPartitions(shard.getNumPartitions());
            dto.setQuotas(quotasDto);
        }

        if (shard.getShardSettings().getMetricsTtl() > 0) {
            dto.setSensorsTtlDays(shard.getShardSettings().getMetricsTtl());
        }

        try {
            dto.setHosts(coremonClient.shardHosts(shard.getNumId()));
        } catch (UnknownShardLocation ignore) {}

        dto.setValidationMode(shard.getValidationMode() == null ? ValidationMode.LEGACY_SKIP : shard.getValidationMode());

        dto.setDecimPolicy(shard.getShardSettings().getRetentionPolicy());

        dto.setSensorNameLabel(shard.getMetricNameLabel());

        dto.setCreatedAt(shard.getCreatedAt());
        dto.setUpdatedAt(shard.getUpdatedAt());
        dto.setCreatedBy(shard.getCreatedBy());
        dto.setUpdatedBy(shard.getUpdatedBy());

        dto.setVersion(shard.getVersion());
        dto.setState(shard.getState());

        if (!shardExtended.isFull()) {
            return dto;
        }
        if (cluster != null) {
            if (dto.sensorsTtlDays == null) {
                dto.sensorsTtlDays = ShardSettings.getMetricsTtlDays(cluster.getShardSettings(), null);
                dto.sensorsTtlDaysDefinedIn = "cluster";
            }
        }
        if (service != null) {
            dto.monitoringModel = service.getShardSettings().getType().name();
            dto.typeDefinedIn = "service";
            if (dto.sensorsTtlDays == null) {
                dto.sensorsTtlDays = ShardSettings.getMetricsTtlDays(service.getShardSettings(), null);
                dto.sensorsTtlDaysDefinedIn = "service";
            }
            if (StringUtils.isEmpty(dto.sensorNameLabel)) {
                dto.sensorNameLabel = service.getMetricNameLabel();
                dto.sensorNameLabelDefinedIn = "service";
            }
        }
        dto.targetErrors = shardExtended.targetErrors();
        dto.quotasStatus = shardExtended.quotas();
        if (!shardExtended.currentQuota().isEmpty()) {
            CurrentQuotasDto quotasDto = new CurrentQuotasDto();
            quotasDto.setSensorsPerUrl(Double.parseDouble(DOUBLE_FORMATTER.format(shardExtended.currentQuota().sensorsPerUrl())));
            quotasDto.setFileSensors(Double.parseDouble(DOUBLE_FORMATTER.format(shardExtended.currentQuota().fileSensors())));
            quotasDto.setMemSensors(Double.parseDouble(DOUBLE_FORMATTER.format(shardExtended.currentQuota().memSensors())));
            quotasDto.setResponseSizeMb(Double.parseDouble(DOUBLE_FORMATTER.format(shardExtended.currentQuota().responseSizeBytes() / (1 << 20))));
            dto.setEstimatedQuotasCurrent(quotasDto);
        }

        return dto;
    }

    @Nonnull
    public static Shard toModel(@Nonnull ShardDto dto, @Nonnull ClusterServiceNames names) {
        QuotasDto quotas = dto.getQuotas();
        ShardState state = Nullables.orDefault(dto.state, ShardState.DEFAULT);
        var settings = getShardSettings(dto);
        return new Shard(
            dto.id,
            dto.numId,
            dto.projectId,
            "",
            dto.clusterId,
            dto.serviceId,
            names.getClusterName(),
            names.getServiceName(),
            "",
            quotas == null ? null : quotas.getMaxSensorsPerUrl(),
            quotas == null ? null : quotas.getMaxFileSensors(),
            quotas == null ? null : quotas.getMaxMemSensors(),
            quotas == null ? null : quotas.getMaxResponseSizeMb() * (1 << 20),
            quotas == null ? null : quotas.getNumPartitions(),
            dto.getValidationMode(),
            dto.sensorNameLabel,
            settings,
            state,
            dto.version,
            dto.createdAt,
            dto.updatedAt,
            dto.createdBy,
            dto.updatedBy,
            Map.of()
        );
    }

    private static ShardSettings getShardSettings(@Nonnull ShardDto shard) {
        return ShardSettings.of(
                ShardSettings.Type.UNSPECIFIED,
                null,
                0,
                Nullables.orDefault(shard.getSensorsTtlDays(), 0),
                Nullables.orDefault(shard.getDecimPolicy(), DecimPolicy.UNDEFINED),
                EMPTY,
                0
        );
    }
}
