package ru.yandex.solomon.gateway.api.v2.dto;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.salmon.fetcher.proto.FetcherApiProto;
import ru.yandex.solomon.proto.UrlContentType;
import ru.yandex.solomon.proto.UrlStatusType;


/**
 * @author Sergey Polovko
 */
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class ShardTargetStatusDto {
    @ParametersAreNullableByDefault
    @JsonInclude(JsonInclude.Include.NON_NULL)
    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class ShardStatus {
        @JsonProperty
        private String shardId;
        @JsonProperty
        private UrlStatusType lastStatus;
        @JsonProperty
        private String lastError;
        @JsonProperty
        private int lastSensorsParsed;
        @JsonProperty
        private int lastSensorsOverflow;
    }

    @JsonProperty
    private String host;
    @JsonProperty
    private String url;
    @JsonProperty
    private String dc;
    @JsonProperty
    private Instant lastFetchTime;
    @JsonProperty
    private long lastFetchDurationMillis;
    /**
     * @deprecated Use lastUrlStatus and shardStatuses instead
     */
    @Deprecated
    @JsonProperty
    private UrlStatusType lastStatus; // DEPRECATED. TODO(SOLOMON-8837): remove after a successful release
    @JsonProperty
    private UrlContentType lastContentType;
    @Deprecated
    @JsonProperty
    private int lastSensorsParsed; // DEPRECATED. TODO(SOLOMON-8837): remove after a successful release
    @Deprecated
    @JsonProperty
    private int lastSensorsOverflow; // DEPRECATED. TODO(SOLOMON-8837): remove after a successful release
    @JsonProperty
    private int lastResponseBytes;
    /**
     * @deprecated Use lastUrlError and shardStatuses instead
     */
    @Deprecated
    @JsonProperty
    private String lastError; // DEPRECATED. TODO(SOLOMON-8837): remove after a successful release
    @JsonProperty
    private UrlStatusType lastUrlStatus;
    @JsonProperty
    private String lastUrlError;
    @JsonProperty
    private List<ShardStatus> shardStatuses;


    @Nonnull
    public static ShardTargetStatusDto fromModel(@Nonnull FetcherApiProto.TargetStatus targetStatus) {
        ShardTargetStatusDto dto = new ShardTargetStatusDto();
        dto.host = targetStatus.getHost();
        dto.url = targetStatus.getUrl();
        dto.dc = targetStatus.getDc();
        dto.lastFetchTime = Instant.ofEpochMilli(targetStatus.getLastFetchTime());
        dto.lastFetchDurationMillis = targetStatus.getLastFetchDurationMillis();
        dto.lastContentType = targetStatus.getLastContentType();
        dto.lastUrlStatus = targetStatus.getLastUrlStatus();
        dto.lastUrlError = targetStatus.getLastUrlError();
        dto.lastResponseBytes = targetStatus.getLastResponseBytes();
        dto.shardStatuses = new ArrayList<ShardStatus>(targetStatus.getShardStatusesCount());

        // TODO(SOLOMON-8837): remove after a successful release
        if (targetStatus.getShardStatusesCount() == 0 &&
            targetStatus.getLastUrlStatus() == UrlStatusType.UNKNOWN_STATUS_TYPE &&
            targetStatus.getLastStatus() != UrlStatusType.UNKNOWN_STATUS_TYPE)
        {
            // we got a response in the OLD api format, so let's create a fake shard with the status
            dto.shardStatuses = new ArrayList<ShardStatus>(1);

            var st = new ShardStatus();
            st.shardId = "";
            st.lastSensorsParsed = targetStatus.getLastMetricsParsed();
            st.lastSensorsOverflow = targetStatus.getLastMetricsOverflow();
            st.lastStatus = targetStatus.getLastStatus();
            st.lastError = targetStatus.getLastError();

            dto.shardStatuses.add(st);
        }

        for (var ss: targetStatus.getShardStatusesList()) {
            var status = new ShardTargetStatusDto.ShardStatus();
            status.shardId = ss.getShardId();
            status.lastStatus = ss.getLastStatus();
            status.lastError = ss.getLastError();
            status.lastSensorsOverflow = ss.getLastMetricsOverflow();
            status.lastSensorsParsed = ss.getLastMetricsParsed();
            dto.shardStatuses.add(status);
        }

        // old API compatibility. TODO(SOLOMON-8837): remove after a successful release
        if (dto.shardStatuses.isEmpty()) {
            // we got an error while fetching or parsing before even processing shard data, so let's just copy
            // the status
            dto.lastStatus = dto.lastUrlStatus;
            dto.lastError = dto.lastUrlError;
        } else {
            // for service providers (size > 1) the status value was the last shard status,
            // so it had no order guarantee and thus we can just select the first status
            // without actually breaking anything
            var shard = dto.shardStatuses.get(0);
            dto.lastStatus = shard.lastStatus;
            dto.lastError = shard.lastError;
            dto.lastSensorsParsed = shard.lastSensorsParsed;
            dto.lastSensorsOverflow = shard.lastSensorsOverflow;
        }

        return dto;
    }
}
