package ru.yandex.solomon.gateway.api.v2.dto;

import java.time.Instant;
import java.util.List;
import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monlib.metrics.labels.validate.StrictValidator;
import ru.yandex.solomon.auth.tvm.FakeTvmLogin;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.metrics.client.combined.DataLimits;

import static ru.yandex.solomon.metrics.client.combined.DataLimits.MAX_INTERVAL_MILLIS;
import static ru.yandex.solomon.metrics.client.combined.DataLimits.MIN_INTERVAL_MILLIS;


/**
 * @author rorewillo
 */
@ParametersAreNonnullByDefault
public class ValidationUtils {

    /**
     * @see ru.yandex.inside.passport.login.PassportLoginValidate#YANDEX_LOGIN_PATTERN
     */
    @SuppressWarnings("JavadocReference")
    private static final Pattern YANDEX_LOGIN_PATTERN =
        Pattern.compile("[a-zA-Z\\d]([\\w\\-.]{0,28}[a-zA-Z\\d])?");

    public static void validateLabelNames(List<String> labels) {
        for (String label : labels) {
            validateLabelName(label);
        }
    }

    private static void validateLabelName(String label) {
        if (StringUtils.isBlank(label)) {
            throw new BadRequestException("label cannot be blank");
        }
        if (LabelKeys.PROJECT.equals(label)) {
            throw new BadRequestException("\"project\" is forbidden label name");
        }
    }

    public static void validateInterval(Instant from, Instant to) {
        long fromMillis = from.toEpochMilli();
        long toMillis = to.toEpochMilli();

        if (fromMillis >= toMillis) {
            throw new BadRequestException("from cannot be greater or equal to: " + from + " >= " + to);
        }

        long interval = toMillis - fromMillis;

        if (interval < MIN_INTERVAL_MILLIS) {
            throw new BadRequestException(
                "interval cannot be less than " + DataLimits.MIN_INTERVAL_SECONDS + " seconds");
        }

        if (interval > MAX_INTERVAL_MILLIS) {
            throw new BadRequestException(
                "interval cannot be greater than " + DataLimits.MAX_INTERVAL_YEARS + " years");
        }
    }

    public static void validateLogin(String title, @Nullable String login) {
        if (StringUtils.isBlank(login)) {
            throw new BadRequestException(title + " cannot be blank");
        }

        if (!YANDEX_LOGIN_PATTERN.matcher(login).matches() && !FakeTvmLogin.isValid(login)) {
            throw new BadRequestException("invalid login: " + login);
        }
    }

    static void validateMetricNameLabel(@Nullable String metricNameLabel) {
        if (metricNameLabel == null || metricNameLabel.isEmpty()) {
            return;
        }

        if (LabelKeys.isShardKeyPart(metricNameLabel)) {
            throw new BadRequestException("metric name cannot be project/cluster/service");
        }

        String message = StrictValidator.validateKey(metricNameLabel);

        if (message != null) {
            throw new BadRequestException(message);
        }
    }
}
