package ru.yandex.solomon.gateway.api.v2.dto.data;

import java.time.Instant;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.gateway.api.v2.dto.ValidationUtils;
import ru.yandex.solomon.gateway.data.DataRequest;
import ru.yandex.solomon.gateway.data.DownsamplingOptions;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.util.jackson.InstantDeserializer;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Oleg Baryshnikov
 */
@ApiModel("DataRequest")
@ParametersAreNonnullByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class DataRequestDto {

    @ApiModelProperty(
        value = "Program",
        example = "\"{sensor='jvm.memory.used', host='cluster', memory='*'} / 1024\"",
        required = true
    )
    @JsonProperty
    // TODO: add alias 'query'
    public String program;

    @ApiModelProperty(
        name = "downsampling",
        value = "Downsampling options",
        position = 2
    )
    @JsonProperty
    @Nullable
    public DownsamplingDto downsampling;

    @ApiModelProperty(
        value = "Range start in UNIX-epoch milliseconds as integer or ISO-8601 format as string",
        required = true,
        position = 3
    )
    @JsonDeserialize(using = InstantDeserializer.class)
    public Instant from;

    @ApiModelProperty(
        value = "Range start in UNIX-epoch milliseconds as integer or ISO-8601 format as string",
        required = true,
        position = 4
    )
    @JsonDeserialize(using = InstantDeserializer.class)
    public Instant to;

    @ApiModelProperty(
        value = "Cluster abbreviation to load data from one of clusters or empty for cross-cluster loading",
        position = 5
    )
    public String forceCluster;

    @Nullable
    @ApiModelProperty(hidden = true)
    public Boolean useNewFormat;

    @Nullable
    @ApiModelProperty(
        value = "Override expression language version to use",
        position = 6
    )
    public SelVersion version;

    @ApiModelProperty(hidden = true)
    @JsonProperty
    public Boolean __oldMode;


    @ApiModelProperty(hidden = true)
    @JsonProperty
    public Boolean __summaryOnly;

    @Nonnull
    public DataRequest toModel(String projectId, String clusterName, RequestProducer producer, Instant deadline, String subjectId) {
        DataRequest.Builder builder = DataRequest.newBuilder();

        builder.setProjectId(projectId);
        builder.setClusterName(clusterName);
        builder.setProgram(Nullables.orEmpty(this.program));
        builder.setForceCluster(Nullables.orEmpty(this.forceCluster));

        builder.setUseNewFormat(Nullables.orFalse(this.useNewFormat));

        Interval interval = Interval.millis(this.from.toEpochMilli(), this.to.toEpochMilli());
        builder.setInterval(interval);

        if (this.downsampling != null) {
            DownsamplingOptions downsampling = this.downsampling.toModel();
            builder.setDownsampling(downsampling);
        }

        if (this.__oldMode != null && this.__oldMode) {
            builder.setOldMode(true);
        }

        if (Boolean.TRUE.equals(this.__summaryOnly)) {
            builder.setSummaryOnly(true);
        }

        if (this.version != null) {
            builder.setVersion(this.version);
        }

        builder.setProducer(producer);
        builder.setDeadline(deadline);
        builder.setSubjectId(subjectId);

        return builder.build();
    }

    public void validate() {
        if (this.downsampling != null) {
            this.downsampling.validate();
        }

        if (this.from == null) {
            throw new BadRequestException("missed parameter: \"from\"");
        }

        if (this.to == null) {
            throw new BadRequestException("missed parameter: \"to\"");
        }

        ValidationUtils.validateInterval(this.from, this.to);
    }
}
