package ru.yandex.solomon.gateway.api.v2.dto.data;

import java.util.Arrays;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueVector;
import ru.yandex.solomon.gateway.data.DataResponse;
import ru.yandex.solomon.metrics.client.combined.OldModeResult;

/**
 * @author Oleg Baryshnikov
 */
@ApiModel("DataResult")
@ParametersAreNullableByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class DataResultDto {

    @ApiModelProperty(
        value = "Boolean result value",
        example = "true"
    )
    @JsonProperty
    public Boolean bool;

    @ApiModelProperty(
        value = "Scalar result value",
        example = "100.0"
    )
    @JsonProperty
    public Double scalar;

    @ApiModelProperty(
        value = "String result value",
        example = "19.9%"
    )
    @JsonProperty
    public String string;

    @ApiModelProperty(
        value = "Duration result value",
        example = "1h"
    )
    @JsonProperty
    public String duration;

    @ApiModelProperty(value = "Time series result value")
    @JsonProperty
    public TimeseriesDto timeseries;

    @ApiModelProperty(value = "Object value")
    @JsonProperty
    public Map<String, DataResultDto> object;

    @ApiModelProperty(value = "Vector of values")
    @JsonProperty
    public DataResultDto[] vector;

    @JsonProperty
    public Boolean __truncated;

    @JsonProperty
    public Boolean __summary;

    @Nonnull
    public static DataResultDto fromModel(@Nonnull SelValue selValue) {
        DataResultDto dto = new DataResultDto();
        if (selValue.type().isDouble()) {
            dto.scalar = selValue.castToScalar().getValue();
        } else if (selValue.type().isBoolean()) {
            dto.bool = selValue.castToBoolean().getValue();
        } else if (selValue.type().isString()) {
            dto.string = selValue.castToString().getValue();
        } else if (selValue.type().isDuration()) {
            dto.duration = selValue.castToDuration().format();
        } else if (selValue.type().isGraphData()) {
            NamedGraphData namedGraphData = selValue.castToGraphData().getNamedGraphData();
            dto.timeseries = TimeseriesDto.fromModel(namedGraphData);
        } else if (selValue.type().isObject()) {
            dto.object = selValue.castToObject().getObject().entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, entry -> fromModel(entry.getValue())));
        } else if (selValue.type().isVector()) {
            dto.vector = fromVectorModel(selValue.castToVector());
        } else {
            // Interval and lambdas isn't supported
            throw new RuntimeException("unsupported result type: " + selValue.type());
        }
        return dto;
    }

    @Nonnull
    public static DataResultDto fromModel(@Nonnull DataResponse model) {
        SelValue selValue = model.getEvalResult();

        DataResultDto dto = fromModel(selValue);

        OldModeResult oldModeResult = model.getOldModeResult();
        if (oldModeResult.isTruncated()) {
            dto.__truncated = true;
        }
        if (oldModeResult.isSummary()) {
            dto.__summary = true;
        }

        return dto;
    }

    @Nonnull
    private static DataResultDto[] fromVectorModel(@Nonnull SelValueVector vector) {
        if (vector.type().isDoubleVector()) {
            return Arrays.stream(vector.doubleArray())
                .mapToObj(value -> {
                    DataResultDto dto = new DataResultDto();
                    dto.scalar = value;
                    return dto;
                })
                .toArray(DataResultDto[]::new);
        }

        return Arrays.stream(vector.valueArray())
            .map(DataResultDto::fromModel)
            .toArray(DataResultDto[]::new);
    }
}
