package ru.yandex.solomon.gateway.api.v2.dto.data;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.data.DownsamplingOptions;
import ru.yandex.solomon.gateway.data.DownsamplingType;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.metrics.client.combined.DataLimits;

/**
 * @author Oleg Baryshnikov
 */
@ApiModel("Downsampling")
@ParametersAreNonnullByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class DownsamplingDto {

    @ApiModelProperty(hidden = true)
    @JsonProperty
    public Integer points;

    @ApiModelProperty(
        value = "Max requested points count for each timeseries",
        example = "500"
    )
    @JsonProperty
    public Integer maxPoints;

    @ApiModelProperty(
        value = "Downsampling grid in millis",
        example = "3600000"
    )
    @JsonProperty
    public Long gridMillis;

    @ApiModelProperty(
        value = "Downsampling aggregation function",
        example = "MAX"
    )
    @JsonProperty
    public Aggregation aggregation;

    @ApiModelProperty(
        value = "Downsampling fill option",
        example = "PREVIOUS"
    )
    public OperationDownsampling.FillOption fill;

    @ApiModelProperty(
            name = "disabled",
            value = "Disable downsampling",
            example = "true"
    )
    public Boolean disabled;

    @ApiModelProperty(
        value = "Disable stepMillis logic",
        example = "true"
    )
    public Boolean ignoreMinStepMillis;

    public DownsamplingOptions toModel() {
        DownsamplingOptions.Builder builder = DownsamplingOptions.newBuilder();

        if (this.aggregation != null) {
            builder.setDownsamplingAggr(aggregation);
        }

        if (this.fill != null) {
            builder.setDownsamplingFill(fill);
        }

        if (this.gridMillis != null) {
            builder.setDownsamplingType(DownsamplingType.BY_INTERVAL);
            builder.setGridMillis(gridMillis);
        }

        if (this.ignoreMinStepMillis != null) {
            builder.setIgnoreMinStepMillis(this.ignoreMinStepMillis);
        }

        Integer maxPoints = this.maxPoints != null ? this.maxPoints : this.points;

        if (maxPoints != null) {
            builder.setDownsamplingType(DownsamplingType.BY_POINTS);
            builder.setPoints(maxPoints);
        }

        if (disabled != null && disabled) {
            builder.setDownsamplingType(DownsamplingType.OFF);
        }

        return builder.build();
    }

    public void validate() {
        int maxPoints = this.maxPoints != null ? this.maxPoints : (this.points != null ? this.points : 0);
        long gridMillis = this.gridMillis == null ? 0 : this.gridMillis;

        if (maxPoints < 0) {
            throw new BadRequestException(maxPoints + " cannot be negative");
        }

        if (maxPoints > 0 && maxPoints < DataLimits.MIN_POINTS_COUNT) {
            throw new BadRequestException("too few points: " + maxPoints + " < " + DataLimits.MIN_POINTS_COUNT);
        }

        if (maxPoints > DataLimits.MAX_POINTS_COUNT) {
            throw new BadRequestException("too many points: " + maxPoints + " > " + DataLimits.MAX_POINTS_COUNT);
        }

        if (gridMillis < 0) {
            throw new BadRequestException(gridMillis + " cannot be negative");
        }
    }
}
