package ru.yandex.solomon.gateway.api.v2.dto.data;

import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.SummaryInt64Snapshot;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.gateway.api.v2.dto.data.values.DoubleValuesDto;
import ru.yandex.solomon.gateway.api.v2.dto.data.values.HistogramDto;
import ru.yandex.solomon.gateway.api.v2.dto.data.values.HistogramValuesDto;
import ru.yandex.solomon.gateway.api.v2.dto.data.values.LogHistogramValuesDto;
import ru.yandex.solomon.gateway.api.v2.dto.data.values.LongValuesDto;
import ru.yandex.solomon.gateway.api.v2.dto.data.values.OldStatsDto;
import ru.yandex.solomon.gateway.api.v2.dto.data.values.ShortLogHistogramDto;
import ru.yandex.solomon.gateway.api.v2.dto.data.values.SummaryDoubleDto;
import ru.yandex.solomon.gateway.api.v2.dto.data.values.SummaryDoubleValuesDto;
import ru.yandex.solomon.gateway.api.v2.dto.data.values.SummaryInt64Dto;
import ru.yandex.solomon.gateway.api.v2.dto.data.values.SummaryInt64ValuesDto;
import ru.yandex.solomon.gateway.api.v2.dto.data.values.ValuesDto;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.aggregation.DoubleSummary;
import ru.yandex.solomon.model.timeseries.aggregation.TimeseriesSummary;
import ru.yandex.solomon.model.type.LogHistogram;

/**
 * @author Oleg Baryshnikov
 */
@ApiModel("Timeseries")
@ParametersAreNonnullByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class TimeseriesDto {

    @ApiModelProperty(
        value = "Alias",
        example = "Man 95%",
        position = 0
    )
    @JsonProperty
    public String alias;

    @ApiModelProperty(
        value = "Metric type",
        example = "DGAUGE",
        required = true,
        position = 1)
    @JsonProperty
    public MetricType kind; // should be removed

    @ApiModelProperty(
        value = "Metric type",
        example = "DGAUGE",
        required = true,
        position = 1)
    @JsonProperty
    public MetricType type;

    @ApiModelProperty(
        value = "Metric name",
        required = false,
        position = 2)
    @JsonProperty
    public String name;

    @ApiModelProperty(
        value = "Labels of metric",
        required = true,
        position = 3
    )
    @JsonProperty
    public Map<String, String> labels;

    @ApiModelProperty(
        value = "Array with timestamps of metric timeseries",
        required = true,
        position = 4
    )
    @JsonProperty
    public long[] timestamps;

    @ApiModelProperty(
        value = "Array with values of metric timeseries",
        required = true,
        position = 5
    )
    @JsonProperty
    public ValuesDto values;

    @JsonProperty
    @Nullable
    public Boolean __show;

    @JsonProperty
    @Nullable
    public OldStatsDto __oldStats;

    public static TimeseriesDto fromModel(NamedGraphData namedGraphData) {
        TimeseriesDto dto = new TimeseriesDto();

        dto.alias = namedGraphData.getAlias();
        dto.kind = namedGraphData.getType();
        dto.type = namedGraphData.getType();
        if (!namedGraphData.getMetricName().isEmpty()) {
            dto.name = namedGraphData.getMetricName();
        }
        dto.labels = namedGraphData.getLabels().toMap();
        fillTimeseries(dto, namedGraphData);

        TimeseriesSummary summary = namedGraphData.getSummary();
        if (summary != null) {
            if (summary instanceof DoubleSummary doubleSummary) {
                dto.__oldStats = OldStatsDto.fromModel(doubleSummary);
            }
        }

        return dto;
    }

    private static void fillTimeseries(
            TimeseriesDto dto,
            NamedGraphData namedGraphData)
    {
        var data = namedGraphData.getAggrGraphDataArrayList();
        if (data.isEmpty()) {
            dto.values = new DoubleValuesDto(new double[0]);
            return;
        }

        var type = namedGraphData.getDataType();
        switch (type) {
            case DGAUGE -> fillDoubleTimeseries(dto, data);
            case IGAUGE -> fillLongTimeseries(dto, data);
            case HIST -> fillHistogramTimeseries(dto, data);
            case LOG_HISTOGRAM -> fillLogHistogramTimeseries(dto, data);
            case ISUMMARY -> fillInt64SummaryTimeseries(dto, data);
            case DSUMMARY -> fillDoubleSummaryTimeseries(dto, data);
            default -> throw new RuntimeException("unknown metric type: " + type);
        }
    }

    private static void fillDoubleTimeseries(
        TimeseriesDto dto,
        AggrGraphDataIterable data)
    {
        int count = data.getRecordCount();

        long[] timestamps = new long[count];
        double[] values = new double[count];
        AggrGraphDataListIterator iterator = data.iterator();

        RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();

        try {
            for (int i = 0; iterator.next(point); ++i) {
                timestamps[i] = point.tsMillis;
                values[i] = point.getValueDivided();
            }
        } finally {
            point.recycle();
        }

        dto.timestamps = timestamps;
        dto.values = new DoubleValuesDto(values);
    }

    private static void fillLongTimeseries(
        TimeseriesDto dto,
        AggrGraphDataIterable data)
    {
        int count = data.getRecordCount();

        long[] timestamps = new long[count];
        long[] values = new long[count];
        AggrGraphDataListIterator iterator = data.iterator();

        RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();
        try {
            for (int i = 0; iterator.next(point); ++i) {
                timestamps[i] = point.tsMillis;
                values[i] = point.longValue;
            }
        } finally {
            point.recycle();
        }

        dto.timestamps = timestamps;
        dto.values = new LongValuesDto(values);
    }

    private static void fillHistogramTimeseries(
        TimeseriesDto dto,
        AggrGraphDataIterable data)
    {
        int count = data.getRecordCount();

        long[] timestamps = new long[count];
        HistogramDto[] values = new HistogramDto[count];
        AggrGraphDataListIterator iterator = data.iterator();

        RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();
        try {
            for (int i = 0; iterator.next(point); ++i) {
                timestamps[i] = point.tsMillis;
                var histogram = point.histogram;

                final HistogramDto histogramDto;

                if (histogram != null) {
                    histogramDto = HistogramDto.fromModel(histogram);
                } else {
                    histogramDto = null;
                }

                values[i] = histogramDto;
            }
        } finally {
            point.recycle();
        }

        dto.timestamps = timestamps;
        dto.values = new HistogramValuesDto(values);
    }

    private static void fillLogHistogramTimeseries(
        TimeseriesDto dto,
        AggrGraphDataIterable data) {
        int count = data.getRecordCount();

        long[] timestamps = new long[count];
        ShortLogHistogramDto[] values = new ShortLogHistogramDto[count];
        AggrGraphDataListIterator iterator = data.iterator();

        RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();
        try {
            for (int i = 0; iterator.next(point); ++i) {
                timestamps[i] = point.tsMillis;
                LogHistogram logHistogram = point.logHistogram;

                final ShortLogHistogramDto logHistogramDto;

                if (logHistogram != null) {
                    logHistogramDto = ShortLogHistogramDto.fromModel(logHistogram);
                } else {
                    logHistogramDto = null;
                }

                values[i] = logHistogramDto;
            }
        } finally {
            point.recycle();
        }

        dto.timestamps = timestamps;
        dto.values = new LogHistogramValuesDto(values);
    }

    private static void fillDoubleSummaryTimeseries(
        TimeseriesDto dto,
        AggrGraphDataIterable data)
    {
        int count = data.getRecordCount();

        long[] timestamps = new long[count];
        SummaryDoubleDto[] values = new SummaryDoubleDto[count];
        AggrGraphDataListIterator iterator = data.iterator();

        RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();
        try {
            for (int i = 0; iterator.next(point); ++i) {
                timestamps[i] = point.tsMillis;

                SummaryDoubleSnapshot summary = point.summaryDouble;

                final SummaryDoubleDto summaryDto;

                if (summary == null) {
                    summaryDto = null;
                } else {
                    summaryDto = SummaryDoubleDto.fromModel(summary);
                }

                values[i] = summaryDto;
            }
        } finally {
            point.recycle();
        }

        dto.timestamps = timestamps;
        dto.values = new SummaryDoubleValuesDto(values);
    }

    private static void fillInt64SummaryTimeseries(
        TimeseriesDto dto,
        AggrGraphDataIterable data)
    {
        int count = data.getRecordCount();

        long[] timestamps = new long[count];
        SummaryInt64Dto[] values = new SummaryInt64Dto[count];
        AggrGraphDataListIterator iterator = data.iterator();

        RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();
        try {
            for (int i = 0; iterator.next(point); ++i) {
                timestamps[i] = point.tsMillis;

                SummaryInt64Snapshot summary = point.summaryInt64;

                final SummaryInt64Dto summaryDto;

                if (summary == null) {
                    summaryDto = null;
                } else {
                    summaryDto = SummaryInt64Dto.fromModel(summary);
                }

                values[i] = summaryDto;
            }
        } finally {
            point.recycle();
        }

        dto.timestamps = timestamps;
        dto.values = new SummaryInt64ValuesDto(values);
    }
}
