package ru.yandex.solomon.gateway.api.v2.managers;

import java.time.Instant;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.GatewayTimeoutException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.core.exceptions.ServiceUnavailableException;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.metabase.protobuf.LabelValidationFilter;
import ru.yandex.solomon.metrics.client.FindRequest;
import ru.yandex.solomon.metrics.client.LabelNamesRequest;
import ru.yandex.solomon.metrics.client.LabelNamesResponse;
import ru.yandex.solomon.metrics.client.LabelValuesRequest;
import ru.yandex.solomon.metrics.client.LabelValuesResponse;
import ru.yandex.solomon.metrics.client.MetabaseStatus;
import ru.yandex.solomon.metrics.client.MetricNamesRequest;
import ru.yandex.solomon.metrics.client.MetricNamesResponse;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.combined.DataLimits;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;
import ru.yandex.solomon.ydb.page.TokenPageOptions;

/**
 * @author Oleg Baryshnikov
 */
@Component
public class MetricsManager {

    private final MetricsClient metricsClient;
    private final ClusterMapper clusterMapper;

    @Autowired
    public MetricsManager(
        MetricsClient metricsClient,
        ClusterMapper clusterMapper)
    {
        this.metricsClient = metricsClient;
        this.clusterMapper = clusterMapper;
    }

    public CompletableFuture<PagedResult<MetricKey>> findMetrics(
        Selectors selectors,
        String clusterAbbr,
        PageOptions pageOptions,
        boolean useNewFormat)
    {
        String destination = getDestination(clusterAbbr);

        validateShardPagination(pageOptions.getOffset(), destination, selectors);

        FindRequest.Builder requestBuilder =
            FindRequest.newBuilder()
                .setSelectors(selectors)
                .setDestination(destination)
                .setUseNewFormat(useNewFormat);

        if (pageOptions.isLimited()) {
            requestBuilder.setLimit(pageOptions.getSize());
            requestBuilder.setOffset(pageOptions.getOffset());
        }

        FindRequest request = requestBuilder.build();

        return metricsClient.find(request)
            .thenApply(response -> {
                checkMetabaseResponse(response.getStatus());

                return PagedResult.of(
                    response.getMetrics(),
                    pageOptions,
                    response.getTotalCount()
                );
            });
    }

    public CompletableFuture<TokenBasePage<MetricKey>> findMetrics(
        Selectors selectors,
        String clusterAbbr,
        TokenPageOptions pageOptions,
        boolean useNewFormat)
    {
        String destination = getDestination(clusterAbbr);

        validateShardPagination(pageOptions.getOffset(), destination, selectors);

        FindRequest.Builder requestBuilder =
            FindRequest.newBuilder()
                .setSelectors(selectors)
                .setDestination(destination)
                .setUseNewFormat(useNewFormat)
                .setLimit(pageOptions.getSize() + 1)
                .setOffset(pageOptions.getOffset());

        FindRequest request = requestBuilder.build();

        return metricsClient.find(request)
            .thenApply(response -> {
                checkMetabaseResponse(response.getStatus());
                return TokenBasePage.of(response.getMetrics(), pageOptions);
            });
    }

    private void validateShardPagination(int offset, @Nullable String destination, Selectors labelSelectors) {
        if (offset > 0) {
            if (!ShardSelectors.isSingleShard(labelSelectors)) {
                throw new BadRequestException("pagination offset isn't implemented for cross-shard selectors");
            }

            if (destination == null || destination.isEmpty()) {
                throw new BadRequestException("pagination offset isn't implemented for cross-cluster request");
            }
        }
    }

    public CompletableFuture<MetricNamesResponse> findMetricNames(
        Selectors selectors,
        String text,
        LabelValidationFilter validationFilter,
        int limit,
        String clusterAbbr,
        Instant deadline)
    {
        if (limit < 0) {
            throw new BadRequestException("Metric names limit cannot be negative: " + limit);
        }
        if (limit > DataLimits.MAX_METRIC_NAMES_COUNT) {
            throw new BadRequestException("Metric names limit cannot be greater than " + DataLimits.MAX_METRIC_NAMES_COUNT + ": " + limit);
        }

        MetricNamesRequest request =
            MetricNamesRequest.newBuilder()
                .setSelectors(selectors)
                .setLimit(limit)
                .setTextSearch(text)
                .setValidationFilter(validationFilter)
                .setDestination(getDestination(clusterAbbr))
                .setDeadline(deadline)
                .build();

        return metricsClient.metricNames(request)
            .thenApply(response -> {
                checkMetabaseResponse(response.getStatus());
                return response;
            });
    }

    public CompletableFuture<LabelValuesResponse> findLabelValues(
        List<String> labels,
        Selectors selectors,
        String text,
        LabelValidationFilter validationFilter,
        int limit,
        boolean useNewFormat,
        String clusterAbbr,
        Instant deadline)
    {
        if (limit < 0) {
            throw new BadRequestException("label values limit cannot be negative: " + limit);
        }
        if (limit > DataLimits.MAX_LABEL_VALUES_COUNT) {
            throw new BadRequestException("label values limit cannot be greater than " + DataLimits.MAX_LABEL_VALUES_COUNT + ": " + limit);
        }

        LabelValuesRequest request =
            LabelValuesRequest.newBuilder()
                .setSelectors(selectors)
                .setNames(new HashSet<>(labels))
                .setLimit(limit)
                .setTextSearch(text)
                .setValidationFilter(validationFilter)
                .setUseNewFormat(useNewFormat)
                .setDestination(getDestination(clusterAbbr))
                .setDeadline(deadline)
                .build();

        return metricsClient.labelValues(request)
            .thenApply(response -> {
                checkMetabaseResponse(response.getStatus());

                return response;
            });
    }

    public CompletableFuture<LabelNamesResponse> findLabelNames(
        Selectors selectors,
        boolean useNewFormat,
        String clusterAbbr,
        Instant deadline)
    {
        LabelNamesRequest request =
            LabelNamesRequest.newBuilder()
                .setSelectors(selectors)
                .setUseNewFormat(useNewFormat)
                .setDestination(getDestination(clusterAbbr))
                .setDeadline(deadline)
                .build();

        return metricsClient.labelNames(request)
            .thenApply(response -> {
                checkMetabaseResponse(response.getStatus());
                return response;
            });
    }

    @Nullable
    private String getDestination(String clusterAbbr) {
        return clusterMapper.byParamOrNull(clusterAbbr);
    }

    private static void checkMetabaseResponse(MetabaseStatus status) {
        String message = Nullables.orEmpty(status.getDescription());
        switch (status.getCode()) {
            case OK:
                return;

            case NOT_FOUND:
            case SHARD_NOT_FOUND:
                throw new NotFoundException(message);

            case SHARD_WRITE_ONLY:
            case INVALID_REQUEST:
                throw new BadRequestException(message);

            case DEADLINE_EXCEEDED:
                throw new GatewayTimeoutException(message);

            case RESOURCE_EXHAUSTED:
            case NODE_UNAVAILABLE:
            case SHARD_NOT_READY:
                throw new ServiceUnavailableException(message);

            default:
                throw new RuntimeException("Metabase error " + status.getCode().name() + ": " + status.getDescription());
        }
    }
}
