package ru.yandex.solomon.gateway.api.v3.cloud.dto;

import java.util.List;
import java.util.concurrent.Callable;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Int64Value;
import yandex.cloud.api.monitoring.v3.ChartWidgetOuterClass;
import yandex.cloud.api.monitoring.v3.DashboardOuterClass;
import yandex.cloud.api.monitoring.v3.DashboardOuterClass.Dashboard;
import yandex.cloud.api.monitoring.v3.DashboardServiceOuterClass;
import yandex.cloud.api.monitoring.v3.DownsamplingOuterClass;
import yandex.cloud.api.monitoring.v3.ParametrizationOuterClass;
import yandex.cloud.api.monitoring.v3.TextWidgetOuterClass;
import yandex.cloud.api.monitoring.v3.TitleWidgetOuterClass;
import yandex.cloud.api.monitoring.v3.WidgetOuterClass;

import ru.yandex.monitoring.api.v3.CreateDashboardRequest;
import ru.yandex.monitoring.api.v3.DeleteDashboardRequest;
import ru.yandex.monitoring.api.v3.GetDashboardRequest;
import ru.yandex.monitoring.api.v3.ListDashboardsRequest;
import ru.yandex.monitoring.api.v3.ListDashboardsResponse;
import ru.yandex.monitoring.api.v3.UpdateDashboardRequest;
import ru.yandex.monitoring.v3.cloud.ChartWidget;
import ru.yandex.monitoring.v3.cloud.CustomParameter;
import ru.yandex.monitoring.v3.cloud.Downsampling;
import ru.yandex.monitoring.v3.cloud.LabelValuesParameter;
import ru.yandex.monitoring.v3.cloud.Parameter;
import ru.yandex.monitoring.v3.cloud.Parametrization;
import ru.yandex.monitoring.v3.cloud.TextParameter;
import ru.yandex.monitoring.v3.cloud.TextWidget;
import ru.yandex.monitoring.v3.cloud.TitleWidget;
import ru.yandex.monitoring.v3.cloud.Widget;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.DashboardDtoConverter;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class CloudDashboardDtoConverter {

    public static Dashboard fromAlphaDashboardModel(ru.yandex.monitoring.v3.cloud.Dashboard oldDashboard, String localId) {
        var widgets = oldDashboard.getWidgetsList().stream()
                .map(CloudDashboardDtoConverter::fromWidgetModel)
                .collect(Collectors.toList());

        var parametrization = fromParametrizationModel(oldDashboard.getParametrization());
        return Dashboard.newBuilder()
                .setFolderId(oldDashboard.getFolderId())
                .setId(oldDashboard.getId())
                .setName(localId)
                .setDescription(oldDashboard.getDescription())
                .addAllWidgets(widgets)
                .setParametrization(parametrization)
                .setCreatedAt(oldDashboard.getCreatedAt())
                .setModifiedAt(oldDashboard.getUpdatedAt())
                .setModifiedBy(oldDashboard.getUpdatedBy())
                .setCreatedBy(oldDashboard.getCreatedBy())
                .putAllLabels(oldDashboard.getLabelsMap())
                .setTitle(oldDashboard.getName())
                .setEtag(String.valueOf(oldDashboard.getVersion()))
                .build();
    }

    public static ru.yandex.monitoring.v3.cloud.Dashboard toAlphaDashboardModel(Dashboard newDashboard) {
        var widgets = newDashboard.getWidgetsList().stream()
                .map(CloudDashboardDtoConverter::toWidgetModel)
                .collect(Collectors.toList());

        var parametrization = toParametrizationModel(newDashboard.getParametrization());
        return ru.yandex.monitoring.v3.cloud.Dashboard.newBuilder()
                .setFolderId(newDashboard.getFolderId())
                .setId(newDashboard.getId())
                .setName(newDashboard.getTitle())
                .setDescription(newDashboard.getDescription())
                .addAllWidgets(widgets)
                .setParametrization(parametrization)
                .setCreatedAt(newDashboard.getCreatedAt())
                .setUpdatedAt(newDashboard.getModifiedAt())
                .setCreatedBy(newDashboard.getCreatedBy())
                .setUpdatedBy(newDashboard.getModifiedBy())
                .putAllLabels(newDashboard.getLabelsMap())
                .setVersion(DashboardDtoConverter.parseVersion(newDashboard.getEtag()))
                .build();
    }

    private static WidgetOuterClass.Widget fromWidgetModel(Widget widget) {
        var builder = WidgetOuterClass.Widget.newBuilder();
        builder.setPosition(fromPositionModel(widget.getPosition()));
        switch (widget.getWidgetCase()) {
            case TEXT -> {
                var text = widget.getText();
                builder.setText(TextWidgetOuterClass.TextWidget.newBuilder()
                        .setText(text.getText()));
            }
            case TITLE -> {
                var title = widget.getTitle();
                builder.setTitle(TitleWidgetOuterClass.TitleWidget.newBuilder()
                        .setText(title.getText())
                        .setSizeValue((title.getSizeValue())));
            }
            case CHART -> {
                var chart = widget.getChart();
                builder.setChart(fromChartWidgetModel(chart));
            }
        }
        return builder.build();
    }

    private static Widget toWidgetModel(WidgetOuterClass.Widget widget) {
        var builder = Widget.newBuilder();
        builder.setPosition(toPositionModel(widget.getPosition()));
        switch (widget.getWidgetCase()) {
            case TEXT -> {
                var text = widget.getText();
                builder.setText(TextWidget.newBuilder()
                        .setText(text.getText()));
            }
            case TITLE -> {
                var title = widget.getTitle();
                builder.setTitle(TitleWidget.newBuilder()
                        .setText(title.getText())
                        .setSizeValue((title.getSizeValue())));
            }
            case CHART -> {
                var chart = widget.getChart();
                builder.setChart(toChartWidgetModel(chart));
            }
        }
        return builder.build();
    }

    private static WidgetOuterClass.Widget.LayoutPosition fromPositionModel(Widget.LayoutPosition position) {
        return WidgetOuterClass.Widget.LayoutPosition.newBuilder()
                .setX(position.getX())
                .setY(position.getY())
                .setW(position.getW())
                .setH(position.getH())
                .build();
    }

    private static Widget.LayoutPosition toPositionModel(WidgetOuterClass.Widget.LayoutPosition position) {
        return Widget.LayoutPosition.newBuilder()
                .setX(position.getX())
                .setY(position.getY())
                .setW(position.getW())
                .setH(position.getH())
                .build();
    }

    private static ChartWidgetOuterClass.ChartWidget fromChartWidgetModel(ChartWidget chart) {
        var builder = ChartWidgetOuterClass.ChartWidget.newBuilder()
                .setId(chart.getId())
                .setTitle(chart.getTitle())
                .setDescription(chart.getDescription())
                .setDisplayLegend(chart.getDisplayLegend())
                .setFreezeValue(chart.getFreezeValue())
                .addAllSeriesOverrides(fromSeriesOverridesModelList(chart.getSeriesOverridesList()));

        if (chart.hasQueries()) {
            builder.setQueries(fromQueriesModel(chart.getQueries()));
        }
        if (chart.hasVisualizationSettings()) {
            builder.setVisualizationSettings(fromVisualizationSettingsModel(chart.getVisualizationSettings()));
        }
        if (chart.hasNameHidingSettings()) {
            builder.setNameHidingSettings(fromNameHidingSettingsModel(chart.getNameHidingSettings()));
        }

        return builder.build();
    }

    private static ChartWidget toChartWidgetModel(ChartWidgetOuterClass.ChartWidget chart) {
        var builder = ChartWidget.newBuilder()
                .setId(chart.getId())
                .setTitle(chart.getTitle())
                .setDescription(chart.getDescription())
                .setDisplayLegend(chart.getDisplayLegend())
                .setFreezeValue(chart.getFreezeValue())
                .addAllSeriesOverrides(toSeriesOverridesModelList(chart.getSeriesOverridesList()));

        if (chart.hasQueries()) {
            builder.setQueries(toQueriesModel(chart.getQueries()));
        }
        if (chart.hasVisualizationSettings()) {
            builder.setVisualizationSettings(toVisualizationSettingsModel(chart.getVisualizationSettings()));
        }
        if (chart.hasNameHidingSettings()) {
            builder.setNameHidingSettings(toNameHidingSettingsModel(chart.getNameHidingSettings()));
        }

        return builder.build();
    }

    private static List<ChartWidgetOuterClass.ChartWidget.SeriesOverrides> fromSeriesOverridesModelList(List<ChartWidget.SeriesOverrides> seriesOverridesList) {
        return seriesOverridesList.stream()
                .map(CloudDashboardDtoConverter::fromSeriesOverridesModel)
                .collect(Collectors.toList());
    }

    private static List<ChartWidget.SeriesOverrides> toSeriesOverridesModelList(List<ChartWidgetOuterClass.ChartWidget.SeriesOverrides> seriesOverridesList) {
        return seriesOverridesList.stream()
                .map(CloudDashboardDtoConverter::toSeriesOverridesModel)
                .collect(Collectors.toList());
    }

    private static ChartWidgetOuterClass.ChartWidget.SeriesOverrides fromSeriesOverridesModel(ChartWidget.SeriesOverrides seriesOverrides) {
        var builder = ChartWidgetOuterClass.ChartWidget.SeriesOverrides.newBuilder()
                .setSettings(fromSeriesOverridesSettingsModel(seriesOverrides.getSettings()));
        switch (seriesOverrides.getTypeCase()) {
            case NAME -> builder.setName(seriesOverrides.getName());
            case TARGET_INDEX -> builder.setTargetIndex(seriesOverrides.getTargetIndex());
        }
        return builder.build();
    }

    private static ChartWidget.SeriesOverrides toSeriesOverridesModel(ChartWidgetOuterClass.ChartWidget.SeriesOverrides seriesOverrides) {
        var builder = ChartWidget.SeriesOverrides.newBuilder()
                .setSettings(toSeriesOverridesSettingsModel(seriesOverrides.getSettings()));
        switch (seriesOverrides.getTypeCase()) {
            case NAME -> builder.setName(seriesOverrides.getName());
            case TARGET_INDEX -> builder.setTargetIndex(seriesOverrides.getTargetIndex());
        }
        return builder.build();
    }

    private static ChartWidgetOuterClass.ChartWidget.SeriesOverrides.SeriesOverrideSettings fromSeriesOverridesSettingsModel(ChartWidget.SeriesOverrides.SeriesOverrideSettings settings) {
        return ChartWidgetOuterClass.ChartWidget.SeriesOverrides.SeriesOverrideSettings.newBuilder()
                .setName(settings.getName())
                .setColor(settings.getColor())
                .setTypeValue(settings.getTypeValue())
                .setStackName(settings.getStackName())
                .setGrowDown(settings.getGrowDown())
                .setYaxisPositionValue(settings.getYaxisPositionValue())
                .build();
    }

    private static ChartWidget.SeriesOverrides.SeriesOverrideSettings toSeriesOverridesSettingsModel(ChartWidgetOuterClass.ChartWidget.SeriesOverrides.SeriesOverrideSettings settings) {
        return ChartWidget.SeriesOverrides.SeriesOverrideSettings.newBuilder()
                .setName(settings.getName())
                .setColor(settings.getColor())
                .setTypeValue(settings.getTypeValue())
                .setStackName(settings.getStackName())
                .setGrowDown(settings.getGrowDown())
                .setYaxisPositionValue(settings.getYaxisPositionValue())
                .build();
    }

    private static ChartWidgetOuterClass.ChartWidget.NameHidingSettings fromNameHidingSettingsModel(ChartWidget.NameHidingSettings nameHidingSettings) {
        return ChartWidgetOuterClass.ChartWidget.NameHidingSettings.newBuilder()
                .addAllNames(nameHidingSettings.getNamesList())
                .setPositive(nameHidingSettings.getPositive())
                .build();
    }

    private static ChartWidget.NameHidingSettings toNameHidingSettingsModel(ChartWidgetOuterClass.ChartWidget.NameHidingSettings nameHidingSettings) {
        return ChartWidget.NameHidingSettings.newBuilder()
                .addAllNames(nameHidingSettings.getNamesList())
                .setPositive(nameHidingSettings.getPositive())
                .build();
    }

    private static ChartWidgetOuterClass.ChartWidget.VisualizationSettings fromVisualizationSettingsModel(ChartWidget.VisualizationSettings visualizationSettings) {
        var builder = ChartWidgetOuterClass.ChartWidget.VisualizationSettings.newBuilder()
                .setTypeValue(visualizationSettings.getTypeValue())
                .setNormalize(visualizationSettings.getNormalize())
                .setInterpolateValue(visualizationSettings.getInterpolateValue())
                .setAggregationValue(visualizationSettings.getAggregationValue())
                .setShowLabels(visualizationSettings.getShowLabels())
                .setTitle(visualizationSettings.getTitle());

        if (visualizationSettings.hasColorSchemeSettings()) {
            builder.setColorSchemeSettings(fromColorSchemeSettingsModel(visualizationSettings.getColorSchemeSettings()));
        }
        if (visualizationSettings.hasHeatmapSettings()) {
            builder.setHeatmapSettings(fromHeatmapSettingsModel(visualizationSettings.getHeatmapSettings()));
        }
        if (visualizationSettings.hasYaxisSettings()) {
            builder.setYaxisSettings(fromYaxisSettingsModel(visualizationSettings.getYaxisSettings()));
        }

        return builder.build();
    }

    private static ChartWidget.VisualizationSettings toVisualizationSettingsModel(ChartWidgetOuterClass.ChartWidget.VisualizationSettings visualizationSettings) {
        var builder = ChartWidget.VisualizationSettings.newBuilder()
                .setTypeValue(visualizationSettings.getTypeValue())
                .setNormalize(visualizationSettings.getNormalize())
                .setInterpolateValue(visualizationSettings.getInterpolateValue())
                .setAggregationValue(visualizationSettings.getAggregationValue())
                .setShowLabels(visualizationSettings.getShowLabels())
                .setTitle(visualizationSettings.getTitle());

        if (visualizationSettings.hasColorSchemeSettings()) {
            builder.setColorSchemeSettings(toColorSchemeSettingsModel(visualizationSettings.getColorSchemeSettings()));
        }
        if (visualizationSettings.hasHeatmapSettings()) {
            builder.setHeatmapSettings(toHeatmapSettingsModel(visualizationSettings.getHeatmapSettings()));
        }
        if (visualizationSettings.hasYaxisSettings()) {
            builder.setYaxisSettings(toYaxisSettingsModel(visualizationSettings.getYaxisSettings()));
        }
        return builder.build();
    }

    private static ChartWidgetOuterClass.ChartWidget.VisualizationSettings.YaxisSettings fromYaxisSettingsModel(ChartWidget.VisualizationSettings.YaxisSettings yaxisSettings) {
        var builder = ChartWidgetOuterClass.ChartWidget.VisualizationSettings.YaxisSettings.newBuilder();
        if (yaxisSettings.hasLeft()) {
            builder.setLeft(fromYaxisModel(yaxisSettings.getLeft()));
        }
        if (yaxisSettings.hasRight()) {
            builder.setRight(fromYaxisModel(yaxisSettings.getRight()));
        }
        return builder.build();
    }

    private static ChartWidget.VisualizationSettings.YaxisSettings toYaxisSettingsModel(ChartWidgetOuterClass.ChartWidget.VisualizationSettings.YaxisSettings yaxisSettings) {
        var builder = ChartWidget.VisualizationSettings.YaxisSettings.newBuilder();
        if (yaxisSettings.hasLeft()) {
            builder.setLeft(toYaxisModel(yaxisSettings.getLeft()));
        }
        if (yaxisSettings.hasRight()) {
            builder.setRight(toYaxisModel(yaxisSettings.getRight()));
        }
        return builder.build();
    }

    private static ChartWidgetOuterClass.ChartWidget.VisualizationSettings.Yaxis fromYaxisModel(ChartWidget.VisualizationSettings.Yaxis yaxis) {
        var builder = ChartWidgetOuterClass.ChartWidget.VisualizationSettings.Yaxis.newBuilder()
                .setTypeValue(yaxis.getTypeValue())
                .setTitle(yaxis.getTitle())
                .setMin(yaxis.getMin())
                .setMax(yaxis.getMax())
                .setUnitFormatValue(yaxis.getUnitFormatValue());

        if (!yaxis.getZeroPrecision()) {
            builder.setPrecision(Int64Value.newBuilder().setValue(yaxis.getPrecision()).build());
        }

        return builder.build();
    }

    private static ChartWidget.VisualizationSettings.Yaxis toYaxisModel(ChartWidgetOuterClass.ChartWidget.VisualizationSettings.Yaxis yaxis) {
        var builder = ChartWidget.VisualizationSettings.Yaxis.newBuilder()
                .setTypeValue(yaxis.getTypeValue())
                .setTitle(yaxis.getTitle())
                .setMin(yaxis.getMin())
                .setMax(yaxis.getMax())
                .setUnitFormatValue(yaxis.getUnitFormatValue());

        if (yaxis.hasPrecision()) {
            builder.setPrecision((int) yaxis.getPrecision().getValue());
        } else {
            builder.setPrecision(0);
            builder.setZeroPrecision(true);
        }
        return builder.build();
    }

    private static ChartWidgetOuterClass.ChartWidget.VisualizationSettings.HeatmapSettings fromHeatmapSettingsModel(ChartWidget.VisualizationSettings.HeatmapSettings heatmapSettings) {
        return ChartWidgetOuterClass.ChartWidget.VisualizationSettings.HeatmapSettings.newBuilder()
                .setGreenValue(heatmapSettings.getGreenValue())
                .setYellowValue(heatmapSettings.getYellowValue())
                .setRedValue(heatmapSettings.getRedValue())
                .setVioletValue(heatmapSettings.getVioletValue())
                .build();
    }

    private static ChartWidget.VisualizationSettings.HeatmapSettings toHeatmapSettingsModel(ChartWidgetOuterClass.ChartWidget.VisualizationSettings.HeatmapSettings heatmapSettings) {
        return ChartWidget.VisualizationSettings.HeatmapSettings.newBuilder()
                .setGreenValue(heatmapSettings.getGreenValue())
                .setYellowValue(heatmapSettings.getYellowValue())
                .setRedValue(heatmapSettings.getRedValue())
                .setVioletValue(heatmapSettings.getVioletValue())
                .build();
    }

    private static ChartWidgetOuterClass.ChartWidget.VisualizationSettings.ColorSchemeSettings fromColorSchemeSettingsModel(ChartWidget.VisualizationSettings.ColorSchemeSettings colorSchemeSettings) {
        var builder = ChartWidgetOuterClass.ChartWidget.VisualizationSettings.ColorSchemeSettings.newBuilder();
        switch (colorSchemeSettings.getSchemeCase()) {
            case AUTO -> builder.setAutomatic(ChartWidgetOuterClass.ChartWidget.VisualizationSettings.ColorSchemeSettings.AutomaticColorScheme.getDefaultInstance());
            case STANDARD -> builder.setStandard(ChartWidgetOuterClass.ChartWidget.VisualizationSettings.ColorSchemeSettings.StandardColorScheme.getDefaultInstance());
            case GRADIENT -> {
                var gradient = colorSchemeSettings.getGradient();
                builder.setGradient(ChartWidgetOuterClass.ChartWidget.VisualizationSettings.ColorSchemeSettings.GradientColorScheme.newBuilder()
                        .setGreenValue(gradient.getGreenValue())
                        .setYellowValue(gradient.getYellowValue())
                        .setRedValue(gradient.getRedValue())
                        .setVioletValue(gradient.getVioletValue()));
            }
        }
        return builder.build();
    }

    private static ChartWidget.VisualizationSettings.ColorSchemeSettings toColorSchemeSettingsModel(ChartWidgetOuterClass.ChartWidget.VisualizationSettings.ColorSchemeSettings colorSchemeSettings) {
        var builder = ChartWidget.VisualizationSettings.ColorSchemeSettings.newBuilder();
        switch (colorSchemeSettings.getSchemeCase()) {
            case AUTOMATIC -> builder.setAuto(ChartWidget.VisualizationSettings.ColorSchemeSettings.AutoColorScheme.getDefaultInstance());
            case STANDARD -> builder.setStandard(ChartWidget.VisualizationSettings.ColorSchemeSettings.StandardColorScheme.getDefaultInstance());
            case GRADIENT -> {
                var gradient = colorSchemeSettings.getGradient();
                builder.setGradient(ChartWidget.VisualizationSettings.ColorSchemeSettings.GradientColorScheme.newBuilder()
                        .setGreenValue(gradient.getGreenValue())
                        .setYellowValue(gradient.getYellowValue())
                        .setRedValue(gradient.getRedValue())
                        .setVioletValue(gradient.getVioletValue()));
            }
        }
        return builder.build();
    }

    private static ChartWidgetOuterClass.ChartWidget.Queries fromQueriesModel(ChartWidget.Queries queries) {
        var builder = ChartWidgetOuterClass.ChartWidget.Queries.newBuilder()
                .addAllTargets(fromTargetsModel(queries.getTargetsList()));

        if (queries.hasDownsampling()) {
            builder.setDownsampling(fromDownsamplingModel(queries.getDownsampling()));
        }

        return builder.build();
    }

    private static ChartWidget.Queries toQueriesModel(ChartWidgetOuterClass.ChartWidget.Queries queries) {
        var builder = ChartWidget.Queries.newBuilder()
                .addAllTargets(toTargetsModel(queries.getTargetsList()));

        if (queries.hasDownsampling()) {
            builder.setDownsampling(toDownsamplingModel(queries.getDownsampling()));
        }

        return builder.build();
    }

    private static DownsamplingOuterClass.Downsampling fromDownsamplingModel(Downsampling downsampling) {
        var builder = DownsamplingOuterClass.Downsampling.newBuilder();
        switch (downsampling.getModeCase()) {
            case MAX_POINTS -> builder.setMaxPoints(downsampling.getMaxPoints());
            case GRID_INTERVAL -> builder.setGridInterval(downsampling.getGridInterval());
            case DISABLED -> builder.setDisabled(downsampling.getDisabled());
        }
        builder.setGridAggregationValue(downsampling.getGridAggregationValue());
        builder.setGapFillingValue(downsampling.getGapFillingValue());
        return builder.build();
    }

    private static Downsampling toDownsamplingModel(DownsamplingOuterClass.Downsampling downsampling) {
        var builder = Downsampling.newBuilder();
        switch (downsampling.getModeCase()) {
            case MAX_POINTS -> builder.setMaxPoints(downsampling.getMaxPoints());
            case GRID_INTERVAL -> builder.setGridInterval(downsampling.getGridInterval());
            case DISABLED -> builder.setDisabled(downsampling.getDisabled());
        }
        builder.setGridAggregationValue(downsampling.getGridAggregationValue());
        builder.setGapFillingValue(downsampling.getGapFillingValue());
        return builder.build();
    }

    private static List<ChartWidgetOuterClass.ChartWidget.Queries.Target> fromTargetsModel(List<ChartWidget.Queries.Target> targetsList) {
        return targetsList.stream()
                .map(CloudDashboardDtoConverter::fromTargetModel)
                .collect(Collectors.toList());
    }

    private static List<ChartWidget.Queries.Target> toTargetsModel(List<ChartWidgetOuterClass.ChartWidget.Queries.Target> targetsList) {
        return targetsList.stream()
                .map(CloudDashboardDtoConverter::toTargetModel)
                .collect(Collectors.toList());
    }

    private static ChartWidgetOuterClass.ChartWidget.Queries.Target fromTargetModel(ChartWidget.Queries.Target target) {
        return ChartWidgetOuterClass.ChartWidget.Queries.Target.newBuilder()
                .setQuery(target.getQuery())
                .setTextMode(target.getTextMode())
                .setHidden(target.getHidden())
                .build();
    }

    private static ChartWidget.Queries.Target toTargetModel(ChartWidgetOuterClass.ChartWidget.Queries.Target target) {
        return ChartWidget.Queries.Target.newBuilder()
                .setQuery(target.getQuery())
                .setTextMode(target.getTextMode())
                .setHidden(target.getHidden())
                .build();
    }

    private static ParametrizationOuterClass.Parametrization fromParametrizationModel(Parametrization oldParametrization) {
        var parameters = oldParametrization.getParametersList().stream()
                .map(CloudDashboardDtoConverter::fromParameterModel)
                .collect(Collectors.toList());
        return ParametrizationOuterClass.Parametrization.newBuilder()
                .addAllParameters(parameters)
                .setSelectors(oldParametrization.getSelectors())
                .build();
    }

    private static Parametrization toParametrizationModel(ParametrizationOuterClass.Parametrization oldParametrization) {
        var parameters = oldParametrization.getParametersList().stream()
                .map(CloudDashboardDtoConverter::toParameterModel)
                .collect(Collectors.toList());
        return Parametrization.newBuilder()
                .addAllParameters(parameters)
                .setSelectors(oldParametrization.getSelectors())
                .build();
    }

    private static ParametrizationOuterClass.Parameter fromParameterModel(Parameter oldParameter) {
        var builder = ParametrizationOuterClass.Parameter.newBuilder()
                .setName(oldParameter.getName())
                .setHidden(oldParameter.getHidden())
                .setTitle(oldParameter.getTitle());

        switch (oldParameter.getDataCase()) {
            case LABEL_VALUES -> {
                var labelValues = oldParameter.getLabelValues();
                builder.setLabelValues(ParametrizationOuterClass.LabelValuesParameter.newBuilder()
                        .setFolderId(labelValues.getFolderId())
                        .setSelectors(labelValues.getSelectors())
                        .setLabelKey(labelValues.getLabelKey())
                        .setMultiselectable(labelValues.getMultiselectable())
                        .addAllDefaultValues(labelValues.getDefaultValuesList()));
            }
            case CUSTOM -> {
                var custom = oldParameter.getCustom();
                builder.setCustom(ParametrizationOuterClass.CustomParameter.newBuilder()
                        .addAllValues(custom.getValuesList())
                        .addAllDefaultValues(custom.getDefaultValuesList())
                        .setMultiselectable(custom.getMultiselectable())
                        .build());
            }
            case TEXT -> {
                var text = oldParameter.getText();
                builder.setText(ParametrizationOuterClass.TextParameter.newBuilder()
                        .setDefaultValue(text.getDefaultValue())
                        .build());
            }
            default -> throw new IllegalStateException("unknown parameter data type: " + oldParameter.getDataCase());
        }

        return builder.build();
    }

    private static Parameter toParameterModel(ParametrizationOuterClass.Parameter oldParameter) {
        var builder = Parameter.newBuilder()
                .setName(oldParameter.getName())
                .setHidden(oldParameter.getHidden())
                .setTitle(oldParameter.getTitle());

        switch (oldParameter.getDataCase()) {
            case LABEL_VALUES -> {
                var labelValues = oldParameter.getLabelValues();
                builder.setLabelValues(LabelValuesParameter.newBuilder()
                        .setFolderId(labelValues.getFolderId())
                        .setSelectors(labelValues.getSelectors())
                        .setLabelKey(labelValues.getLabelKey())
                        .setMultiselectable(labelValues.getMultiselectable())
                        .addAllDefaultValues(labelValues.getDefaultValuesList()));
            }
            case CUSTOM -> {
                var custom = oldParameter.getCustom();
                builder.setCustom(CustomParameter.newBuilder()
                        .addAllValues(custom.getValuesList())
                        .addAllDefaultValues(custom.getDefaultValuesList())
                        .setMultiselectable(custom.getMultiselectable())
                        .build());
            }
            case TEXT -> {
                var text = oldParameter.getText();
                builder.setText(TextParameter.newBuilder()
                        .setDefaultValue(text.getDefaultValue())
                        .build());
            }
            default -> throw new IllegalStateException("unknown parameter data type: " + oldParameter.getDataCase());
        }

        return builder.build();
    }

    public DashboardOuterClass.Dashboard convertFromIntranet(ru.yandex.monitoring.api.v3.Dashboard request) {
        return safe(() -> DashboardOuterClass.Dashboard.parseFrom(request.toByteArray()));
    }

    public ru.yandex.monitoring.api.v3.Dashboard convertToIntranet(DashboardOuterClass.Dashboard request) {
        return safe(() -> ru.yandex.monitoring.api.v3.Dashboard.parseFrom(request.toByteArray()));
    }

    public GetDashboardRequest convertToIntranet(DashboardServiceOuterClass.GetDashboardRequest request) {
        return safe(() -> GetDashboardRequest.parseFrom(request.toByteArray()));
    }

    public DashboardServiceOuterClass.GetDashboardRequest convertFromIntranet(GetDashboardRequest request) {
        return safe(() -> DashboardServiceOuterClass.GetDashboardRequest.parseFrom(request.toByteArray()));
    }

    public ListDashboardsRequest convertToIntranet(DashboardServiceOuterClass.ListDashboardsRequest request) {
        return safe(() -> ListDashboardsRequest.parseFrom(request.toByteArray()));
    }

    public DashboardServiceOuterClass.ListDashboardsRequest convertFromIntranet(ListDashboardsRequest request) {
        return safe(() -> DashboardServiceOuterClass.ListDashboardsRequest.parseFrom(request.toByteArray()));
    }

    public DashboardServiceOuterClass.ListDashboardsResponse convertFromIntranet(ListDashboardsResponse request) {
        return safe(() -> DashboardServiceOuterClass.ListDashboardsResponse.parseFrom(request.toByteArray()));
    }

    public ListDashboardsResponse convertToIntranet(DashboardServiceOuterClass.ListDashboardsResponse request) {
        return safe(() -> ListDashboardsResponse.parseFrom(request.toByteArray()));
    }

    public CreateDashboardRequest convertToIntranet(DashboardServiceOuterClass.CreateDashboardRequest request) {
        return safe(() -> CreateDashboardRequest.parseFrom(request.toByteArray()));
    }

    public DashboardServiceOuterClass.CreateDashboardRequest convertFromIntranet(CreateDashboardRequest request) {
        return safe(() -> DashboardServiceOuterClass.CreateDashboardRequest.parseFrom(request.toByteArray()));
    }

    public UpdateDashboardRequest convertToIntranet(DashboardServiceOuterClass.UpdateDashboardRequest request) {
        return safe(() -> UpdateDashboardRequest.parseFrom(request.toByteArray()));
    }

    public DashboardServiceOuterClass.UpdateDashboardRequest convertFromIntranet(UpdateDashboardRequest request) {
        return safe(() -> DashboardServiceOuterClass.UpdateDashboardRequest.parseFrom(request.toByteArray()));
    }

    public DeleteDashboardRequest convertToIntranet(DashboardServiceOuterClass.DeleteDashboardRequest request) {
        return safe(() -> DeleteDashboardRequest.parseFrom(request.toByteArray()));
    }

    public DashboardServiceOuterClass.DeleteDashboardRequest convertFromIntranet(DeleteDashboardRequest request) {
        return safe(() -> DashboardServiceOuterClass.DeleteDashboardRequest.parseFrom(request.toByteArray()));
    }

    private <T> T safe(Callable<T> supplier) {
        try {
            return supplier.call();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
}
