package ru.yandex.solomon.gateway.api.v3.cloud.grpc;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import io.grpc.stub.StreamObserver;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import yandex.cloud.api.monitoring.v3.DashboardOuterClass;
import yandex.cloud.api.monitoring.v3.DashboardServiceGrpc;
import yandex.cloud.api.monitoring.v3.DashboardServiceOuterClass;
import yandex.cloud.api.operation.OperationOuterClass;

import ru.yandex.grpc.utils.PublicGrpcService;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.gateway.api.v3.cloud.dto.CloudDashboardDtoConverter;
import ru.yandex.solomon.gateway.api.v3.cloud.dto.CloudOperationDtoConverter;
import ru.yandex.solomon.gateway.api.v3.intranet.DashboardService;
import ru.yandex.solomon.grpc.handler.GrpcMethodHandler;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class GrpcCloudDashboardService extends DashboardServiceGrpc.DashboardServiceImplBase implements PublicGrpcService {
    private final DashboardService service;
    private final CloudDashboardDtoConverter converter;
    private final CloudOperationDtoConverter operationConverter;

    @Autowired
    public GrpcCloudDashboardService(DashboardService service) {
        this.service = service;
        converter = new CloudDashboardDtoConverter();
        operationConverter = new CloudOperationDtoConverter();
    }

    @Override
    public void get(
            DashboardServiceOuterClass.GetDashboardRequest request,
            StreamObserver<DashboardOuterClass.Dashboard> responseObserver)
    {
        GrpcMethodHandler.handle(this::doGet, request, responseObserver);
    }

    private CompletableFuture<DashboardOuterClass.Dashboard> doGet(
            DashboardServiceOuterClass.GetDashboardRequest request,
            AuthSubject subject)
    {
        return CompletableFutures.safeCall(() -> service.get(converter.convertToIntranet(request), subject))
                .thenApply(converter::convertFromIntranet);
    }

    @Override
    public void list(DashboardServiceOuterClass.ListDashboardsRequest request,
                     StreamObserver<DashboardServiceOuterClass.ListDashboardsResponse> responseObserver)
    {
        GrpcMethodHandler.handle(this::doList, request, responseObserver);
    }

    private CompletableFuture<DashboardServiceOuterClass.ListDashboardsResponse> doList(
            DashboardServiceOuterClass.ListDashboardsRequest request,
            AuthSubject subject)
    {
        return CompletableFutures.safeCall(() -> service.list(converter.convertToIntranet(request), subject))
                .thenApply(converter::convertFromIntranet);
    }

    @Override
    public void create(DashboardServiceOuterClass.CreateDashboardRequest request,
                       StreamObserver<OperationOuterClass.Operation> responseObserver)
    {
        GrpcMethodHandler.handle(this::doCreate, request, responseObserver, operationConverter::wrapCloud);
    }

    private CompletableFuture<DashboardOuterClass.Dashboard> doCreate(
            DashboardServiceOuterClass.CreateDashboardRequest request,
            AuthSubject subject)
    {
        return CompletableFutures.safeCall(() -> service.create(converter.convertToIntranet(request), subject))
                .thenApply(converter::convertFromIntranet);
    }

    @Override
    public void update(DashboardServiceOuterClass.UpdateDashboardRequest request,
                       StreamObserver<OperationOuterClass.Operation> responseObserver)
    {
        GrpcMethodHandler.handle(this::doUpdate, request, responseObserver, operationConverter::wrapCloud);
    }

    private CompletableFuture<DashboardOuterClass.Dashboard> doUpdate(
            DashboardServiceOuterClass.UpdateDashboardRequest request,
            AuthSubject subject)
    {
        return CompletableFutures.safeCall(() -> service.update(converter.convertToIntranet(request), subject))
                .thenApply(converter::convertFromIntranet);
    }

    @Override
    public void delete(DashboardServiceOuterClass.DeleteDashboardRequest request,
                       StreamObserver<OperationOuterClass.Operation> responseObserver)
    {
        GrpcMethodHandler.handle(this::doDelete, request, responseObserver, operationConverter::wrapCloud);
    }

    private CompletableFuture<Empty> doDelete(
            DashboardServiceOuterClass.DeleteDashboardRequest request,
            AuthSubject subject)
    {
        return CompletableFutures.safeCall(() -> service.delete(converter.convertToIntranet(request), subject));
    }

    @Override
    public void listOperations(DashboardServiceOuterClass.ListDashboardOperationsRequest request,
                               StreamObserver<DashboardServiceOuterClass.ListDashboardOperationsResponse> responseObserver)
    {
        GrpcMethodHandler.SimpleHandlerFunc<DashboardServiceOuterClass.ListDashboardOperationsRequest, DashboardServiceOuterClass.ListDashboardOperationsResponse> returnEmpty =
                (request1, subject) -> DashboardServiceOuterClass.ListDashboardOperationsResponse.getDefaultInstance();
        GrpcMethodHandler.handle(returnEmpty, request, responseObserver);
    }
}
