package ru.yandex.solomon.gateway.api.v3.cloud.priv.dto;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import com.google.protobuf.util.Timestamps;
import yandex.cloud.priv.monitoring.v3.ServiceProviderOuterClass.ServiceProvider;
import yandex.cloud.priv.monitoring.v3.ServiceProviderOuterClass.ServiceProvider.Reference;
import yandex.cloud.priv.monitoring.v3.ServiceProviderOuterClass.ServiceProvider.ShardSettings;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.core.db.model.ReferenceConf;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.ServiceProviderShardSettings;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ServiceProviderDtoConverter {
    public static ru.yandex.solomon.core.db.model.ServiceProvider toModel(ServiceProvider serviceProvider, int version) {
        var iamServiceAccountIds = StringUtils.isEmpty(serviceProvider.getIamServiceAccountId())
                ? serviceProvider.getIamServiceAccountIdsList()
                : List.of(serviceProvider.getIamServiceAccountId());
        return ru.yandex.solomon.core.db.model.ServiceProvider.newBuilder()
                .setId(serviceProvider.getId())
                .setDescription(serviceProvider.getDescription())
                .setShardSettings(toModel(serviceProvider.getShardSettings()))
                .setReferences(toModel(serviceProvider.getReferencesList()))
                .setCloudId(serviceProvider.getCloudId())
                .setIamServiceAccountIds(iamServiceAccountIds)
                .setCreatedAtMillis(Timestamps.toMillis(serviceProvider.getCreatedAt()))
                .setUpdatedAtMillis(Timestamps.toMillis(serviceProvider.getModifiedAt()))
                .setCreatedBy(serviceProvider.getCreatedBy())
                .setUpdatedBy(serviceProvider.getModifiedBy())
                .setVersion(version)
                .build();
    }

    public static ServiceProvider fromModel(ru.yandex.solomon.core.db.model.ServiceProvider serviceProvider) {
        return ServiceProvider.newBuilder()
                .setId(serviceProvider.getId())
                .setDescription(serviceProvider.getDescription())
                .setShardSettings(fromModel(serviceProvider.getShardSettings()))
                .addAllReferences(fromModel(serviceProvider.getReferences()))
                .setCloudId(serviceProvider.getCloudId())
                .setIamServiceAccountId(Iterables.getFirst(serviceProvider.getIamServiceAccountIds(), ""))
                .addAllIamServiceAccountIds(serviceProvider.getIamServiceAccountIds())
                .setCreatedAt(Timestamps.fromMillis(serviceProvider.getCreatedAtMillis()))
                .setModifiedAt(Timestamps.fromMillis(serviceProvider.getUpdatedAtMillis()))
                .setCreatedBy(serviceProvider.getCreatedBy())
                .setModifiedBy(serviceProvider.getUpdatedBy())
                .build();
    }

    private static ServiceProviderShardSettings toModel(ShardSettings shardSettings) {
        ServiceMetricConf.AggrRule[] aggrRules = shardSettings.getAggrRulesList().stream()
                .map(ServiceProviderDtoConverter::toModel)
                .toArray(ServiceMetricConf.AggrRule[]::new);

        ServiceMetricConf metricConf = ServiceMetricConf.of(aggrRules, shardSettings.getMemOnly());

        return new ServiceProviderShardSettings(
                metricConf,
                (int) shardSettings.getMetricsTtlDays(),
                (int) shardSettings.getGridSeconds(),
                0); // TODO: when API will be finalized add this field
    }

    private static ServiceMetricConf.AggrRule toModel(ShardSettings.AggrRule aggrRule) {
        return new ServiceMetricConf.AggrRule(aggrRule.getConditionsList().toArray(String[]::new),
                aggrRule.getTargetsList().toArray(String[]::new), null);
    }

    private static ShardSettings fromModel(ServiceProviderShardSettings shardSettings) {
        List<ShardSettings.AggrRule> aggrRules = Arrays.stream(shardSettings.getMetricConf().getAggrRules())
                .map(aggrRule -> ShardSettings.AggrRule.newBuilder()
                        .addAllConditions(Arrays.asList(aggrRule.getCond()))
                        .addAllTargets(Arrays.asList(aggrRule.getTarget()))
                        .build())
                .collect(Collectors.toList());

        return ShardSettings.newBuilder()
                .addAllAggrRules(aggrRules)
                .setMemOnly(shardSettings.getMetricConf().isRawDataMemOnly())
                .setMetricsTtlDays(shardSettings.getMetricsTtlDays())
                .setGridSeconds(shardSettings.getGrid())
                .build();
    }

    private static List<ReferenceConf> toModel(List<Reference> references) {
        return references.stream()
                .map(ServiceProviderDtoConverter::toModel)
                .collect(Collectors.toList());
    }

    private static ReferenceConf toModel(Reference reference) {
        return new ReferenceConf(
                reference.getLabel(),
                reference.getServicesList(),
                reference.getTypesList(),
                reference.getCrossFolder()
        );
    }

    private static List<Reference> fromModel(List<ReferenceConf> references) {
        return references.stream()
                .map(ServiceProviderDtoConverter::fromModel)
                .collect(Collectors.toList());
    }

    private static Reference fromModel(ReferenceConf reference) {
        return Reference.newBuilder()
                .setLabel(reference.label)
                .addAllServices(reference.services)
                .addAllTypes(reference.types)
                .setCrossFolder(reference.crossFolder)
                .build();
    }
}
