package ru.yandex.solomon.gateway.api.v3.cloud.priv.grpc;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.stub.StreamObserver;
import org.springframework.stereotype.Component;
import yandex.cloud.priv.operation.PO;

import ru.yandex.cloud.priv.monitoring.v3.AlertTemplate;
import ru.yandex.cloud.priv.monitoring.v3.AlertTemplateServiceGrpc;
import ru.yandex.cloud.priv.monitoring.v3.CreateAlertTemplateRequest;
import ru.yandex.cloud.priv.monitoring.v3.DeployAlertTemplateRequest;
import ru.yandex.cloud.priv.monitoring.v3.ListAlertTemplateRequest;
import ru.yandex.cloud.priv.monitoring.v3.ListAlertTemplateResponse;
import ru.yandex.cloud.priv.monitoring.v3.ListAlertTemplateVersionsRequest;
import ru.yandex.cloud.priv.monitoring.v3.ListAlertTemplateVersionsResponse;
import ru.yandex.cloud.priv.monitoring.v3.PublishAlertTemplateRequest;
import ru.yandex.cloud.priv.monitoring.v3.ReadAlertTemplateRequest;
import ru.yandex.cloud.priv.monitoring.v3.UnpublishAlertTemplateRequest;
import ru.yandex.grpc.utils.GrpcService;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.gateway.api.v3.cloud.priv.dto.CloudAlertTemplateDtoConverter;
import ru.yandex.solomon.gateway.api.v3.cloud.priv.dto.CloudOperationDtoConverter;
import ru.yandex.solomon.gateway.api.v3.intranet.AlertTemplateService;
import ru.yandex.solomon.grpc.handler.GrpcMethodHandler;

/**
 * @author Alexey Trushkin
 */
@Component
@ParametersAreNonnullByDefault
public class GrpcCloudAlertTemplateService extends AlertTemplateServiceGrpc.AlertTemplateServiceImplBase implements GrpcService {

    private final CloudOperationDtoConverter operationConverter;
    private final AlertTemplateService service;
    private final CloudAlertTemplateDtoConverter cloudAlertTemplateDtoConverter;

    public GrpcCloudAlertTemplateService(AlertTemplateService alertTemplateService) {
        this.service = alertTemplateService;
        operationConverter = new CloudOperationDtoConverter();
        cloudAlertTemplateDtoConverter = new CloudAlertTemplateDtoConverter();
    }

    @Override
    public void create(CreateAlertTemplateRequest requestCloud, StreamObserver<PO.Operation> responseObserver) {
        GrpcMethodHandler.handle(this::doCreate, requestCloud, responseObserver);
    }

    private CompletableFuture<PO.Operation> doCreate(CreateAlertTemplateRequest request, AuthSubject subject) {
        return CompletableFutures.safeCall(() -> service.create(cloudAlertTemplateDtoConverter.convertToInternal(request), subject))
                .thenApply(cloudAlertTemplateDtoConverter::convertToCloud)
                .handle(operationConverter::wrapCloud);
    }

    @Override
    public void read(ReadAlertTemplateRequest request, StreamObserver<AlertTemplate> responseObserver) {
        GrpcMethodHandler.handle(this::doRead, request, responseObserver);
    }

    private CompletableFuture<AlertTemplate> doRead(ReadAlertTemplateRequest request, AuthSubject subject) {
        return CompletableFutures.safeCall(() -> service.read(cloudAlertTemplateDtoConverter.convertToInternal(request), subject))
                .thenApply(cloudAlertTemplateDtoConverter::convertToCloud);
    }

    @Override
    public void publish(PublishAlertTemplateRequest request, StreamObserver<PO.Operation> responseObserver) {
        GrpcMethodHandler.handle(this::doPublish, request, responseObserver);
    }

    private CompletableFuture<PO.Operation> doPublish(PublishAlertTemplateRequest request, AuthSubject subject) {
        return CompletableFutures.safeCall(() -> service.publish(cloudAlertTemplateDtoConverter.convertToInternal(request), subject))
                .thenApply(cloudAlertTemplateDtoConverter::convertToCloud)
                .handle(operationConverter::wrapCloud);
    }

    @Override
    public void unpublish(UnpublishAlertTemplateRequest request, StreamObserver<PO.Operation> responseObserver) {
        GrpcMethodHandler.handle(this::doUnpublish, request, responseObserver);
    }

    private CompletableFuture<PO.Operation> doUnpublish(UnpublishAlertTemplateRequest request, AuthSubject subject) {
        return CompletableFutures.safeCall(() -> service.unpublish(cloudAlertTemplateDtoConverter.convertToInternal(request), subject))
                .handle(operationConverter::wrapCloud);
    }

    @Override
    public void deploy(DeployAlertTemplateRequest request, StreamObserver<PO.Operation> responseObserver) {
        GrpcMethodHandler.handle(this::doDeploy, request, responseObserver);
    }

    private CompletableFuture<PO.Operation> doDeploy(DeployAlertTemplateRequest request, AuthSubject subject) {
        return CompletableFutures.safeCall(() -> service.deploy(cloudAlertTemplateDtoConverter.convertToInternal(request), subject))
                .handle(operationConverter::wrapCloud)
                .thenApply(operation -> operation.toBuilder()
                        .setDone(false)
                        .setDescription("Deploy alert template " + request.getTemplateId() + ", version " + request.getTemplateVersionTag())
                        .build());
    }

    @Override
    public void listVersions(ListAlertTemplateVersionsRequest request, StreamObserver<ListAlertTemplateVersionsResponse> responseObserver) {
        GrpcMethodHandler.handle(this::doListAlertTemplateVersions, request, responseObserver);
    }

    private CompletableFuture<ListAlertTemplateVersionsResponse> doListAlertTemplateVersions(ListAlertTemplateVersionsRequest request, AuthSubject subject) {
        return CompletableFutures.safeCall(() -> service.listTemplateVersions(cloudAlertTemplateDtoConverter.convertToInternal(request), subject))
                .thenApply(cloudAlertTemplateDtoConverter::convertToCloud);
    }

    @Override
    public void list(ListAlertTemplateRequest request, StreamObserver<ListAlertTemplateResponse> responseObserver) {
        GrpcMethodHandler.handle(this::doListAlertTemplate, request, responseObserver);
    }

    private CompletableFuture<ListAlertTemplateResponse> doListAlertTemplate(ListAlertTemplateRequest request, AuthSubject subject) {
        return CompletableFutures.safeCall(() -> service.listTemplates(cloudAlertTemplateDtoConverter.convertToInternal(request), subject))
                .thenApply(cloudAlertTemplateDtoConverter::convertToCloud);
    }
}
