package ru.yandex.solomon.gateway.api.v3.cloud.priv.impl;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Any;
import com.google.protobuf.Empty;
import com.google.protobuf.Timestamp;
import com.google.protobuf.util.Timestamps;
import org.apache.commons.lang3.tuple.Pair;
import yandex.cloud.priv.monitoring.v3.ServiceProviderOuterClass;
import yandex.cloud.priv.monitoring.v3.ServiceProviderServiceOuterClass;
import yandex.cloud.priv.operation.PO;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.gateway.api.utils.IdGenerator;
import ru.yandex.solomon.gateway.api.v3.cloud.priv.ServiceProviderService;
import ru.yandex.solomon.gateway.api.v3.cloud.priv.dto.ServiceProviderDtoConverter;
import ru.yandex.solomon.gateway.api.v3.cloud.priv.validators.ServiceProviderValidator;
import ru.yandex.solomon.ydb.page.TokenBasePage;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ServiceProviderServiceImpl implements ServiceProviderService {
    private final InternalAuthorizer authorizer;
    private final ServiceProvidersDao serviceProvidersDao;
    private final String entityIdPrefix;


    public ServiceProviderServiceImpl(
            InternalAuthorizer authorizer,
            ServiceProvidersDao serviceProvidersDao,
            String entityIdPrefix)
    {
        this.entityIdPrefix = entityIdPrefix;
        this.authorizer = authorizer;
        this.serviceProvidersDao = serviceProvidersDao;
    }

    @Override
    public CompletableFuture<Pair<ServiceProviderOuterClass.ServiceProvider, Integer>> get(ServiceProviderServiceOuterClass.GetServiceProviderRequest request, AuthSubject subject) {
        ServiceProviderValidator.validate(request);

        return serviceProvidersDao.read(request.getServiceProviderId()).thenApply(serviceProviderOpt -> {
            if (serviceProviderOpt.isEmpty()) {
                throw serviceProviderNotFound(request.getServiceProviderId());
            }
            var serviceProvider = serviceProviderOpt.get();

            return Pair.of(ServiceProviderDtoConverter.fromModel(serviceProvider), serviceProvider.getVersion());
        });
    }

    @Override
    public CompletableFuture<ServiceProviderServiceOuterClass.ListServiceProvidersResponse> list(ServiceProviderServiceOuterClass.ListServiceProvidersRequest request, AuthSubject subject) {
        ServiceProviderValidator.validate(request);

        return serviceProvidersDao.list(request.getFilter(), (int) request.getPageSize(), request.getPageToken())
                .thenApply(page -> {
                    TokenBasePage<ServiceProviderOuterClass.ServiceProvider> mappedPage = page.map(ServiceProviderDtoConverter::fromModel);
                    return ServiceProviderServiceOuterClass.ListServiceProvidersResponse.newBuilder()
                            .addAllServiceProviders(mappedPage.getItems())
                            .setNextPageToken(mappedPage.getNextPageToken())
                            .build();
                });
    }

    @Override
    public CompletableFuture<PO.Operation> create(ServiceProviderServiceOuterClass.CreateServiceProviderRequest request, AuthSubject subject) {
        Timestamp now = Timestamps.fromMillis(System.currentTimeMillis());

        ServiceProviderValidator.validate(request);

        String login = subject.getUniqueId();
        String operationId = IdGenerator.generateId(entityIdPrefix);

        ServiceProviderOuterClass.ServiceProvider serviceProvider = ServiceProviderOuterClass.ServiceProvider.newBuilder()
                .setId(request.getServiceProviderId())
                .setDescription(request.getDescription())
                .setShardSettings(request.getShardSettings())
                .addAllReferences(request.getReferencesList())
                .setCloudId(request.getCloudId())
                .setIamServiceAccountId(request.getIamServiceAccountId())
                .addAllIamServiceAccountIds(request.getIamServiceAccountIdsList())
                .setCreatedAt(now)
                .setModifiedAt(now)
                .setCreatedBy(login)
                .setModifiedBy(login)
                .build();

        return authorizer.authorize(subject)
                .thenCompose(account -> serviceProvidersDao.insert(ServiceProviderDtoConverter.toModel(serviceProvider, 0)))
                .thenApply(inserted -> {
                    if (!inserted) {
                        throw new ConflictException("service provider " + request.getServiceProviderId() + " already exists");
                    }
                    var metadata = ServiceProviderServiceOuterClass.CreateServiceProviderMetadata.newBuilder()
                            .setServiceProviderId(serviceProvider.getId())
                            .build();

                    return PO.Operation.newBuilder()
                            .setId(operationId)
                            .setCreatedAt(now)
                            .setModifiedAt(now)
                            .setDone(true)
                            .setMetadata(Any.pack(metadata))
                            .setResponse(Any.pack(serviceProvider))
                            .build();
                });
    }

    @Override
    public CompletableFuture<PO.Operation> update(ServiceProviderServiceOuterClass.UpdateServiceProviderRequest request, AuthSubject subject, int etag) {
        Timestamp now = Timestamps.fromMillis(System.currentTimeMillis());

        ServiceProviderValidator.validate(request);

        String operationId = IdGenerator.generateId(entityIdPrefix);

        ServiceProviderOuterClass.ServiceProvider serviceProvider = ServiceProviderOuterClass.ServiceProvider.newBuilder()
                .setId(request.getServiceProviderId())
                .setDescription(request.getDescription())
                .setShardSettings(request.getShardSettings())
                .addAllReferences(request.getReferencesList())
                .setIamServiceAccountId(request.getIamServiceAccountId())
                .addAllIamServiceAccountIds(request.getIamServiceAccountIdsList())
                .setModifiedAt(now)
                .setModifiedBy(subject.getUniqueId())
                .build();

        return authorizer.authorize(subject)
                .thenCompose(account -> serviceProvidersDao.update(ServiceProviderDtoConverter.toModel(serviceProvider, etag)))
                .thenCompose(updatedOpt -> {
                    if (updatedOpt.isPresent()) {
                        ServiceProviderOuterClass.ServiceProvider updatedServiceProvider =
                                ServiceProviderDtoConverter.fromModel(updatedOpt.get());

                        var metadata = ServiceProviderServiceOuterClass.UpdateServiceProviderMetadata.newBuilder()
                                .setServiceProviderId(serviceProvider.getId())
                                .build();

                        var operation = PO.Operation.newBuilder()
                                .setId(operationId)
                                .setCreatedAt(now)
                                .setModifiedAt(now)
                                .setDone(true)
                                .setMetadata(Any.pack(metadata))
                                .setResponse(Any.pack(updatedServiceProvider))
                                .build();

                        return CompletableFuture.completedFuture(operation);
                    }

                    return serviceProvidersDao.exists(serviceProvider.getId())
                            .thenApply(exists -> {
                                if (exists) {
                                    String message = String.format(
                                            "service provider \"%s\" with version %s is out of date",
                                            serviceProvider.getId(),
                                            etag
                                    );
                                    throw new ConflictException(message);
                                }
                                throw serviceProviderNotFound(serviceProvider.getId());
                            });
                });
    }

    @Override
    public CompletableFuture<PO.Operation> delete(ServiceProviderServiceOuterClass.DeleteServiceProviderRequest request, AuthSubject subject) {
        Timestamp now = Timestamps.fromMillis(System.currentTimeMillis());
        String operationId = IdGenerator.generateId(entityIdPrefix);

        ServiceProviderValidator.validate(request);

        return authorizer.authorize(subject)
                .thenCompose(account -> serviceProvidersDao.delete(request.getServiceProviderId()))
                .thenApply(deleted -> {
                    if (!deleted) {
                        throw serviceProviderNotFound(request.getServiceProviderId());
                    }
                    var metadata = ServiceProviderServiceOuterClass.DeleteServiceProviderMetadata.newBuilder()
                            .setServiceProviderId(request.getServiceProviderId())
                            .build();

                    return PO.Operation.newBuilder()
                            .setId(operationId)
                            .setCreatedAt(now)
                            .setModifiedAt(now)
                            .setDone(true)
                            .setMetadata(Any.pack(metadata))
                            .setResponse(Any.pack(Empty.getDefaultInstance()))
                            .build();
                });
    }

    private static NotFoundException serviceProviderNotFound(String id) {
        return new NotFoundException("no service provider with id " + id + " found");
    }
}
