package ru.yandex.solomon.gateway.api.v3.cloud.priv.rest;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import yandex.cloud.priv.monitoring.v3.ServiceProviderServiceOuterClass;

import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.gateway.api.utils.grpc.EtagInterceptor;
import ru.yandex.solomon.gateway.api.v3.cloud.priv.ServiceProviderService;
import ru.yandex.solomon.gateway.api.v3.utils.ProtoJsonUtils;

/**
 * @author Oleg Baryshnikov
 */
@RestController
@RequestMapping(produces = MediaType.APPLICATION_JSON_VALUE)
@ParametersAreNonnullByDefault
public class CloudServiceProviderController {
    private final ServiceProviderService service;

    @Autowired
    public CloudServiceProviderController(ServiceProviderService service) {
        this.service = service;
    }

    @RequestMapping(path = "/monitoring/v3/serviceProviders", method = RequestMethod.GET)
    CompletableFuture<String> list(
            @RequireAuth AuthSubject subject,
            @RequestParam(value = "filter", defaultValue = "") String filter,
            @RequestParam(value = "pageSize", defaultValue = "100") int pageSize,
            @RequestParam(value = "pageToken", defaultValue = "") String pageToken,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        var listRequest =
                ServiceProviderServiceOuterClass.ListServiceProvidersRequest.newBuilder()
                .setFilter(filter)
                .setPageSize(pageSize)
                .setPageToken(pageToken)
                .build();

        return service.list(listRequest, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/monitoring/v3/serviceProviders", method = RequestMethod.POST)
    CompletableFuture<String> create(
            @RequireAuth AuthSubject subject,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body)
    {
        var builder = ServiceProviderServiceOuterClass.CreateServiceProviderRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        var createRequest = builder.build();
        return service.create(createRequest, subject)
                .thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/monitoring/v3/serviceProviders/{serviceProviderId}", method = RequestMethod.GET)
    CompletableFuture<ResponseEntity<String>> get(
            @RequireAuth AuthSubject subject,
            @PathVariable("serviceProviderId") String serviceProviderId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        var getRequest = ServiceProviderServiceOuterClass.GetServiceProviderRequest.newBuilder()
                .setServiceProviderId(serviceProviderId)
                .build();
        return service.get(getRequest, subject)
                .thenApply(response -> {
                    var dto = response.getLeft();
                    var etag = response.getRight();
                    String json = ProtoJsonUtils.toJson(dto, pretty);
                    return ResponseEntity.ok()
                            .eTag(Integer.toString(etag))
                            .body(json);
                });
    }

    @RequestMapping(path = "/monitoring/v3/serviceProviders/{serviceProviderId}", method = RequestMethod.PUT)
    CompletableFuture<String> update(
            @RequireAuth AuthSubject subject,
            @PathVariable("serviceProviderId") String serviceProviderId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty,
            @RequestBody byte[] body,
            ServerHttpRequest request)
    {
        int etag = EtagInterceptor.parseEtagHeader(request);
        var builder = ServiceProviderServiceOuterClass.UpdateServiceProviderRequest.newBuilder();
        ProtoJsonUtils.fromJson(body, builder);
        builder.setServiceProviderId(serviceProviderId);
        var updateRequest = builder.build();
        return service.update(updateRequest, subject, etag).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }

    @RequestMapping(path = "/monitoring/v3/serviceProviders/{serviceProviderId}", method = RequestMethod.DELETE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    CompletableFuture<String> delete(
            @RequireAuth AuthSubject subject,
            @PathVariable("serviceProviderId") String serviceProviderId,
            @RequestParam(value = "_pretty", defaultValue = "false") boolean pretty)
    {
        var deleteRequest = ServiceProviderServiceOuterClass.DeleteServiceProviderRequest.newBuilder()
                .setServiceProviderId(serviceProviderId)
                .build();

        return service.delete(deleteRequest, subject).thenApply(response -> ProtoJsonUtils.toJson(response, pretty));
    }
}
