package ru.yandex.solomon.gateway.api.v3.cloud.priv.validators;

import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import yandex.cloud.priv.monitoring.v3.ServiceProviderOuterClass.ServiceProvider.Reference;
import yandex.cloud.priv.monitoring.v3.ServiceProviderOuterClass.ServiceProvider.ShardSettings;
import yandex.cloud.priv.monitoring.v3.ServiceProviderServiceOuterClass.CreateServiceProviderRequest;
import yandex.cloud.priv.monitoring.v3.ServiceProviderServiceOuterClass.DeleteServiceProviderRequest;
import yandex.cloud.priv.monitoring.v3.ServiceProviderServiceOuterClass.GetServiceProviderRequest;
import yandex.cloud.priv.monitoring.v3.ServiceProviderServiceOuterClass.ListServiceProvidersRequest;
import yandex.cloud.priv.monitoring.v3.ServiceProviderServiceOuterClass.UpdateServiceProviderRequest;

import ru.yandex.monlib.metrics.labels.validate.LabelsValidator;
import ru.yandex.solomon.core.conf.aggr.LabelValueExpr;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.validators.IdValidator;
import ru.yandex.solomon.labels.LabelValidator;
import ru.yandex.solomon.util.time.InstantUtils;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ServiceProviderValidator {
    private static final int MAX_AGGR_RULES_COUNT = 10;

    public static void validate(ListServiceProvidersRequest request) {
    }

    public static void validate(GetServiceProviderRequest request) {
        validateId(request.getServiceProviderId());
    }

    public static void validate(DeleteServiceProviderRequest request) {
        validateId(request.getServiceProviderId());
    }

    public static void validate(CreateServiceProviderRequest request) {
        validateId(request.getServiceProviderId());
        IdValidator.ensureValid(request.getServiceProviderId(), "service provider");
        if (StringUtils.isBlank(request.getCloudId())) {
            throw new BadRequestException("cloud ID must be not blank in cloud service provider");
        }
        validateShardSettings(request.getShardSettings());
        validateReferences(request.getReferencesList());
    }

    public static void validate(UpdateServiceProviderRequest request) {
        validateId(request.getServiceProviderId());
        validateShardSettings(request.getShardSettings());
        validateReferences(request.getReferencesList());
    }

    private static void validateId(String serviceProviderId) {
        if (StringUtils.isBlank(serviceProviderId)) {
            throw new BadRequestException("service provider ID cannot be blank");
        }
    }

    private static void validateShardSettings(ShardSettings shardSettings) {
        validateAggrRules(shardSettings.getAggrRulesList());
        validateGrid(shardSettings.getGridSeconds());
    }

    private static void validateAggrRules(List<ShardSettings.AggrRule> aggrRules) {
        if (aggrRules.size() > MAX_AGGR_RULES_COUNT) {
            throw new BadRequestException("too many aggregation rules, max count is " + MAX_AGGR_RULES_COUNT);
        }

        for (ShardSettings.AggrRule rule : aggrRules) {
            List<String> cond = rule.getConditionsList();
            List<String> target = rule.getTargetsList();
            if (cond.isEmpty() && target.isEmpty()) {
                continue;
            }
            checkNonBlankStrings(cond, "conditions");
            checkNonBlankStrings(target, "targets");
            for (String c : cond) {
                checkLabel(c, "condition in aggregation rule");
            }
            for (String t : target) {
                checkLabel(t, "target in aggregation rule");
            }
        }
    }

    private static void checkNonBlankStrings(@Nonnull List<String> strs, String fieldName) {
        if (strs.isEmpty()) {
            throw new BadRequestException(fieldName + " in aggregation rule cannot be an empty");
        }
        for (String str : strs) {
            if (StringUtils.isBlank(str)) {
                throw new BadRequestException(fieldName + " in aggregation rule cannot have blank element");
            }
        }
    }

    private static void checkLabel(@Nonnull String labelStr, String message) {
        String[] nameValue = StringUtils.split(labelStr, '=');
        if (nameValue.length != 2) {
            throw new BadRequestException(message + " must be in 'name=value' format, but got: '" + labelStr + '\'');
        }
        if (!LabelValidator.isValidNameChars(nameValue[0])) {
            throw new BadRequestException("invalid label name for " + message + ", got: '" + nameValue[0] + '\'');
        }
        if (!LabelValueExpr.isValid(nameValue[1])) {
            throw new BadRequestException("invalid label value expression for " + message + ", got: '" + nameValue[1] + '\'');
        }
    }

    private static void validateGrid(long gridSeconds) {
        if (gridSeconds != Service.GRID_ABSENT && gridSeconds != Service.GRID_UNKNOWN) {
            if (gridSeconds < -1) {
                throw new BadRequestException("grid seconds can't be less than -1");
            }
            long decimGridSec = TimeUnit.MINUTES.toSeconds(5);
            if (!InstantUtils.checkGridMultiplicity(decimGridSec, gridSeconds)) {
                throw new BadRequestException("grid seconds (" + gridSeconds + ") must be multiple of 5m");
            }
        }
    }

    private static void validateReferences(List<Reference> references) {
        for (Reference reference : references) {
            validateReference(reference);
        }
    }

    private static void validateReference(Reference reference) {
        if (reference.getLabel().isEmpty()) {
            throw new BadRequestException("label in reference can't be empty");
        }
        if (!LabelsValidator.isKeyValid(reference.getLabel())) {
            throw new BadRequestException("label in reference is invalid");
        }
    }
}
