package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.time.Instant;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Durations;
import com.google.protobuf.util.Timestamps;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.monitoring.api.v3.Alert;
import ru.yandex.monitoring.api.v3.AlertTimeseries;
import ru.yandex.monitoring.api.v3.AlertType;
import ru.yandex.monitoring.api.v3.ChannelState;
import ru.yandex.monitoring.api.v3.Comparison;
import ru.yandex.monitoring.api.v3.CreateAlertRequest;
import ru.yandex.monitoring.api.v3.DeleteAlertRequest;
import ru.yandex.monitoring.api.v3.DetailedEvaluationStatus;
import ru.yandex.monitoring.api.v3.EvaluationStats;
import ru.yandex.monitoring.api.v3.EvaluationStatus;
import ru.yandex.monitoring.api.v3.ExplainAlertEvaluationRequest;
import ru.yandex.monitoring.api.v3.ExplainAlertEvaluationResponse;
import ru.yandex.monitoring.api.v3.ExplainNewAlertEvaluationRequest;
import ru.yandex.monitoring.api.v3.ExplainNewAlertEvaluationResponse;
import ru.yandex.monitoring.api.v3.ExplainSubAlertEvaluationRequest;
import ru.yandex.monitoring.api.v3.ExplainSubAlertEvaluationResponse;
import ru.yandex.monitoring.api.v3.ExpressionAlert;
import ru.yandex.monitoring.api.v3.GetAlertEvaluationStateRequest;
import ru.yandex.monitoring.api.v3.GetAlertEvaluationStateResponse;
import ru.yandex.monitoring.api.v3.GetAlertEvaluationStatsRequest;
import ru.yandex.monitoring.api.v3.GetAlertEvaluationStatsResponse;
import ru.yandex.monitoring.api.v3.GetAlertNotificationStateRequest;
import ru.yandex.monitoring.api.v3.GetAlertNotificationStateResponse;
import ru.yandex.monitoring.api.v3.GetAlertNotificationStatsRequest;
import ru.yandex.monitoring.api.v3.GetAlertNotificationStatsResponse;
import ru.yandex.monitoring.api.v3.GetAlertStatsRequest;
import ru.yandex.monitoring.api.v3.GetAlertStatsResponse;
import ru.yandex.monitoring.api.v3.GetSubAlertEvaluationStateRequest;
import ru.yandex.monitoring.api.v3.GetSubAlertEvaluationStateResponse;
import ru.yandex.monitoring.api.v3.GetSubAlertNotificationStateRequest;
import ru.yandex.monitoring.api.v3.GetSubAlertNotificationStateResponse;
import ru.yandex.monitoring.api.v3.GetSubAlertRequest;
import ru.yandex.monitoring.api.v3.ListAlertsRequest;
import ru.yandex.monitoring.api.v3.ListAlertsResponse;
import ru.yandex.monitoring.api.v3.ListFullAlertsRequest;
import ru.yandex.monitoring.api.v3.ListFullAlertsResponse;
import ru.yandex.monitoring.api.v3.ListSubAlertsRequest;
import ru.yandex.monitoring.api.v3.ListSubAlertsResponse;
import ru.yandex.monitoring.api.v3.NoMetricsPolicy;
import ru.yandex.monitoring.api.v3.NoPointsPolicy;
import ru.yandex.monitoring.api.v3.NotificationConfig;
import ru.yandex.monitoring.api.v3.NotificationStats;
import ru.yandex.monitoring.api.v3.NotificationStatus;
import ru.yandex.monitoring.api.v3.PredicateRule;
import ru.yandex.monitoring.api.v3.SubAlert;
import ru.yandex.monitoring.api.v3.SubAlertListItem;
import ru.yandex.monitoring.api.v3.TargetStatus;
import ru.yandex.monitoring.api.v3.ThresholdAlert;
import ru.yandex.monitoring.api.v3.ThresholdType;
import ru.yandex.monitoring.api.v3.UpdateAlertRequest;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.protobuf.EAlertState;
import ru.yandex.solomon.alert.protobuf.EAlertType;
import ru.yandex.solomon.alert.protobuf.ECompare;
import ru.yandex.solomon.alert.protobuf.EOrderDirection;
import ru.yandex.solomon.alert.protobuf.EThresholdType;
import ru.yandex.solomon.alert.protobuf.ResolvedEmptyPolicy;
import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.alert.protobuf.TAlertTimeSeries;
import ru.yandex.solomon.alert.protobuf.TChannelConfig;
import ru.yandex.solomon.alert.protobuf.TCreateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TDefaultChannelConfig;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertRequest;
import ru.yandex.solomon.alert.protobuf.TEvaluationState;
import ru.yandex.solomon.alert.protobuf.TEvaluationStats;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TExpression;
import ru.yandex.solomon.alert.protobuf.TListAlert;
import ru.yandex.solomon.alert.protobuf.TListAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListSubAlert;
import ru.yandex.solomon.alert.protobuf.TListSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TNotificationChannelOptions;
import ru.yandex.solomon.alert.protobuf.TNotificationState;
import ru.yandex.solomon.alert.protobuf.TNotificationStats;
import ru.yandex.solomon.alert.protobuf.TNotificationStatus;
import ru.yandex.solomon.alert.protobuf.TPredicateRule;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStatsResponse;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadProjectStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertResponse;
import ru.yandex.solomon.alert.protobuf.TSubAlert;
import ru.yandex.solomon.alert.protobuf.TThreshold;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.model.protobuf.Label;
import ru.yandex.solomon.model.protobuf.Selector;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class AlertDtoConverter {

    public static ExplainNewAlertEvaluationResponse fromProto(ru.yandex.solomon.alert.protobuf.TExplainEvaluationResponse response) {
        return ExplainNewAlertEvaluationResponse.newBuilder()
                .setStatus(AlertDtoConverter.fromProto(response.getEvaluationStatus()))
                .addAllSeries(response.getSeriesList().stream()
                        .map(AlertDtoConverter::fromProto)
                        .collect(Collectors.toList()))
                .build();
    }

    public static TExplainEvaluationRequest toProto(
            ExplainNewAlertEvaluationRequest request,
            long evaluationTimeMillis,
            long deadlineMillis) {
        TAlert alert = AlertDtoConverter.toProto(request);

        TExplainEvaluationRequest.Builder explainRequestBuilder = TExplainEvaluationRequest.newBuilder()
                .setEvaluationTimeMillis(evaluationTimeMillis)
                .setDeadlineMillis(deadlineMillis);

        if (request.getSubAlertLabelsCount() > 0 && request.getGroupByLabelsCount() > 0) {
            var protoLabels = LabelConverter.labelsToProtoList(Labels.of(request.getSubAlertLabelsMap()));
            explainRequestBuilder.setSubAlert(TSubAlert.newBuilder()
                    .addAllGroupKey(protoLabels)
                    .setProjectId(request.getProjectId())
                    .setParent(alert)
                    .build());
        } else {
            explainRequestBuilder.setAlert(alert);
        }

        return explainRequestBuilder.build();
    }

    public static ExplainAlertEvaluationResponse fromAlertProto(ru.yandex.solomon.alert.protobuf.TExplainEvaluationResponse response) {
        return ExplainAlertEvaluationResponse.newBuilder()
                .setStatus(AlertDtoConverter.fromProto(response.getEvaluationStatus()))
                .addAllSeries(response.getSeriesList().stream()
                        .map(AlertDtoConverter::fromProto)
                        .collect(Collectors.toList()))
                .build();
    }

    public static TExplainEvaluationRequest toProto(
            ExplainAlertEvaluationRequest request,
            long evaluationTimeMillis,
            long deadlineMillis) {
        var builder = TExplainEvaluationRequest.TReferenceToAlert.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return TExplainEvaluationRequest.newBuilder()
                .setReference(builder.setAlertId(request.getAlertId())
                        .build())
                .setEvaluationTimeMillis(evaluationTimeMillis)
                .setDeadlineMillis(deadlineMillis)
                .build();
    }

    public static GetAlertNotificationStateResponse fromAlertProto(ru.yandex.solomon.alert.protobuf.TReadNotificationStateResponse response) {
        return GetAlertNotificationStateResponse.newBuilder()
                .addAllChannelStates(response.getStatesList().stream()
                        .map(AlertDtoConverter::fromProto)
                        .collect(Collectors.toList()))
                .build();
    }

    public static TReadNotificationStateRequest toProto(GetAlertNotificationStateRequest request) {
        var builder = TReadNotificationStateRequest.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return builder.setAlertId(request.getAlertId())
                .build();
    }

    public static GetAlertEvaluationStateResponse fromProto(ru.yandex.solomon.alert.protobuf.TReadEvaluationStateResponse response) {
        TEvaluationState state = response.getState();
        return GetAlertEvaluationStateResponse.newBuilder()
                .setStatus(AlertDtoConverter.fromProto(state.getStatus()))
                .setPreviousStatus(AlertDtoConverter.fromProto(state.getPreviousStatus()))
                .setLastEvaluationTime(Timestamps.fromMillis(state.getLatestEvalMillis()))
                .setLastStatusChangeTime(Timestamps.fromMillis(state.getSinceMillis()))
                .build();
    }

    public static TReadEvaluationStateRequest toProto(GetAlertEvaluationStateRequest request) {
        var builder = TReadEvaluationStateRequest.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return builder.setAlertId(request.getAlertId())
                .build();
    }

    public static TReadNotificationStatsRequest toProto(GetAlertNotificationStatsRequest request) {
        var builder = TReadNotificationStatsRequest.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return builder.setAlertId(request.getAlertId())
                .build();
    }

    public static GetAlertNotificationStatsResponse fromProto(ru.yandex.solomon.alert.protobuf.TReadNotificationStatsResponse response) {
        TNotificationStats stats = response.getStats();
        return GetAlertNotificationStatsResponse.newBuilder()
                .setNotificationStats(AlertDtoConverter.fromProto(stats))
                .build();
    }

    public static Alert fromProto(ru.yandex.solomon.alert.protobuf.TUpdateAlertResponse response) {
        return AlertDtoConverter.fromProto(response.getAlert());
    }

    public static Alert fromProto(ru.yandex.solomon.alert.protobuf.TCreateAlertResponse response) {
        return AlertDtoConverter.fromProto(response.getAlert());
    }

    public static TCreateAlertRequest toProto(CreateAlertRequest request, Instant now, String login) {
        var builder = TAlert.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        builder.setId(request.getAlertId())
                .setCreatedAt(now.toEpochMilli())
                .setCreatedBy(login)
                .setUpdatedAt(now.toEpochMilli())
                .setUpdatedBy(login)
                .setName(request.getName())
                .setDescription(request.getDescription())
                .addAllGroupByLabels(request.getGroupByLabelsList())
                .setPeriodMillis(Durations.toMillis(request.getWindow()))
                .setDelaySeconds((int) Durations.toSeconds(request.getDelay()))
                .putAllAnnotations(request.getAnnotationsMap())
                .setResolvedEmptyPolicy(toProto(request.getNoMetricsPolicy()))
                .setNoPointsPolicy(toProto(request.getNoPointsPolicy()))
                .putAllConfiguredNotificationChannels(toProtoMap(request.getChannelsList()))
                .putAllLabels(request.getLabelsMap());

        switch (request.getTypeCase()) {
            case THRESHOLD -> builder.setThreshold(toProto(request.getThreshold()));
            case EXPRESSION -> builder.setExpression(toProto(request.getExpression()));
            default -> throw new UnsupportedOperationException("unsupported alert type: " + request.getTypeCase());
        }

        return TCreateAlertRequest.newBuilder()
                .setAlert(builder.build())
                .build();
    }

    public static GetAlertStatsResponse fromProto(ru.yandex.solomon.alert.protobuf.TReadProjectStatsResponse response) {
        return GetAlertStatsResponse.newBuilder()
                .setAlertCount(response.getAlertsCount())
                .setEvaluationStats(AlertDtoConverter.fromEvaluationStatsProto(response.getEvaluationStats()))
                .setNotificationStats(AlertDtoConverter.fromProto(response.getNotificationStats()))
                .build();
    }

    public static TReadProjectStatsRequest toProto(GetAlertStatsRequest request) {
        return switch (request.getContainerCase()) {
            case PROJECT_ID -> TReadProjectStatsRequest.newBuilder()
                    .setProjectId(request.getProjectId())
                    .build();
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        };
    }

    public static GetAlertEvaluationStatsResponse fromProto(TReadEvaluationStatsResponse response) {
        TEvaluationStats stats = response.getStats();
        return GetAlertEvaluationStatsResponse.newBuilder()
                .setEvaluationStats(AlertDtoConverter.fromEvaluationStatsProto(stats))
                .build();
    }

    public static TReadEvaluationStatsRequest toProto(GetAlertEvaluationStatsRequest request) {
        var builder = TReadEvaluationStatsRequest.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return builder.setAlertId(request.getAlertId())
                .build();
    }

    public static TDeleteAlertRequest toProto(DeleteAlertRequest request) {
        var builder = TDeleteAlertRequest.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return builder.setAlertId(request.getAlertId())
                .build();
    }

    public static ListFullAlertsResponse fromProtoFull(ru.yandex.solomon.alert.protobuf.TListAlertResponse response) {
        List<Alert> alerts = response.getAlertList().getAlertsList().stream()
                .map(AlertDtoConverter::fromProto)
                .collect(Collectors.toList());
        return ListFullAlertsResponse.newBuilder()
                .addAllAlerts(alerts)
                .setNextPageToken(response.getNextPageToken())
                .build();
    }

    public static TListAlertRequest toProto(ListFullAlertsRequest request) {
        var builder = TListAlertRequest.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return builder.setFilterByName(request.getFilter())
                .setPageSize((int) request.getPageSize())
                .setPageToken(request.getPageToken())
                .setFullResultModel(true)
                .build();
    }

    public static ListAlertsResponse fromProto(ru.yandex.solomon.alert.protobuf.TListAlertResponse response) {
        List<ListAlertsResponse.AlertListItem> alerts = response.getAlertsList().stream()
                .map(AlertDtoConverter::fromProto)
                .collect(Collectors.toList());
        return ListAlertsResponse.newBuilder()
                .addAllAlerts(alerts)
                .setNextPageToken(response.getNextPageToken())
                .build();
    }

    public static TListAlertRequest toProto(ListAlertsRequest request) {
        var builder = TListAlertRequest.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return builder.setFilterByName(request.getFilterByName())
                .addAllFilterByState(request.getFilterByStatusesList().stream().map(AlertDtoConverter::toProto).collect(Collectors.toList()))
                .addAllFilterByType(request.getFilterByTypesList().stream().map(AlertDtoConverter::toProto).collect(Collectors.toList()))
                .addAllFilterByEvaluationStatus(request.getFilterByEvaluationStatusesList().stream().map(AlertDtoConverter::toProto).collect(Collectors.toList()))
                .addAllFilterByNotificationStatus(request.getFilterByNotificationStatusesList().stream().map(AlertDtoConverter::toProto).collect(Collectors.toList()))
                .addAllFilterByNotificationId(request.getFilterByChannelIdsList())
                .setOrderByName(AlertDtoConverter.fromProto(request.getOrderByName()))
                .setPageSize((int) request.getPageSize())
                .setPageToken(request.getPageToken())
                .build();
    }

    public static TListSubAlertRequest toProto(ListSubAlertsRequest request) {
        final List<Selector> protoSelectors;

        if (request.getFilterByLabels().isEmpty()) {
            protoSelectors = Collections.emptyList();
        } else {
            Selectors selectors = Selectors.parse(request.getFilterByLabels());
            protoSelectors = LabelSelectorConverter.selectorsToProto(selectors);
        }

        List<TEvaluationStatus.ECode> filterByEvaluationStatus = request.getFilterByEvaluationStatusesList().stream()
                .map(AlertDtoConverter::toProto)
                .collect(Collectors.toList());

        List<TNotificationStatus.ECode> filterByNotificationStatus = request.getFilterByNotificationStatusesList().stream()
                .map(AlertDtoConverter::toProto)
                .collect(Collectors.toList());

        var builder = TListSubAlertRequest.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }

        return builder.setParentId(request.getAlertId())
                .addAllFilterByEvaluationStatus(filterByEvaluationStatus)
                .addAllFilterByLabels(protoSelectors)
                .addAllFilterByNotificationId(request.getFilterByChannelIdsList())
                .addAllFilterByNotificationStatus(filterByNotificationStatus)
                .addAllAnnotationKeys(request.getAnnotationKeysList())
                .setPageToken(request.getPageToken())
                .setPageSize((int) request.getPageSize())
                .setOrderByLabels(toProto(request.getOrderByLabels()))
                .build();
    }

    private static EOrderDirection toProto(ListSubAlertsRequest.SortOrder order) {
        if (order == ListSubAlertsRequest.SortOrder.DESC) {
            return EOrderDirection.DESC;
        }
        return EOrderDirection.ASC;
    }

    public static TReadSubAlertRequest toProto(GetSubAlertRequest request) {
        var builder = TReadSubAlertRequest.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return builder.setParentId(request.getAlertId())
                .setAlertId(request.getSubAlertId())
                .build();
    }

    public static TReadEvaluationStateRequest toProto(GetSubAlertEvaluationStateRequest request) {
        var builder = TReadEvaluationStateRequest.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return builder.setParentId(request.getAlertId())
                .setAlertId(request.getSubAlertId())
                .build();
    }

    public static TReadNotificationStateRequest toProto(GetSubAlertNotificationStateRequest request) {
        var builder = TReadNotificationStateRequest.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return builder.setParentId(request.getAlertId())
                .setAlertId(request.getSubAlertId())
                .build();
    }

    public static TExplainEvaluationRequest toProto(ExplainSubAlertEvaluationRequest request, long evaluationTimeMillis, long deadlineMillis) {
        var builder = TExplainEvaluationRequest.TReferenceToAlert.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return TExplainEvaluationRequest.newBuilder()
                .setReference(builder.setParentId(request.getAlertId())
                        .setAlertId(request.getSubAlertId())
                        .build())
                .setEvaluationTimeMillis(evaluationTimeMillis)
                .setDeadlineMillis(deadlineMillis)
                .build();
    }

    public static ListSubAlertsResponse fromProto(ru.yandex.solomon.alert.protobuf.TListSubAlertResponse response) {
        return ListSubAlertsResponse.newBuilder()
                .addAllSubAlerts(response.getAlertsList().stream().map(AlertDtoConverter::fromProto).collect(Collectors.toList()))
                .setNextPageToken(response.getNextPageToken())
                .build();
    }

    public static GetSubAlertNotificationStateResponse fromSubAlertProto(ru.yandex.solomon.alert.protobuf.TReadNotificationStateResponse response) {
        return GetSubAlertNotificationStateResponse.newBuilder()
                .addAllChannelStates(response.getStatesList().stream()
                        .map(AlertDtoConverter::fromProto)
                        .collect(Collectors.toList()))
                .build();
    }

    public static ExplainSubAlertEvaluationResponse fromSubAlertProto(ru.yandex.solomon.alert.protobuf.TExplainEvaluationResponse response) {
        return ExplainSubAlertEvaluationResponse.newBuilder()
                .setStatus(AlertDtoConverter.fromProto(response.getEvaluationStatus()))
                .addAllSeries(response.getSeriesList().stream().map(AlertDtoConverter::fromProto).collect(Collectors.toList()))
                .build();
    }

    public static GetSubAlertEvaluationStateResponse fromProto(TEvaluationState state) {
        return GetSubAlertEvaluationStateResponse.newBuilder()
                .setLastEvaluationTime(Timestamps.fromMillis(state.getLatestEvalMillis()))
                .setLastStatusChangeTime(Timestamps.fromMillis(state.getSinceMillis()))
                .setStatus(AlertDtoConverter.fromProto(state.getStatus()))
                .setPreviousStatus(AlertDtoConverter.fromProto(state.getPreviousStatus()))
                .build();
    }

    private static EvaluationStats fromEvaluationStatsProto(TEvaluationStats stats) {
        return EvaluationStats.newBuilder()
                .setOk(stats.getCountOk())
                .setWarn(stats.getCountWarn())
                .setAlarm(stats.getCountAlarm())
                .setError(stats.getCountError())
                .setNoData(stats.getCountNoData())
                .build();
    }

    private static NotificationStats fromProto(TNotificationStats stats) {
        return NotificationStats.newBuilder()
                .setUnknown(stats.getCountUnknown())
                .setSuccess(stats.getCountSuccess())
                .setError(stats.getCountError())
                .setInvalidRequest(stats.getCountInvalidRequest())
                .setAbsentNotificationChannel(stats.getCountAbsentNotificationChannel())
                .setRetryError(stats.getCountRetryError())
                .setResourceExhausted(stats.getCountResourceExhausted())
                .setPermissionDenied(stats.getCountPermissionDenied())
                .setNotSubscribed(stats.getCountNotSubscribed())
                .build();
    }

    private static ChannelState fromProto(TNotificationState state) {
        return ChannelState.newBuilder()
                .setProjectId(state.getProjectId())
                .setChannelId(state.getNotificationChannelId())
                .setStatus(fromProto(state.getStatus().getCode()))
                .setLastEvaluationTime(Timestamps.fromMillis(state.getLatestEvalMillis()))
                .setLastSuccessfulTime(Timestamps.fromMillis(state.getLatestSuccessMillis()))
                .build();
    }

    private static NotificationStatus fromProto(TNotificationStatus.ECode status) {
        return switch (status) {
            case SKIP_REPEAT -> NotificationStatus.SKIP_REPEAT;
            case SKIP_BY_STATUS -> NotificationStatus.SKIP_BY_STATUS;
            case SUCCESS -> NotificationStatus.SUCCESS;
            case ERROR -> NotificationStatus.ERROR;
            case INVALID_REQUEST -> NotificationStatus.INVALID_REQUEST;
            case OBSOLETE -> NotificationStatus.OBSOLETE;
            case ABSENT_NOTIFICATION_CHANNEL -> NotificationStatus.ABSENT_NOTIFICATION_CHANNEL;
            case ERROR_ABLE_TO_RETRY -> NotificationStatus.ERROR_ABLE_TO_RETRY;
            case RESOURCE_EXHAUSTED -> NotificationStatus.RESOURCE_EXHAUSTED;
            case PERMISSION_DENIED -> NotificationStatus.PERMISSION_DENIED;
            case NOT_SUBSCRIBED -> NotificationStatus.NOT_SUBSCRIBED;
            default -> NotificationStatus.NOTIFICATION_STATUS_UNSPECIFIED;
        };
    }

    private static TNotificationStatus.ECode toProto(NotificationStatus status) {
        return switch (status) {
            case SKIP_REPEAT -> TNotificationStatus.ECode.SKIP_REPEAT;
            case SKIP_BY_STATUS -> TNotificationStatus.ECode.SKIP_BY_STATUS;
            case SUCCESS -> TNotificationStatus.ECode.SUCCESS;
            case ERROR -> TNotificationStatus.ECode.ERROR;
            case INVALID_REQUEST -> TNotificationStatus.ECode.INVALID_REQUEST;
            case OBSOLETE -> TNotificationStatus.ECode.OBSOLETE;
            case ABSENT_NOTIFICATION_CHANNEL -> TNotificationStatus.ECode.ABSENT_NOTIFICATION_CHANNEL;
            case ERROR_ABLE_TO_RETRY -> TNotificationStatus.ECode.ERROR_ABLE_TO_RETRY;
            case RESOURCE_EXHAUSTED -> TNotificationStatus.ECode.RESOURCE_EXHAUSTED;
            case PERMISSION_DENIED -> TNotificationStatus.ECode.PERMISSION_DENIED;
            case NOT_SUBSCRIBED -> TNotificationStatus.ECode.NOT_SUBSCRIBED;
            default -> throw new IllegalArgumentException("unknown notification status: " + status);
        };
    }

    private static AlertTimeseries fromProto(TAlertTimeSeries timeseries) {
        var timestamps = timeseries.getTimeMillisList()
                .stream().map(Timestamps::fromMillis)
                .collect(Collectors.toList());

        return AlertTimeseries.newBuilder()
                .setAlias(timeseries.getAlias())
                .addAllTimestamps(timestamps)
                .addAllValues(timeseries.getValuesList())
                .build();
    }

    public static Alert fromProto(TAlert alert) {
        var builder = Alert.newBuilder()
                .setId(alert.getId())
                .setProjectId(alert.getProjectId())
                .setCreatedAt(Timestamps.fromMillis(alert.getCreatedAt()))
                .setCreatedBy(alert.getCreatedBy())
                .setModifiedAt(Timestamps.fromMillis(alert.getUpdatedAt()))
                .setModifiedBy(alert.getUpdatedBy())
                .setName(alert.getName())
                .setDescription(alert.getDescription())
                .setStatus(fromProto(alert.getState()))
                .addAllGroupByLabels(alert.getGroupByLabelsList())
                .setWindow(Durations.fromMillis(alert.getPeriodMillis()))
                .setDelay(Durations.fromSeconds(alert.getDelaySeconds()))
                .putAllAnnotations(alert.getAnnotationsMap())
                .setNoMetricsPolicy(fromProto(alert.getResolvedEmptyPolicy()))
                .setNoPointsPolicy(fromProto(alert.getNoPointsPolicy()))
                .addAllChannels(fromProto(alert.getConfiguredNotificationChannelsMap()))
                .putAllLabels(alert.getLabelsMap());

        switch (alert.getTypeCase()) {
            case THRESHOLD -> builder.setThreshold(fromProto(alert.getThreshold()));
            case EXPRESSION -> builder.setExpression(fromProto(alert.getExpression()));
            default -> throw new UnsupportedOperationException("unsupported alert type: " + alert.getTypeCase());
        }

        return builder.build();
    }

    private static ThresholdAlert fromProto(TThreshold threshold) {
        return ThresholdAlert.newBuilder()
                .setSelectors(Selectors.format(LabelSelectorConverter.protoToSelectors(threshold.getNewSelectors())))
                .setTransformations(threshold.getTransformations())
                .addAllPredicateRules(fromProtoList(threshold.getPredicateRulesList()))
                .build();
    }

    private static TThreshold toProto(ThresholdAlert threshold) {
        return TThreshold.newBuilder()
                .setNewSelectors(LabelSelectorConverter.selectorsToNewProto(Selectors.parse(threshold.getSelectors())))
                .setTransformations(threshold.getTransformations())
                .addAllPredicateRules(toProtoList(threshold.getPredicateRulesList()))
                .build();
    }

    private static List<PredicateRule> fromProtoList(List<TPredicateRule> predicateRules) {
        return predicateRules.stream()
                .map(AlertDtoConverter::fromProto)
                .collect(Collectors.toList());
    }

    private static List<TPredicateRule> toProtoList(List<PredicateRule> predicateRules) {
        return predicateRules.stream()
                .map(AlertDtoConverter::toProto)
                .collect(Collectors.toList());
    }

    private static PredicateRule fromProto(TPredicateRule predicateRule) {
        return PredicateRule.newBuilder()
                .setType(fromProto(predicateRule.getThresholdType()))
                .setThreshold(predicateRule.getThreshold())
                .setComparison(fromProto(predicateRule.getComparison()))
                .setTargetStatus(fromProto(predicateRule.getTargetStatus()))
                .build();
    }

    private static TPredicateRule toProto(PredicateRule predicateRule) {
        return TPredicateRule.newBuilder()
                .setThresholdType(toProto(predicateRule.getType()))
                .setThreshold(predicateRule.getThreshold())
                .setComparison(toProto(predicateRule.getComparison()))
                .setTargetStatus(toProto(predicateRule.getTargetStatus()))
                .build();
    }

    protected static TargetStatus fromProto(TPredicateRule.ETargetStatus targetStatus) {
        return switch (targetStatus) {
            case OK -> TargetStatus.TARGET_STATUS_OK;
            case WARN -> TargetStatus.TARGET_STATUS_WARN;
            case ALARM -> TargetStatus.TARGET_STATUS_ALARM;
            case NO_DATA -> TargetStatus.TARGET_STATUS_NO_DATA;
            default -> TargetStatus.TARGET_STATUS_UNSPECIFIED;
        };
    }

    protected static TPredicateRule.ETargetStatus toProto(TargetStatus targetStatus) {
        return switch (targetStatus) {
            case TARGET_STATUS_OK -> TPredicateRule.ETargetStatus.OK;
            case TARGET_STATUS_WARN -> TPredicateRule.ETargetStatus.WARN;
            case TARGET_STATUS_ALARM -> TPredicateRule.ETargetStatus.ALARM;
            case TARGET_STATUS_NO_DATA -> TPredicateRule.ETargetStatus.NO_DATA;
            default -> TPredicateRule.ETargetStatus.UNSPECIFIED;
        };
    }

    protected static Comparison fromProto(ECompare comparison) {
        return switch (comparison) {
            case EQ -> Comparison.COMPARISON_EQ;
            case NE -> Comparison.COMPARISON_NE;
            case GT -> Comparison.COMPARISON_GT;
            case LT -> Comparison.COMPARISON_LT;
            case GTE -> Comparison.COMPARISON_GTE;
            case LTE -> Comparison.COMPARISON_LTE;
            default -> Comparison.COMPARISON_UNSPECIFIED;
        };
    }

    protected static ECompare toProto(Comparison comparison) {
        return switch (comparison) {
            case COMPARISON_EQ -> ECompare.EQ;
            case COMPARISON_NE -> ECompare.NE;
            case COMPARISON_GT -> ECompare.GT;
            case COMPARISON_LT -> ECompare.LT;
            case COMPARISON_GTE -> ECompare.GTE;
            case COMPARISON_LTE -> ECompare.LTE;
            default -> ECompare.UNRECOGNIZED;
        };
    }

    protected static ThresholdType fromProto(EThresholdType thresholdType) {
        return switch (thresholdType) {
            case AT_LEAST_ONE -> ThresholdType.THRESHOLD_TYPE_AT_LEAST_ONE;
            case AT_ALL_TIMES -> ThresholdType.THRESHOLD_TYPE_AT_ALL_TIMES;
            case LAST_NON_NAN -> ThresholdType.THRESHOLD_TYPE_LAST_NON_NAN;
            case AVG -> ThresholdType.THRESHOLD_TYPE_AVG;
            case MIN -> ThresholdType.THRESHOLD_TYPE_MIN;
            case MAX -> ThresholdType.THRESHOLD_TYPE_MAX;
            case SUM -> ThresholdType.THRESHOLD_TYPE_SUM;
            case COUNT -> ThresholdType.THRESHOLD_TYPE_COUNT;
            default -> ThresholdType.THRESHOLD_TYPE_UNSPECIFIED;
        };
    }

    protected static EThresholdType toProto(ThresholdType thresholdType) {
        return switch (thresholdType) {
            case THRESHOLD_TYPE_AT_LEAST_ONE -> EThresholdType.AT_LEAST_ONE;
            case THRESHOLD_TYPE_AT_ALL_TIMES -> EThresholdType.AT_ALL_TIMES;
            case THRESHOLD_TYPE_LAST_NON_NAN -> EThresholdType.LAST_NON_NAN;
            case THRESHOLD_TYPE_AVG -> EThresholdType.AVG;
            case THRESHOLD_TYPE_MIN -> EThresholdType.MIN;
            case THRESHOLD_TYPE_MAX -> EThresholdType.MAX;
            case THRESHOLD_TYPE_SUM -> EThresholdType.SUM;
            case THRESHOLD_TYPE_COUNT -> EThresholdType.COUNT;
            default -> EThresholdType.UNRECOGNIZED;
        };
    }

    private static ExpressionAlert fromProto(TExpression expression) {
        final String program;
        if (StringUtils.isBlank(expression.getCheckExpression())) {
            program = expression.getProgram();
        } else {
            program = expression.getProgram() + "\nalarm_if(" + expression.getCheckExpression() + ");";
        }

        return ExpressionAlert.newBuilder()
                .setProgram(program)
                .build();
    }

    private static TExpression toProto(ExpressionAlert expression) {
        return TExpression.newBuilder()
                .setProgram(expression.getProgram())
                .build();
    }

    private static List<NotificationConfig> fromProto(Map<String, TNotificationChannelOptions> channelsMap) {
        return channelsMap.entrySet().stream()
                .map(entry -> fromProto(entry.getKey(), entry.getValue()))
                .collect(Collectors.toList());
    }

    private static Map<String, TNotificationChannelOptions> toProtoMap(List<NotificationConfig> channelsList) {
        return channelsList.stream()
                .collect(Collectors.toMap(NotificationConfig::getChannelId, AlertDtoConverter::toProto));
    }

    private static NotificationConfig fromProto(String channelId, TNotificationChannelOptions options) {
        var builder = NotificationConfig.newBuilder()
                .setChannelId(channelId);

        if (options.getOptionsCase() == TNotificationChannelOptions.OptionsCase.CHANNELCONFIG) {
            TChannelConfig channelConfig = options.getChannelConfig();
            builder.setRepeatPeriod(Durations.fromMillis(channelConfig.getRepeatNotifyDelayMillis()));
            builder.addAllNotifyAboutStatuses(fromProto(channelConfig.getNotifyAboutStatusesList()));
        }

        return builder.build();
    }

    private static TNotificationChannelOptions toProto(NotificationConfig config) {
        if (config.getNotifyAboutStatusesCount() == 0) {
            return TNotificationChannelOptions.newBuilder()
                    .setDefaultChannelConfig(TDefaultChannelConfig.getDefaultInstance())
                    .build();
        }

        TChannelConfig channelConfig = TChannelConfig.newBuilder()
                .setRepeatNotifyDelayMillis(Durations.toSeconds(config.getRepeatPeriod()))
                .addAllNotifyAboutStatuses(toProto(config.getNotifyAboutStatusesList()))
                .build();

        return TNotificationChannelOptions.newBuilder()
                .setChannelConfig(channelConfig)
                .build();
    }

    private static List<EvaluationStatus> fromProto(List<TEvaluationStatus.ECode> notifyAboutStatuses) {
        return notifyAboutStatuses.stream()
                .map(AlertDtoConverter::fromProto)
                .collect(Collectors.toList());
    }

    private static List<TEvaluationStatus.ECode> toProto(List<EvaluationStatus> notifyAboutStatuses) {
        return notifyAboutStatuses.stream()
                .map(AlertDtoConverter::toProto)
                .collect(Collectors.toList());
    }

    protected static EvaluationStatus fromProto(TEvaluationStatus.ECode status) {
        return switch (status) {
            case OK -> EvaluationStatus.EVALUATION_STATUS_OK;
            case WARN -> EvaluationStatus.EVALUATION_STATUS_WARN;
            case ALARM -> EvaluationStatus.EVALUATION_STATUS_ALARM;
            case NO_DATA -> EvaluationStatus.EVALUATION_STATUS_NO_DATA;
            case ERROR -> EvaluationStatus.EVALUATION_STATUS_ERROR;
            default -> EvaluationStatus.EVALUATION_STATUS_UNSPECIFIED;
        };
    }

    protected static TEvaluationStatus.ECode toProto(EvaluationStatus status) {
        return switch (status) {
            case EVALUATION_STATUS_OK -> TEvaluationStatus.ECode.OK;
            case EVALUATION_STATUS_WARN -> TEvaluationStatus.ECode.WARN;
            case EVALUATION_STATUS_ALARM -> TEvaluationStatus.ECode.ALARM;
            case EVALUATION_STATUS_NO_DATA -> TEvaluationStatus.ECode.NO_DATA;
            case EVALUATION_STATUS_ERROR -> TEvaluationStatus.ECode.ERROR;
            default -> TEvaluationStatus.ECode.UNRECOGNIZED;
        };
    }

    protected static NoPointsPolicy fromProto(ru.yandex.solomon.alert.protobuf.NoPointsPolicy policy) {
        return switch (policy) {
            case NO_POINTS_OK -> NoPointsPolicy.NO_POINTS_POLICY_OK;
            case NO_POINTS_WARN -> NoPointsPolicy.NO_POINTS_POLICY_WARN;
            case NO_POINTS_ALARM -> NoPointsPolicy.NO_POINTS_POLICY_ALARM;
            case NO_POINTS_NO_DATA -> NoPointsPolicy.NO_POINTS_POLICY_NO_DATA;
            case NO_POINTS_MANUAL -> NoPointsPolicy.NO_POINTS_POLICY_MANUAL;
            default -> NoPointsPolicy.NO_POINTS_POLICY_UNSPECIFIED;
        };
    }

    protected static ru.yandex.solomon.alert.protobuf.NoPointsPolicy toProto(NoPointsPolicy policy) {
        return switch (policy) {
            case NO_POINTS_POLICY_OK -> ru.yandex.solomon.alert.protobuf.NoPointsPolicy.NO_POINTS_OK;
            case NO_POINTS_POLICY_WARN -> ru.yandex.solomon.alert.protobuf.NoPointsPolicy.NO_POINTS_WARN;
            case NO_POINTS_POLICY_ALARM -> ru.yandex.solomon.alert.protobuf.NoPointsPolicy.NO_POINTS_ALARM;
            case NO_POINTS_POLICY_NO_DATA -> ru.yandex.solomon.alert.protobuf.NoPointsPolicy.NO_POINTS_NO_DATA;
            case NO_POINTS_POLICY_MANUAL -> ru.yandex.solomon.alert.protobuf.NoPointsPolicy.NO_POINTS_MANUAL;
            default -> ru.yandex.solomon.alert.protobuf.NoPointsPolicy.NO_POINTS_DEFAULT;
        };
    }

    protected static NoMetricsPolicy fromProto(ResolvedEmptyPolicy policy) {
        return switch (policy) {
            case RESOLVED_EMPTY_OK -> NoMetricsPolicy.NO_METRICS_POLICY_OK;
            case RESOLVED_EMPTY_WARN -> NoMetricsPolicy.NO_METRICS_POLICY_WARN;
            case RESOLVED_EMPTY_ALARM -> NoMetricsPolicy.NO_METRICS_POLICY_ALARM;
            case RESOLVED_EMPTY_NO_DATA -> NoMetricsPolicy.NO_METRICS_POLICY_NO_DATA;
            case RESOLVED_EMPTY_MANUAL -> NoMetricsPolicy.NO_METRICS_POLICY_MANUAL;
            default -> NoMetricsPolicy.NO_METRICS_POLICY_UNSPECIFIED;
        };
    }

    protected static ResolvedEmptyPolicy toProto(NoMetricsPolicy policy) {
        return switch (policy) {
            case NO_METRICS_POLICY_OK -> ResolvedEmptyPolicy.RESOLVED_EMPTY_OK;
            case NO_METRICS_POLICY_WARN -> ResolvedEmptyPolicy.RESOLVED_EMPTY_WARN;
            case NO_METRICS_POLICY_ALARM -> ResolvedEmptyPolicy.RESOLVED_EMPTY_ALARM;
            case NO_METRICS_POLICY_NO_DATA -> ResolvedEmptyPolicy.RESOLVED_EMPTY_NO_DATA;
            case NO_METRICS_POLICY_MANUAL -> ResolvedEmptyPolicy.RESOLVED_EMPTY_MANUAL;
            default -> ResolvedEmptyPolicy.RESOLVED_EMPTY_DEFAULT;
        };
    }

    private static Alert.Status fromProto(EAlertState state) {
        return switch (state) {
            case ACTIVE -> Alert.Status.ACTIVE;
            case DELETED -> Alert.Status.DELETED;
            case MUTED -> Alert.Status.INACTIVE;
            default -> Alert.Status.STATUS_UNSPECIFIED;
        };
    }

    private static EAlertState toProto(Alert.Status status) {
        return switch (status) {
            case DELETED -> EAlertState.DELETED;
            case INACTIVE -> EAlertState.MUTED;
            default -> EAlertState.ACTIVE;
        };
    }

    public static TAlert toProto(UpdateAlertRequest request, int etag, String login, Instant now) {
        var builder = TAlert.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        builder.setId(request.getAlertId())
                .setUpdatedAt(now.toEpochMilli())
                .setUpdatedBy(login)
                .setName(request.getName())
                .setDescription(request.getDescription())
                .addAllGroupByLabels(request.getGroupByLabelsList())
                .setPeriodMillis(Durations.toMillis(request.getWindow()))
                .setDelaySeconds((int) Durations.toSeconds(request.getDelay()))
                .putAllAnnotations(request.getAnnotationsMap())
                .setResolvedEmptyPolicy(toProto(request.getNoMetricsPolicy()))
                .setNoPointsPolicy(toProto(request.getNoPointsPolicy()))
                .putAllConfiguredNotificationChannels(toProtoMap(request.getChannelsList()))
                .putAllLabels(request.getLabelsMap())
                .setVersion(etag);

        switch (request.getTypeCase()) {
            case THRESHOLD -> builder.setThreshold(toProto(request.getThreshold()));
            case EXPRESSION -> builder.setExpression(toProto(request.getExpression()));
            default -> throw new UnsupportedOperationException("unsupported alert type: " + request.getTypeCase());
        }

        return builder.build();
    }

    private static TAlert toProto(ExplainNewAlertEvaluationRequest request) {
        var builder = TAlert.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        builder.setName(request.getName())
                .setDescription(request.getDescription())
                .addAllGroupByLabels(request.getGroupByLabelsList())
                .setPeriodMillis(Durations.toMillis(request.getWindow()))
                .setDelaySeconds((int) Durations.toSeconds(request.getDelay()))
                .putAllAnnotations(request.getAnnotationsMap())
                .setResolvedEmptyPolicy(toProto(request.getNoMetricsPolicy()))
                .setNoPointsPolicy(toProto(request.getNoPointsPolicy()));

        switch (request.getTypeCase()) {
            case THRESHOLD -> builder.setThreshold(toProto(request.getThreshold()));
            case EXPRESSION -> builder.setExpression(toProto(request.getExpression()));
            default -> throw new UnsupportedOperationException("unsupported alert type: " + request.getTypeCase());
        }

        return builder.build();
    }

    public static SubAlert fromProto(TReadSubAlertResponse response) {
        TSubAlert subAlert = response.getSubAlert();

        LinkedHashMap<String, String> labels = subAlert.getGroupKeyList()
                .stream()
                .collect(Collectors.toMap(
                        Label::getKey,
                        Label::getValue,
                        (u, v) -> {
                            throw new IllegalStateException("duplicate key " + u);
                        },
                        () -> new LinkedHashMap<>(subAlert.getGroupKeyCount())));

        return SubAlert.newBuilder()
                .setProjectId(subAlert.getProjectId())
                .setId(subAlert.getId())
                .setParent(AlertDtoConverter.fromProto(subAlert.getParent()))
                .putAllLabels(labels)
                .build();
    }

    private static SubAlertListItem fromProto(TListSubAlert subAlertListItem) {
        return SubAlertListItem.newBuilder()
                .setProjectId(subAlertListItem.getProjectId())
                .setAlertId(subAlertListItem.getParentId())
                .setId(subAlertListItem.getId())
                .setStatus(fromProto(subAlertListItem.getEvaluationStatusCode()))
                .setNotificationStats(fromProto(subAlertListItem.getNotificationStats()))
                .setLastEvaluationTime(Timestamps.fromMillis(subAlertListItem.getLatestEvalMillis()))
                .putAllLabels(LabelConverter.protoToLabels(subAlertListItem.getLabelsList()).toMap())
                .putAllAnnotations(subAlertListItem.getAnnotationsMap())
                .build();
    }


    private static ListAlertsResponse.AlertListItem fromProto(TListAlert alertListItem) {
        return ListAlertsResponse.AlertListItem.newBuilder()
                .setProjectId(alertListItem.getProjectId())
                .setId(alertListItem.getId())
                .setName(alertListItem.getName())
                .setType(fromProto(alertListItem.getAlertType()))
                .setEvaluationStats(fromEvaluationStatsProto(alertListItem.getEvaluationStats()))
                .setNotificationStats(fromProto(alertListItem.getNotificationStats()))
                .setMultiAlert(alertListItem.getMultiAlert())
                .build();
    }

    private static EAlertType toProto(AlertType alertType) {
        return switch (alertType) {
            case ALERT_TYPE_THRESHOLD -> EAlertType.THRESHOLD;
            case ALERT_TYPE_EXPRESSION -> EAlertType.EXPRESSION;
            default -> throw new BadRequestException("Unexpected value: " + alertType);
        };
    }

    private static AlertType fromProto(EAlertType alertType) {
        return switch (alertType) {
            case THRESHOLD -> AlertType.ALERT_TYPE_THRESHOLD;
            case EXPRESSION -> AlertType.ALERT_TYPE_EXPRESSION;
            default -> AlertType.ALERT_TYPE_UNSPECIFIED;
        };
    }

    private static EOrderDirection fromProto(ListAlertsRequest.SortOrder sortDirection) {
        if (sortDirection == ListAlertsRequest.SortOrder.DESC) {
            return EOrderDirection.DESC;
        }
        return EOrderDirection.ASC;
    }

    private static DetailedEvaluationStatus fromProto(TEvaluationStatus status) {
        return DetailedEvaluationStatus.newBuilder()
                .setStatus(fromProto(status.getCode()))
                .putAllAnnotations(status.getAnnotationsMap())
                .build();
    }
}
