package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Durations;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.monitoring.api.v3.DoubleParameter;
import ru.yandex.monitoring.api.v3.ExpressionTemplate;
import ru.yandex.monitoring.api.v3.IntegerParameter;
import ru.yandex.monitoring.api.v3.LabelValuesParameter;
import ru.yandex.monitoring.api.v3.ListAlertTemplateResponse;
import ru.yandex.monitoring.api.v3.Parameter;
import ru.yandex.monitoring.api.v3.TextParameter;
import ru.yandex.monitoring.api.v3.TextValuesParameter;
import ru.yandex.monitoring.api.v3.ThresholdTemplate;
import ru.yandex.monitoring.api.v3.UnpublishAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.AlertTemplate;
import ru.yandex.solomon.alert.protobuf.AlertTemplateParameter;
import ru.yandex.solomon.alert.protobuf.AlertTemplateStatus;
import ru.yandex.solomon.alert.protobuf.CreateAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.DeleteAlertTemplatePublicationRequest;
import ru.yandex.solomon.alert.protobuf.DeployAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.DoubleTemplateParameter;
import ru.yandex.solomon.alert.protobuf.Expression;
import ru.yandex.solomon.alert.protobuf.IntegerTemplateParameter;
import ru.yandex.solomon.alert.protobuf.LabelListTemplateParameter;
import ru.yandex.solomon.alert.protobuf.ListAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.ListAlertTemplateVersionsRequest;
import ru.yandex.solomon.alert.protobuf.ListAlertTemplateVersionsResponse;
import ru.yandex.solomon.alert.protobuf.PredicateRule;
import ru.yandex.solomon.alert.protobuf.PublishAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.ReadAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.TemplateDeployPolicy;
import ru.yandex.solomon.alert.protobuf.TextListTemplateParameter;
import ru.yandex.solomon.alert.protobuf.TextTemplateParameter;
import ru.yandex.solomon.alert.protobuf.Threshold;
import ru.yandex.solomon.gateway.api.utils.IdGenerator;

import static ru.yandex.monitoring.api.v3.AlertTemplateParameter.Type.ALERT_DATA;
import static ru.yandex.monitoring.api.v3.AlertTemplateParameter.Type.SERVICE_PROVIDER_DATA;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class AlertTemplateDtoConverter {

    public CreateAlertTemplateRequest toInternalProto(ru.yandex.monitoring.api.v3.CreateAlertTemplateRequest request) {
        return CreateAlertTemplateRequest.newBuilder()
                .setAlertTemplate(toInternalProto(request.getAlertTemplate(), request.getServiceProviderId()))
                .build();
    }

    public AlertTemplate toInternalProto(ru.yandex.monitoring.api.v3.AlertTemplate alertTemplate, String serviceProviderId) {
        final AlertTemplate.Builder builder = AlertTemplate.newBuilder();
        switch (alertTemplate.getTypeCase()) {
            case THRESHOLD_TEMPLATE -> builder.setThreshold(thresholdToInternalProto(alertTemplate.getThresholdTemplate()))
                    .addAllGroupByLabels(alertTemplate.getThresholdTemplate().getGroupByLabelsList())
                    .setPeriodMillis(Durations.toMillis(alertTemplate.getThresholdTemplate().getWindow()))
                    .setDelaySeconds((int) Durations.toSeconds(alertTemplate.getThresholdTemplate().getDelay()))
                    .setResolvedEmptyPolicy(AlertDtoConverter.toProto(alertTemplate.getThresholdTemplate().getNoMetricsPolicy()))
                    .setNoPointsPolicy(AlertDtoConverter.toProto(alertTemplate.getThresholdTemplate().getNoPointsPolicy()));
            case EXPRESSION_TEMPLATE -> builder.setExpression(expressionToInternalProto(alertTemplate.getExpressionTemplate()))
                    .addAllGroupByLabels(alertTemplate.getExpressionTemplate().getGroupByLabelsList())
                    .setPeriodMillis(Durations.toMillis(alertTemplate.getExpressionTemplate().getWindow()))
                    .setDelaySeconds((int) Durations.toSeconds(alertTemplate.getExpressionTemplate().getDelay()))
                    .setResolvedEmptyPolicy(AlertDtoConverter.toProto(alertTemplate.getExpressionTemplate().getNoMetricsPolicy()))
                    .setNoPointsPolicy(AlertDtoConverter.toProto(alertTemplate.getExpressionTemplate().getNoPointsPolicy()));
            default -> throw new UnsupportedOperationException("Unsupported alert template type: " + alertTemplate);
        }
        builder.setId(alertTemplate.getId())
                .setTemplateVersionTag(StringUtils.isEmpty(alertTemplate.getTemplateVersionTag())
                        ? IdGenerator.generateInternalId()
                        : alertTemplate.getTemplateVersionTag())
                .setServiceProviderId(serviceProviderId)
                .setName(alertTemplate.getName())
                .setDescription(alertTemplate.getDescription())
                .setCreatedAt(alertTemplate.getCreatedAt())
                .setCreatedBy(alertTemplate.getCreatedBy())
                .putAllAnnotations(alertTemplate.getAnnotationsMap())
                .putAllLabels(alertTemplate.getLabelsMap())
                .setDefault(alertTemplate.getIsDefault())
                .setAlertTemplateStatus(toInternalProto(alertTemplate.getAlertTemplateStatus()))
                .setSeverityValue(alertTemplate.getSeverityValue())
                .build();
        for (var p : alertTemplate.getAlertTemplateParametersList()) {
            switch (p.getType()) {
                case SERVICE_PROVIDER_DATA -> builder.addAlertTemplateParameters(toInternalProto(p));
                case ALERT_DATA -> builder.addAlertTemplateThresholds(toInternalProto(p));
                default -> throw new UnsupportedOperationException("Unsupported alert template parameter type: " + p.getType());
            }
        }
        return builder.build();
    }

    private AlertTemplateParameter toInternalProto(ru.yandex.monitoring.api.v3.AlertTemplateParameter param) {
        var builder = AlertTemplateParameter.newBuilder();
        final Parameter parameter = param.getParameter();
        switch (parameter.getDataCase()) {
            case LABEL_VALUES -> builder.setLabelListParameter(LabelListTemplateParameter.newBuilder()
                    .setName(parameter.getName())
                    .setTitle(parameter.getTitle())
                    .setSelectors(parameter.getLabelValues().getSelectors())
                    .setLabelKey(parameter.getLabelValues().getLabelKey())
                    .setProjectId(parameter.getLabelValues().getProjectId())
                    .setMultiselectable(parameter.getLabelValues().getMultiselectable())
                    .addAllDefaultValues(parameter.getLabelValues().getDefaultValuesList())
                    .setDescription(parameter.getDescription())
                    .build());
            case TEXT -> builder.setTextParameter(TextTemplateParameter.newBuilder()
                    .setName(parameter.getName())
                    .setTitle(parameter.getTitle())
                    .setDefaultValue(parameter.getText().getDefaultValue())
                    .setDescription(parameter.getDescription())
                    .build());
            case INTEGER_PARAMETER -> builder.setIntegerParameter(IntegerTemplateParameter.newBuilder()
                    .setName(parameter.getName())
                    .setTitle(parameter.getTitle())
                    .setDefaultValue(parameter.getIntegerParameter().getDefaultValue())
                    .setDescription(parameter.getDescription())
                    .setUnitFormat(parameter.getIntegerParameter().getUnitFormat())
                    .build());
            case DOUBLE_PARAMETER -> builder.setDoubleParameter(DoubleTemplateParameter.newBuilder()
                    .setName(parameter.getName())
                    .setTitle(parameter.getTitle())
                    .setDefaultValue(parameter.getDoubleParameter().getDefaultValue())
                    .setDescription(parameter.getDescription())
                    .setUnitFormat(parameter.getDoubleParameter().getUnitFormat())
                    .build());
            case TEXT_VALUES -> builder.setTextListParameter(TextListTemplateParameter.newBuilder()
                    .setName(parameter.getName())
                    .setTitle(parameter.getTitle())
                    .addAllDefaultValues(parameter.getTextValues().getDefaultValuesList())
                    .setDescription(parameter.getDescription())
                    .build());
            default -> throw new UnsupportedOperationException("Unsupported alert template parameter type: " + parameter.getDataCase());
        }
        return builder.build();
    }

    private Threshold thresholdToInternalProto(ThresholdTemplate template) {
        // TODO(alextrushkin) implement query in SOLOMON-7869
        Threshold.Builder builder = Threshold.newBuilder()
                .setSelectors(template.getQuery());

        template.getTemplatePredicateRulesList().stream()
                .map(this::toInternalProto)
                .forEach(builder::addPredicateRules);

        return builder.build();
    }

    private ru.yandex.solomon.alert.protobuf.PredicateRule toInternalProto(ThresholdTemplate.TemplatePredicateRule templatePredicateRule) {
        var builder = PredicateRule.newBuilder()
                .setThresholdType(AlertDtoConverter.toProto(templatePredicateRule.getThresholdType()))
                .setComparison(AlertDtoConverter.toProto(templatePredicateRule.getComparison()))
                .setTargetStatus(AlertDtoConverter.toProto(templatePredicateRule.getTargetStatus()));

        switch (templatePredicateRule.getThresholdCase()) {
            case VALUE -> builder.setThreshold(templatePredicateRule.getValue());
            case THRESHOLD_PARAMETER_TEMPLATE -> builder.setThresholdParameterTemplate(templatePredicateRule.getThresholdParameterTemplate());
            default -> throw new UnsupportedOperationException("Unsupported threshold type: " + templatePredicateRule.getThresholdCase());
        }
        return builder.build();
    }

    private Expression expressionToInternalProto(ExpressionTemplate template) {
        return Expression.newBuilder()
                .setProgram(template.getProgram())
                .build();
    }

    public ru.yandex.monitoring.api.v3.AlertTemplate fromInternalProto(AlertTemplate alertTemplate) {
        var builder = ru.yandex.monitoring.api.v3.AlertTemplate.newBuilder();
        switch (alertTemplate.getTypeCase()) {
            case THRESHOLD -> builder.setThresholdTemplate(fromInternalProto(alertTemplate, alertTemplate.getThreshold()));
            case EXPRESSION -> builder.setExpressionTemplate(fromInternalProto(alertTemplate, alertTemplate.getExpression()));
            default -> throw new UnsupportedOperationException("Unsupported alert template type: " + alertTemplate);
        }
        builder.setId(alertTemplate.getId())
                .setTemplateVersionTag(alertTemplate.getTemplateVersionTag())
                .setServiceProviderId(alertTemplate.getServiceProviderId())
                .setName(alertTemplate.getName())
                .setDescription(alertTemplate.getDescription())
                .setCreatedAt(alertTemplate.getCreatedAt())
                .setCreatedBy(alertTemplate.getCreatedBy())
                .putAllAnnotations(alertTemplate.getAnnotationsMap())
                .putAllLabels(alertTemplate.getLabelsMap())
                .setAlertTemplateStatus(fromInternalProto(alertTemplate.getAlertTemplateStatus()))
                .setIsDefault(alertTemplate.getDefault())
                .setSeverityValue(alertTemplate.getSeverityValue())
                .build();
        for (var p : alertTemplate.getAlertTemplateParametersList()) {
            builder.addAlertTemplateParameters(ru.yandex.monitoring.api.v3.AlertTemplateParameter.newBuilder()
                    .setType(SERVICE_PROVIDER_DATA)
                    .setParameter(fromInternalProto(p))
                    .build());
        }
        for (var p : alertTemplate.getAlertTemplateThresholdsList()) {
            builder.addAlertTemplateParameters(ru.yandex.monitoring.api.v3.AlertTemplateParameter.newBuilder()
                    .setType(ALERT_DATA)
                    .setParameter(fromInternalProto(p))
                    .build());
        }
        return builder.build();
    }

    private Parameter fromInternalProto(AlertTemplateParameter param) {
        var builder = Parameter.newBuilder();
        switch (param.getParameterCase()) {
            case DOUBLE_PARAMETER -> builder.setDoubleParameter(DoubleParameter.newBuilder()
                    .setDefaultValue(param.getDoubleParameter().getDefaultValue())
                    .setUnitFormat(param.getDoubleParameter().getUnitFormat())
                    .build())
                    .setTitle(param.getDoubleParameter().getTitle())
                    .setDescription(param.getDoubleParameter().getDescription())
                    .setName(param.getDoubleParameter().getName());
            case INTEGER_PARAMETER -> builder.setIntegerParameter(IntegerParameter.newBuilder()
                    .setDefaultValue(param.getIntegerParameter().getDefaultValue())
                    .setUnitFormat(param.getIntegerParameter().getUnitFormat())
                    .build())
                    .setTitle(param.getIntegerParameter().getTitle())
                    .setDescription(param.getIntegerParameter().getDescription())
                    .setName(param.getIntegerParameter().getName());
            case TEXT_PARAMETER -> builder.setText(TextParameter.newBuilder()
                    .setDefaultValue(param.getTextParameter().getDefaultValue())
                    .build())
                    .setTitle(param.getTextParameter().getTitle())
                    .setDescription(param.getTextParameter().getDescription())
                    .setName(param.getTextParameter().getName());
            case TEXT_LIST_PARAMETER -> builder.setTextValues(TextValuesParameter.newBuilder()
                    .addAllDefaultValues(param.getTextListParameter().getDefaultValuesList())
                    .build())
                    .setTitle(param.getTextListParameter().getTitle())
                    .setDescription(param.getTextListParameter().getDescription())
                    .setName(param.getTextListParameter().getName());
            case LABEL_LIST_PARAMETER -> builder.setLabelValues(LabelValuesParameter.newBuilder()
                    .addAllDefaultValues(param.getLabelListParameter().getDefaultValuesList())
                    .setLabelKey(param.getLabelListParameter().getLabelKey())
                    .setSelectors(param.getLabelListParameter().getSelectors())
                    .setProjectId(param.getLabelListParameter().getProjectId())
                    .setMultiselectable(param.getLabelListParameter().getMultiselectable())
                    .build())
                    .setTitle(param.getLabelListParameter().getTitle())
                    .setDescription(param.getLabelListParameter().getDescription())
                    .setName(param.getLabelListParameter().getName());
            default -> throw new UnsupportedOperationException("Unsupported alert template parameter type: " + param.getParameterCase());
        }
        return builder.build();
    }

    private ThresholdTemplate fromInternalProto(AlertTemplate alertTemplate, Threshold threshold) {
        return ThresholdTemplate.newBuilder()
                .setQuery(threshold.getSelectors())
                .addAllTemplatePredicateRules(threshold.getPredicateRulesList().stream()
                        .map(this::fromInternalProto)
                        .collect(Collectors.toList()))
                .addAllGroupByLabels(alertTemplate.getGroupByLabelsList())
                .setNoPointsPolicy(AlertDtoConverter.fromProto(alertTemplate.getNoPointsPolicy()))
                .setNoMetricsPolicy(AlertDtoConverter.fromProto(alertTemplate.getResolvedEmptyPolicy()))
                .setDelay(Durations.fromSeconds(alertTemplate.getDelaySeconds()))
                .setWindow(Durations.fromMillis(alertTemplate.getPeriodMillis()))
                .build();
    }

    private ThresholdTemplate.TemplatePredicateRule fromInternalProto(PredicateRule predicateRule) {
        var builder = ThresholdTemplate.TemplatePredicateRule.newBuilder()
                .setThresholdType(AlertDtoConverter.fromProto(predicateRule.getThresholdType()))
                .setComparison(AlertDtoConverter.fromProto(predicateRule.getComparison()))
                .setTargetStatus(AlertDtoConverter.fromProto(predicateRule.getTargetStatus()));
        if (StringUtils.isEmpty(predicateRule.getThresholdParameterTemplate())) {
            builder.setValue(predicateRule.getThreshold());
        } else {
            builder.setThresholdParameterTemplate(predicateRule.getThresholdParameterTemplate());
        }
        return builder.build();
    }

    private ExpressionTemplate fromInternalProto(AlertTemplate alertTemplate, Expression expression) {
        return ExpressionTemplate.newBuilder()
                .setProgram(expression.getProgram())
                .addAllGroupByLabels(alertTemplate.getGroupByLabelsList())
                .setNoPointsPolicy(AlertDtoConverter.fromProto(alertTemplate.getNoPointsPolicy()))
                .setNoMetricsPolicy(AlertDtoConverter.fromProto(alertTemplate.getResolvedEmptyPolicy()))
                .setDelay(Durations.fromSeconds(alertTemplate.getDelaySeconds()))
                .setWindow(Durations.fromMillis(alertTemplate.getPeriodMillis()))
                .build();
    }

    public ReadAlertTemplateRequest toInternalProto(ru.yandex.monitoring.api.v3.ReadAlertTemplateRequest request) {
        return ReadAlertTemplateRequest.newBuilder()
                .setTemplateId(request.getTemplateId())
                .setTemplateVersionTag(request.getTemplateVersionTag())
                .build();
    }

    public PublishAlertTemplateRequest toInternalProto(ru.yandex.monitoring.api.v3.PublishAlertTemplateRequest request) {
        return PublishAlertTemplateRequest.newBuilder()
                .setTemplateId(request.getTemplateId())
                .setTemplateVersionTag(request.getTemplateVersionTag())
                .build();
    }

    public DeleteAlertTemplatePublicationRequest toInternalProto(UnpublishAlertTemplateRequest request) {
        return DeleteAlertTemplatePublicationRequest.newBuilder()
                .setTemplateId(request.getTemplateId())
                .build();
    }

    public DeployAlertTemplateRequest toInternalProto(ru.yandex.monitoring.api.v3.DeployAlertTemplateRequest request) {
        return DeployAlertTemplateRequest.newBuilder()
                .setTemplateId(request.getTemplateId())
                .setTemplateVersionTag(request.getTemplateVersionTag())
                .setTemplateDeployPolicy(toInternalModel(request.getTemplateDeployPolicy()))
                .build();
    }

    private TemplateDeployPolicy toInternalModel(ru.yandex.monitoring.api.v3.DeployAlertTemplateRequest.TemplateDeployPolicy templateDeployPolicy) {
        switch (templateDeployPolicy) {
            case AUTO -> {
                return TemplateDeployPolicy.TEMPLATE_DEPLOY_POLICY_AUTO;
            }
            case MANUAL -> {
                return TemplateDeployPolicy.TEMPLATE_DEPLOY_POLICY_MANUAL;
            }
            default -> throw new UnsupportedOperationException("Unsupported deploy policy: " + templateDeployPolicy);
        }
    }

    private ru.yandex.monitoring.api.v3.AlertTemplateStatus fromInternalProto(AlertTemplateStatus alertTemplateStatus) {
        switch (alertTemplateStatus) {
            case ALERT_TEMPLATE_STATUS_UNSPECIFIED -> {
                return ru.yandex.monitoring.api.v3.AlertTemplateStatus.ALERT_TEMPLATE_STATUS_UNSPECIFIED;
            }
            case ALERT_TEMPLATE_STATUS_PUBLISHED -> {
                return ru.yandex.monitoring.api.v3.AlertTemplateStatus.ALERT_TEMPLATE_STATUS_PUBLISHED;
            }
            case ALERT_TEMPLATE_STATUS_DRAFT -> {
                return ru.yandex.monitoring.api.v3.AlertTemplateStatus.ALERT_TEMPLATE_STATUS_DRAFT;
            }
            case UNRECOGNIZED -> {
                return ru.yandex.monitoring.api.v3.AlertTemplateStatus.UNRECOGNIZED;
            }
            default -> throw new UnsupportedOperationException("Unsupported AlertTemplateStatus: " + alertTemplateStatus);
        }
    }

    private AlertTemplateStatus toInternalProto(ru.yandex.monitoring.api.v3.AlertTemplateStatus alertTemplateStatus) {
        switch (alertTemplateStatus) {
            case ALERT_TEMPLATE_STATUS_UNSPECIFIED -> {
                return AlertTemplateStatus.ALERT_TEMPLATE_STATUS_UNSPECIFIED;
            }
            case ALERT_TEMPLATE_STATUS_PUBLISHED -> {
                return AlertTemplateStatus.ALERT_TEMPLATE_STATUS_PUBLISHED;
            }
            case ALERT_TEMPLATE_STATUS_DRAFT -> {
                return AlertTemplateStatus.ALERT_TEMPLATE_STATUS_DRAFT;
            }
            case UNRECOGNIZED -> {
                return AlertTemplateStatus.UNRECOGNIZED;
            }
            default -> throw new UnsupportedOperationException("Unsupported AlertTemplateStatus: " + alertTemplateStatus);
        }
    }

    public ListAlertTemplateVersionsRequest toInternalProto(ru.yandex.monitoring.api.v3.ListAlertTemplateVersionsRequest request) {
        return ListAlertTemplateVersionsRequest.newBuilder()
                .setTemplateId(request.getTemplateId())
                .setPageSize(request.getPageSize())
                .setPageToken(request.getPageToken())
                .build();
    }

    public ru.yandex.monitoring.api.v3.ListAlertTemplateVersionsResponse fromInternalProto(ListAlertTemplateVersionsResponse response) {
        return ru.yandex.monitoring.api.v3.ListAlertTemplateVersionsResponse.newBuilder()
                .setNextPageToken(response.getNextPageToken())
                .addAllAlertTemplates(response.getAlertTemplatesList().stream()
                        .map(this::fromInternalProto)
                        .collect(Collectors.toList()))
                .build();
    }

    public ListAlertTemplateRequest toInternalProto(ru.yandex.monitoring.api.v3.ListAlertTemplateRequest request) {
        return ListAlertTemplateRequest.newBuilder()
                .setServiceProviderId(request.getServiceProviderId())
                .setPageSize(request.getPageSize())
                .setPageToken(request.getPageToken())
                .setNameFilter(request.getNameFilter())
                .addAllAlertTemplateStatusesFilter(request.getAlertTemplateStatusesFilterList().stream()
                        .map(this::toInternalProto)
                        .collect(Collectors.toList()))
                .build();
    }

    public ListAlertTemplateResponse fromInternalProto(ru.yandex.solomon.alert.protobuf.ListAlertTemplateResponse request) {
        return ListAlertTemplateResponse.newBuilder()
                .setNextPageToken(request.getNextPageToken())
                .addAllAlertTemplates(request.getAlertTemplatesList().stream()
                        .map(this::fromInternalProto)
                        .collect(Collectors.toList()))
                .build();
    }
}
