package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.time.Instant;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Durations;
import com.google.protobuf.util.Timestamps;

import ru.yandex.monitoring.api.v3.Channel;
import ru.yandex.monitoring.api.v3.CloudEmailChannel;
import ru.yandex.monitoring.api.v3.CloudMobilePushChannel;
import ru.yandex.monitoring.api.v3.CloudSmsChannel;
import ru.yandex.monitoring.api.v3.CreateChannelRequest;
import ru.yandex.monitoring.api.v3.DatalensEmailChannel;
import ru.yandex.monitoring.api.v3.DeleteChannelRequest;
import ru.yandex.monitoring.api.v3.EmailChannel;
import ru.yandex.monitoring.api.v3.GetChannelRequest;
import ru.yandex.monitoring.api.v3.JugglerChannel;
import ru.yandex.monitoring.api.v3.ListChannelsRequest;
import ru.yandex.monitoring.api.v3.ListChannelsResponse;
import ru.yandex.monitoring.api.v3.SmsChannel;
import ru.yandex.monitoring.api.v3.TelegramChannel;
import ru.yandex.monitoring.api.v3.UpdateChannelRequest;
import ru.yandex.monitoring.api.v3.WebhookChannel;
import ru.yandex.monitoring.api.v3.YandexMessengerChannel;
import ru.yandex.solomon.alert.protobuf.EOrderDirection;
import ru.yandex.solomon.alert.protobuf.TCreateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TListNotificationsRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.notification.ENotificationChannelType;
import ru.yandex.solomon.alert.protobuf.notification.TCloudEmailType;
import ru.yandex.solomon.alert.protobuf.notification.TCloudPushType;
import ru.yandex.solomon.alert.protobuf.notification.TCloudSmsType;
import ru.yandex.solomon.alert.protobuf.notification.TDatalensEmailType;
import ru.yandex.solomon.alert.protobuf.notification.TEmailType;
import ru.yandex.solomon.alert.protobuf.notification.THeader;
import ru.yandex.solomon.alert.protobuf.notification.TJugglerType;
import ru.yandex.solomon.alert.protobuf.notification.TNotification;
import ru.yandex.solomon.alert.protobuf.notification.TSmsType;
import ru.yandex.solomon.alert.protobuf.notification.TTelegramType;
import ru.yandex.solomon.alert.protobuf.notification.TWebhookType;
import ru.yandex.solomon.alert.protobuf.notification.TYaChatsType;

import static java.util.stream.Collectors.toMap;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public final class ChannelDtoConverter {
    public static TReadNotificationRequest toProto(GetChannelRequest request) {
        var builder = TReadNotificationRequest.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return builder.setNotificationId(request.getChannelId())
                .build();
    }

    public static ListChannelsResponse fromProto(ru.yandex.solomon.alert.protobuf.TListNotificationsResponse response) {
        return ListChannelsResponse.newBuilder()
                .addAllChannels(response.getNotificationList().stream()
                        .map(ChannelDtoConverter::fromProto)
                        .collect(Collectors.toList()))
                .setNextPageToken(response.getNextPageToken())
                .build();
    }

    public static TListNotificationsRequest toProto(ListChannelsRequest request) {
        var builder = TListNotificationsRequest.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return builder.setFilterByName(request.getFilter())
                .addAllFilterByType(request.getFilterByTypeList().stream()
                        .map(ChannelDtoConverter::toProto)
                        .collect(Collectors.toList()))
                .setPageSize((int) request.getPageSize())
                .setOrderByName(toProto(request.getOrderByName()))
                .setOrderByType(toProto(request.getOrderByType()))
                .setPageToken(request.getPageToken())
                .build();
    }

    private static EOrderDirection toProto(ListChannelsRequest.SortOrder order) {
        if (order == ListChannelsRequest.SortOrder.DESC) {
            return EOrderDirection.DESC;
        }
        return EOrderDirection.ASC;
    }

    private static ENotificationChannelType toProto(ListChannelsRequest.ChannelType channelType) {
        return switch (channelType) {
            case EMAIL -> ENotificationChannelType.EMAIL;
            case JUGGLER -> ENotificationChannelType.JUGGLER;
            case WEBHOOK -> ENotificationChannelType.WEBHOOK;
            case SMS -> ENotificationChannelType.SMS;
            case TELEGRAM -> ENotificationChannelType.TELEGRAM;
            case CLOUD_EMAIL -> ENotificationChannelType.CLOUD_EMAIL;
            case CLOUD_SMS -> ENotificationChannelType.CLOUD_SMS;
            case YA_CHATS -> ENotificationChannelType.YA_CHATS;
            case DATALENS_EMAIL -> ENotificationChannelType.DATALENS_EMAIL;
            case CLOUD_PUSH -> ENotificationChannelType.CLOUD_PUSH;
            default -> ENotificationChannelType.UNKNOWN;
        };
    }

    public static TCreateNotificationRequest toProto(CreateChannelRequest request, Instant now, String login) {
        var builder = TNotification.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        builder.setId(request.getChannelId())
                .setCreatedAt(now.toEpochMilli())
                .setUpdatedAt(now.toEpochMilli())
                .setCreatedBy(login)
                .setUpdatedBy(login)
                .setName(request.getName())
                .setDescription(request.getDescription())
                .addAllNotifyAboutStatuses(request.getNotifyAboutStatusesList().stream()
                        .map(AlertDtoConverter::toProto)
                        .collect(Collectors.toList()))
                .setRepeatNotifyDelayMillis(Durations.toMillis(request.getRepeatPeriod()))
                .putAllLabels(request.getLabelsMap());

        switch (request.getTypeCase()) {
            case EMAIL -> builder.setEmail(ChannelDtoConverter.toProto(request.getEmail()));
            case JUGGLER -> builder.setJuggler(ChannelDtoConverter.toProto(request.getJuggler()));
            case WEBHOOK -> builder.setWebkook(ChannelDtoConverter.toProto(request.getWebhook()));
            case SMS -> builder.setSms(ChannelDtoConverter.toProto(request.getSms()));
            case TELEGRAM -> builder.setTelegram(ChannelDtoConverter.toProto(request.getTelegram()));
            case CLOUD_EMAIL -> builder.setCloudEmail(ChannelDtoConverter.toProto(request.getCloudEmail()));
            case CLOUD_SMS -> builder.setCloudSms(ChannelDtoConverter.toProto(request.getCloudSms()));
            case CLOUD_MOBILE_PUSH -> builder.setCloudPush(ChannelDtoConverter.toProto(request.getCloudMobilePush()));
            case YANDEX_MESSENGER -> builder.setYaChats(ChannelDtoConverter.toProto(request.getYandexMessenger()));
            case DATALENS_EMAIL -> builder.setDatalensEmail(ChannelDtoConverter.toProto(request.getDatalensEmail()));
            default -> throw new IllegalStateException("unknown yandex messenger channel type: " + request.getTypeCase());
        }

        return TCreateNotificationRequest.newBuilder()
                .setNotification(builder.build())
                .build();
    }

    public static TUpdateNotificationRequest toProto(
            UpdateChannelRequest request,
            int etag,
            Instant now,
            String login)
    {
        var builder = TNotification.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }

        builder.setId(request.getChannelId())
                .setUpdatedAt(now.toEpochMilli())
                .setUpdatedBy(login)
                .setName(request.getName())
                .setDescription(request.getDescription())
                .addAllNotifyAboutStatuses(request.getNotifyAboutStatusesList().stream()
                        .map(AlertDtoConverter::toProto)
                        .collect(Collectors.toList()))
                .setRepeatNotifyDelayMillis(Durations.toMillis(request.getRepeatPeriod()))
                .setVersion(etag)
                .putAllLabels(request.getLabelsMap());

        switch (request.getTypeCase()) {
            case EMAIL -> builder.setEmail(ChannelDtoConverter.toProto(request.getEmail()));
            case JUGGLER -> builder.setJuggler(ChannelDtoConverter.toProto(request.getJuggler()));
            case WEBHOOK -> builder.setWebkook(ChannelDtoConverter.toProto(request.getWebhook()));
            case SMS -> builder.setSms(ChannelDtoConverter.toProto(request.getSms()));
            case TELEGRAM -> builder.setTelegram(ChannelDtoConverter.toProto(request.getTelegram()));
            case CLOUD_EMAIL -> builder.setCloudEmail(ChannelDtoConverter.toProto(request.getCloudEmail()));
            case CLOUD_SMS -> builder.setCloudSms(ChannelDtoConverter.toProto(request.getCloudSms()));
            case CLOUD_MOBILE_PUSH -> builder.setCloudPush(ChannelDtoConverter.toProto(request.getCloudMobilePush()));
            case YANDEX_MESSENGER -> builder.setYaChats(ChannelDtoConverter.toProto(request.getYandexMessenger()));
            case DATALENS_EMAIL -> builder.setDatalensEmail(ChannelDtoConverter.toProto(request.getDatalensEmail()));
            default -> throw new IllegalStateException("unknown yandex messenger channel type: " + request.getTypeCase());
        }

        return TUpdateNotificationRequest.newBuilder()
                .setNotification(builder.build())
                .build();
    }

    public static Channel fromProto(TNotification channel) {
        var builder = Channel.newBuilder()
                .setId(channel.getId())
                .setProjectId(channel.getProjectId())
                .setCreatedAt(Timestamps.fromMillis(channel.getCreatedAt()))
                .setModifiedAt(Timestamps.fromMillis(channel.getUpdatedAt()))
                .setCreatedBy(channel.getCreatedBy())
                .setModifiedBy(channel.getUpdatedBy())
                .setName(channel.getName())
                .setDescription(channel.getDescription())
                .addAllNotifyAboutStatuses(channel.getNotifyAboutStatusesList().stream()
                        .map(AlertDtoConverter::fromProto)
                        .collect(Collectors.toList()))
                .setRepeatPeriod(Durations.fromSeconds(channel.getRepeatNotifyDelayMillis()))
                .putAllLabels(channel.getLabelsMap());

        switch (channel.getTypeCase()) {
            case EMAIL -> builder.setEmail(fromProto(channel.getEmail()));
            case WEBKOOK -> builder.setWebhook(fromProto(channel.getWebkook()));
            case JUGGLER -> builder.setJuggler(fromProto(channel.getJuggler()));
            case SMS -> builder.setSms(fromProto(channel.getSms()));
            case TELEGRAM -> builder.setTelegram(fromProto(channel.getTelegram()));
            case CLOUDEMAIL -> builder.setCloudEmail(fromProto(channel.getCloudEmail()));
            case CLOUDSMS -> builder.setCloudSms(fromProto(channel.getCloudSms()));
            case CLOUDPUSH -> builder.setCloudMobilePush(fromProto(channel.getCloudPush()));
            case YACHATS -> builder.setYandexMessenger(fromProto(channel.getYaChats()));
            case DATALENSEMAIL -> builder.setDatalensEmail(fromProto(channel.getDatalensEmail()));
            default -> throw new IllegalStateException("unknown channel type: " + channel.getTypeCase());
        }

        return builder.build();
    }

    public static TDeleteNotificationRequest toProto(DeleteChannelRequest request) {
        var builder = TDeleteNotificationRequest.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return builder.setNotificationId(request.getChannelId())
                .build();
    }

    private static TYaChatsType toProto(YandexMessengerChannel yaMessenger) {
        var builder = TYaChatsType.newBuilder()
                .setTextTemplate(yaMessenger.getTextTemplate());

        switch (yaMessenger.getTypeCase()) {
            case LOGIN -> builder.setLogin(yaMessenger.getLogin());
            case GROUP_ID -> builder.setGroupId(yaMessenger.getGroupId());
            default -> throw new IllegalStateException("unkown yandex messenger channel type: " + yaMessenger.getTypeCase());
        }

        return builder.build();
    }

    private static TCloudSmsType toProto(CloudSmsChannel cloudSms) {
        return TCloudSmsType.newBuilder()
                .addAllRecipients(cloudSms.getRecipientsList())
                .build();
    }

    private static TCloudEmailType toProto(CloudEmailChannel cloudEmail) {
        return TCloudEmailType.newBuilder()
                .addAllRecipients(cloudEmail.getRecipientsList())
                .build();
    }

    private static TTelegramType toProto(TelegramChannel telegram) {
        var builder = TTelegramType.newBuilder()
                .setTextTemplate(telegram.getTextTemplate())
                .setSendScreenshot(telegram.getSendScreenshot());

        switch (telegram.getTypeCase()) {
            case LOGIN -> builder.setLogin(telegram.getLogin());
            case GROUP_TITLE -> builder.setGroupTitle(telegram.getGroupTitle());
            default -> throw new IllegalStateException("unkown telegram channel type: " + telegram.getTypeCase());
        }

        return builder.build();
    }

    private static TSmsType toProto(SmsChannel sms) {
        return TSmsType.newBuilder()
                .setLogin(sms.getLogin())
                .setPhone(sms.getPhone())
                .setTextTemplate(sms.getTextTemplate())
                .build();
    }

    private static TWebhookType toProto(WebhookChannel webhook) {
        return TWebhookType.newBuilder()
                .setUrl(webhook.getUrl())
                .setTemplate(webhook.getBodyTemplate())
                .addAllHeaders(webhook.getHeadersMap().entrySet().stream()
                        .map(entry -> THeader.newBuilder().setName(entry.getKey()).setValue(entry.getValue()).build())
                        .collect(Collectors.toList()))
                .build();
    }

    private static TJugglerType toProto(JugglerChannel juggler) {
        return TJugglerType.newBuilder()
                .setHost(juggler.getHost())
                .setService(juggler.getService())
                .setDescription(juggler.getDescription())
                .addAllTags(juggler.getTagsList())
                .build();
    }

    private static TEmailType toProto(EmailChannel email) {
        return TEmailType.newBuilder()
                .addAllRecipients(email.getRecipientsList())
                .setSubjectTemplate(email.getSubjectTemplate())
                .setContentTemplate(email.getContentTemplate())
                .build();
    }

    private static TDatalensEmailType toProto(DatalensEmailChannel datalensEmail) {
        return TDatalensEmailType.newBuilder()
                .addAllRecipients(datalensEmail.getRecipientsList())
                .build();
    }

    private static TCloudPushType toProto(CloudMobilePushChannel cloudPush) {
        return TCloudPushType.newBuilder()
                .addAllRecipients(cloudPush.getRecipientsList())
                .build();
    }

    private static DatalensEmailChannel fromProto(TDatalensEmailType datalensEmail) {
        return DatalensEmailChannel.newBuilder()
                .addAllRecipients(datalensEmail.getRecipientsList())
                .build();
    }

    private static YandexMessengerChannel fromProto(TYaChatsType yaChats) {
        var yaBuilder = YandexMessengerChannel.newBuilder()
                .setTextTemplate(yaChats.getTextTemplate());

        switch (yaChats.getReceiverCase()) {
            case LOGIN -> yaBuilder.setLogin(yaChats.getLogin());
            case GROUPID -> yaBuilder.setGroupId(yaChats.getGroupId());
            default -> throw new IllegalStateException("unknown Yandex messenger channel type: " + yaChats.getReceiverCase());
        }

        return yaBuilder.build();
    }

    private static CloudMobilePushChannel fromProto(TCloudPushType cloudPush) {
        return CloudMobilePushChannel.newBuilder()
                .addAllRecipients(cloudPush.getRecipientsList())
                .build();
    }

    private static CloudSmsChannel fromProto(TCloudSmsType cloudSms) {
        return CloudSmsChannel.newBuilder()
                .addAllRecipients(cloudSms.getRecipientsList())
                .build();
    }

    private static CloudEmailChannel fromProto(TCloudEmailType cloudEmail) {
        return CloudEmailChannel.newBuilder()
                .addAllRecipients(cloudEmail.getRecipientsList())
                .build();
    }

    private static TelegramChannel fromProto(TTelegramType telegram) {
        TelegramChannel.Builder telegramBuilder = TelegramChannel.newBuilder()
                .setTextTemplate(telegram.getTextTemplate())
                .setSendScreenshot(telegram.getSendScreenshot());
        switch (telegram.getReceiverCase()) {
            case LOGIN -> telegramBuilder.setLogin(telegram.getLogin());
            case GROUPTITLE -> telegramBuilder.setGroupTitle(telegram.getGroupTitle());
            default -> throw new IllegalStateException("unknown telegram channel type: " + telegram.getReceiverCase());
        }
        return telegramBuilder.build();
    }

    private static SmsChannel fromProto(TSmsType sms) {
        return SmsChannel.newBuilder()
                .setLogin(sms.getLogin())
                .setTextTemplate(sms.getTextTemplate())
                .build();
    }

    private static JugglerChannel fromProto(TJugglerType juggler) {
        return JugglerChannel.newBuilder()
                .setHost(juggler.getHost())
                .setService(juggler.getService())
                .setDescription(juggler.getDescription())
                .addAllTags(juggler.getTagsList())
                .build();
    }

    private static WebhookChannel fromProto(TWebhookType webkook) {
        return WebhookChannel.newBuilder()
                .setUrl(webkook.getUrl())
                .setBodyTemplate(webkook.getTemplate())
                .putAllHeaders(webkook.getHeadersList()
                        .stream()
                        .collect(toMap(THeader::getName, THeader::getValue)))
                .build();
    }

    private static EmailChannel fromProto(TEmailType email) {
        return EmailChannel.newBuilder()
                .addAllRecipients(email.getRecipientsList())
                .setSubjectTemplate(email.getSubjectTemplate())
                .setContentTemplate(email.getContentTemplate())
                .build();
    }
}
