package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Timestamps;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.monitoring.api.v3.CloudDns;
import ru.yandex.monitoring.api.v3.Cluster;
import ru.yandex.monitoring.api.v3.ConductorGroup;
import ru.yandex.monitoring.api.v3.ConductorTag;
import ru.yandex.monitoring.api.v3.CreateClusterRequest;
import ru.yandex.monitoring.api.v3.HostUrl;
import ru.yandex.monitoring.api.v3.InstanceGroup;
import ru.yandex.monitoring.api.v3.NannyGroup;
import ru.yandex.monitoring.api.v3.Network;
import ru.yandex.monitoring.api.v3.PatternHost;
import ru.yandex.monitoring.api.v3.QloudGroup;
import ru.yandex.monitoring.api.v3.UpdateClusterRequest;
import ru.yandex.monitoring.api.v3.YpCluster;
import ru.yandex.solomon.core.db.model.CloudEnv;
import ru.yandex.solomon.core.db.model.ClusterCloudDnsConf;
import ru.yandex.solomon.core.db.model.ClusterConductorGroupConf;
import ru.yandex.solomon.core.db.model.ClusterConductorTagConf;
import ru.yandex.solomon.core.db.model.ClusterHostListConf;
import ru.yandex.solomon.core.db.model.ClusterHostUrlConf;
import ru.yandex.solomon.core.db.model.ClusterInstanceGroupConf;
import ru.yandex.solomon.core.db.model.ClusterNannyGroupConf;
import ru.yandex.solomon.core.db.model.ClusterNetworkConf;
import ru.yandex.solomon.core.db.model.ClusterQloudGroupConf;
import ru.yandex.solomon.core.db.model.ClusterYpConf;
import ru.yandex.solomon.core.db.model.NannyEnv;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ClusterDtoConverter {
    public static Pair<Cluster, Integer> fromModelWithVersion(ru.yandex.solomon.core.db.model.Cluster model) {
        var dto = fromModel(model);
        var version = model.getVersion();
        return Pair.of(dto, version);
    }

    public static Cluster fromModel(ru.yandex.solomon.core.db.model.Cluster model) {
        var builder = Cluster.newBuilder()
                .setProjectId(model.getProjectId())
                .setId(model.getId())
                .setCreatedAt(Timestamps.fromMillis(model.getUpdatedAtMillis()))
                .setModifiedAt(Timestamps.fromMillis(model.getUpdatedAtMillis()))
                .setCreatedBy(model.getCreatedBy())
                .setModifiedBy(model.getUpdatedBy())
                .setLabelName(model.getName())
                .setDescription(model.getDescription())
                .putAllLabels(model.getLabels());

        for (var host : model.getHosts()) {
            builder.addHosts(PatternHost.newBuilder()
                    .setUrlPattern(host.getUrlPattern())
                    .setDc(host.getDc())
                    .setRanges(host.getRanges())
                    .putAllLabels(toLabelsMap(host.getLabels()))
                    .build());
        }
        for (var hostUrl : model.getHostUrls()) {
            builder.addHostUrls(HostUrl.newBuilder()
                    .setUrl(hostUrl.getUrl())
                    .setIgnorePorts(hostUrl.isIgnorePorts())
                    .putAllLabels(toLabelsMap(hostUrl.getLabels()))
                    .build());
        }
        for (var conductorGroup : model.getConductorGroups()) {
            builder.addConductorGroups(ConductorGroup.newBuilder()
                    .setGroup(conductorGroup.getGroup())
                    .putAllLabels(toLabelsMap(conductorGroup.getLabels()))
                    .build());
        }
        for (var conductorTag : model.getConductorTags()) {
            builder.addConductorTags(ConductorTag.newBuilder()
                    .setName(conductorTag.getName())
                    .putAllLabels(toLabelsMap(conductorTag.getLabels()))
                    .build());
        }

        for (var nannyGroup : model.getNannyGroups()) {
            builder.addNannyGroups(NannyGroup.newBuilder()
                    .setService(nannyGroup.getService())
                    .setUseFetchedPort(nannyGroup.isUseFetchedPort())
                    .setEnv(toProto(nannyGroup.getEnv()))
                    .setPortShift(nannyGroup.getPortShift())
                    .addAllCfgGroup(List.of(nannyGroup.getCfgGroup()))
                    .putAllLabels(toLabelsMap(nannyGroup.getLabels()))
                    .build());
        }

        for (var qloudGroup : model.getQloudGroups()) {
            builder.addQloudGroups(QloudGroup.newBuilder()
                    .setComponent(qloudGroup.getComponent())
                    .setEnvironment(qloudGroup.getEnvironment())
                    .setApplication(qloudGroup.getApplication())
                    .setProject(qloudGroup.getProject())
                    .setDeployment(qloudGroup.getDeployment())
                    .putAllLabels(toLabelsMap(qloudGroup.getLabels()))
                    .build());
        }

        for (var network : model.getNetworks()) {
            builder.addNetworks(Network.newBuilder()
                    .setNetwork(network.getNetwork())
                    .setPort(network.getPort())
                    .putAllLabels(toLabelsMap(network.getLabels()))
                    .build());
        }

        for (var ypCluster : model.getYpClusters()) {
            builder.addYpClusters(YpCluster.newBuilder()
                    .setPodSetId(ypCluster.getPodSetId())
                    .setEndpointSetId(ypCluster.getEndpointSetId())
                    .setCluster(ypCluster.getCluster())
                    .setYpLabel(ypCluster.getYpLabel())
                    .setTvmLabel(ypCluster.getTvmLabel())
                    .putAllLabels(toLabelsMap(ypCluster.getLabels()))
                    .build());
        }

        for (var instanceGroup : model.getInstanceGroups()) {
            builder.addInstanceGroups(InstanceGroup.newBuilder()
                    .setInstanceGroupId(instanceGroup.getInstanceGroupId())
                    .setFolderId(instanceGroup.getFolderId())
                    .putAllLabels(toLabelsMap(instanceGroup.getLabels()))
                    .build());
        }

        for (var item : model.getCloudDns()) {
            builder.addCloudDns(CloudDns.newBuilder()
                .setEnv(toProto(item.getEnv()))
                .setName(item.getName())
                .putAllLabels(toLabelsMap(item.getLabels()))
                .build());
        }

        builder.setShardSettings(ShardSettingsDtoConverter.fromModel(model.getShardSettings()));

        return builder.build();
    }

    private static ru.yandex.monitoring.api.v3.CloudEnv toProto(CloudEnv env) {
        return switch (env) {
            case UNKNOWN -> ru.yandex.monitoring.api.v3.CloudEnv.CLOUD_ENV_UNSPECIFIED;
            case PREPROD -> ru.yandex.monitoring.api.v3.CloudEnv.PREPROD;
            case PROD -> ru.yandex.monitoring.api.v3.CloudEnv.PROD;
            case HWLAB -> ru.yandex.monitoring.api.v3.CloudEnv.HWLAB;
            case TESTING -> ru.yandex.monitoring.api.v3.CloudEnv.TESTING;
            case ISRAEL -> ru.yandex.monitoring.api.v3.CloudEnv.ISRAEL;
        };
    }

    private static CloudEnv fromProto(ru.yandex.monitoring.api.v3.CloudEnv env) {
        return switch (env) {
            case CLOUD_ENV_UNSPECIFIED, UNRECOGNIZED -> CloudEnv.UNKNOWN;
            case PREPROD -> CloudEnv.PREPROD;
            case PROD -> CloudEnv.PROD;
            case HWLAB -> CloudEnv.HWLAB;
            case TESTING -> CloudEnv.TESTING;
            case ISRAEL -> CloudEnv.ISRAEL;
        };
    }

    private static ru.yandex.monitoring.api.v3.NannyEnv toProto(NannyEnv env) {
        return switch (env) {
            case PRODUCTION -> ru.yandex.monitoring.api.v3.NannyEnv.PRODUCTION;
            case ADMIN -> ru.yandex.monitoring.api.v3.NannyEnv.ADMIN;
        };
    }

    private static NannyEnv fromProto(ru.yandex.monitoring.api.v3.NannyEnv env) {
        return switch (env) {
            case NANNY_ENV_UNSPECIFIED, UNRECOGNIZED -> NannyEnv.PRODUCTION;
            case PRODUCTION -> NannyEnv.PRODUCTION;
            case ADMIN -> NannyEnv.ADMIN;
        };
    }

    public static ru.yandex.solomon.core.db.model.Cluster toModel(
        CreateClusterRequest request,
        String login,
        Instant now,
        String folderId)
    {
        var builder = ru.yandex.solomon.core.db.model.Cluster.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        builder.setId(request.getClusterId())
                .setFolderId(folderId)
                .setCreatedAt(now)
                .setUpdatedAt(now)
                .setCreatedBy(login)
                .setUpdatedBy(login)
                .setName(request.getLabelName())
                .setDescription(request.getDescription())
                .setLabels(request.getLabelsMap());

        builder.setHosts(toHostsListModel(request.getHostsList()));
        builder.setHostUrls(toHostUrlsListModel(request.getHostUrlsList()));
        builder.setConductorGroups(toConductorGroupsListModel(request.getConductorGroupsList()));
        builder.setConductorTags(toConductorTagsListModel(request.getConductorTagsList()));
        builder.setNannyGroups(toNannyGroupsListModel(request.getNannyGroupsList()));
        builder.setQloudGroups(toQloudGroupsListModel(request.getQloudGroupsList()));
        builder.setNetworks(toNetworksListModel(request.getNetworksList()));
        builder.setYpClusters(toYpClustersListModel(request.getYpClustersList()));
        builder.setInstanceGroups(toInstanceGroupsListModel(request.getInstanceGroupsList()));
        builder.setCloudDns(toCloudDnsListModel(request.getCloudDnsList()));

        if (request.hasShardSettings()) {
            builder.setShardSettings(ShardSettingsDtoConverter.toModel(request.getShardSettings()));
        }

        return builder.build();
    }

    public static ru.yandex.solomon.core.db.model.Cluster toModel(
        UpdateClusterRequest request,
        String login,
        Instant now,
        int etag,
        String folderId)
    {
        var builder = ru.yandex.solomon.core.db.model.Cluster.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        builder.setId(request.getClusterId())
                .setFolderId(folderId)
                .setUpdatedAt(now)
                .setUpdatedBy(login)
                .setName(request.getLabelName())
                .setDescription(request.getDescription())
                .setVersion(etag)
                .setLabels(request.getLabelsMap());

        builder.setHosts(toHostsListModel(request.getHostsList()));
        builder.setHostUrls(toHostUrlsListModel(request.getHostUrlsList()));
        builder.setConductorGroups(toConductorGroupsListModel(request.getConductorGroupsList()));
        builder.setConductorTags(toConductorTagsListModel(request.getConductorTagsList()));
        builder.setNannyGroups(toNannyGroupsListModel(request.getNannyGroupsList()));
        builder.setQloudGroups(toQloudGroupsListModel(request.getQloudGroupsList()));
        builder.setNetworks(toNetworksListModel(request.getNetworksList()));
        builder.setYpClusters(toYpClustersListModel(request.getYpClustersList()));
        builder.setInstanceGroups(toInstanceGroupsListModel(request.getInstanceGroupsList()));
        builder.setCloudDns(toCloudDnsListModel(request.getCloudDnsList()));

        if (request.hasShardSettings()) {
            builder.setShardSettings(ShardSettingsDtoConverter.toModel(request.getShardSettings()));
        }

        return builder.build();
    }

    private static List<ClusterInstanceGroupConf> toInstanceGroupsListModel(List<InstanceGroup> instanceGroupsList) {
        return instanceGroupsList.stream().map(instanceGroup -> ClusterInstanceGroupConf.of(
                instanceGroup.getInstanceGroupId(),
                instanceGroup.getFolderId(),
                fromLabelsMap(instanceGroup.getLabelsMap())
        )).collect(Collectors.toList());
    }

    private static List<ClusterCloudDnsConf> toCloudDnsListModel(List<CloudDns> cloudDnsList) {
        return cloudDnsList.stream().map(item -> ClusterCloudDnsConf.of(
            fromProto(item.getEnv()),
            item.getName(),
            fromLabelsMap(item.getLabelsMap())
        )).collect(Collectors.toList());
    }

    private static List<ClusterYpConf> toYpClustersListModel(List<YpCluster> ypClustersList) {
        return ypClustersList.stream().map(ypCluster -> ClusterYpConf.of(
                ypCluster.getPodSetId(),
                ypCluster.getEndpointSetId(),
                ypCluster.getCluster(),
                fromLabelsMap(ypCluster.getLabelsMap()),
                ypCluster.getTvmLabel(),
                ypCluster.getYpLabel()
        )).collect(Collectors.toList());
    }

    private static List<ClusterNetworkConf> toNetworksListModel(List<Network> networksList) {
        return networksList.stream().map(network -> ClusterNetworkConf.of(
                network.getNetwork(),
                (int) network.getPort(),
                fromLabelsMap(network.getLabelsMap())
        )).collect(Collectors.toList());
    }

    private static List<ClusterQloudGroupConf> toQloudGroupsListModel(List<QloudGroup> qloudGroupsList) {
        return qloudGroupsList.stream().map(qloudGroup -> ClusterQloudGroupConf.of(
                qloudGroup.getComponent(),
                qloudGroup.getEnvironment(),
                qloudGroup.getApplication(),
                qloudGroup.getProject(),
                qloudGroup.getDeployment(),
                fromLabelsMap(qloudGroup.getLabelsMap())
        )).collect(Collectors.toList());
    }

    private static List<ClusterNannyGroupConf> toNannyGroupsListModel(List<NannyGroup> nannyGroupsList) {
        return nannyGroupsList.stream().map(nannyGroup -> ClusterNannyGroupConf.of(
                nannyGroup.getService(),
                fromLabelsMap(nannyGroup.getLabelsMap()),
                nannyGroup.getUseFetchedPort(),
                fromProto(nannyGroup.getEnv()),
                (int) nannyGroup.getPortShift(),
                nannyGroup.getCfgGroupList().toArray(String[]::new)
        )).collect(Collectors.toList());
    }

    private static List<ClusterConductorTagConf> toConductorTagsListModel(List<ConductorTag> conductorTagsList) {
        return conductorTagsList.stream().map(conductorTag -> ClusterConductorTagConf.of(
                conductorTag.getName(),
                fromLabelsMap(conductorTag.getLabelsMap())
        )).collect(Collectors.toList());
    }

    private static List<ClusterConductorGroupConf> toConductorGroupsListModel(List<ConductorGroup> conductorGroupsList) {
        return conductorGroupsList.stream().map(conductorGroup -> ClusterConductorGroupConf.of(
                conductorGroup.getGroup(),
                fromLabelsMap(conductorGroup.getLabelsMap())
        )).collect(Collectors.toList());
    }

    private static List<ClusterHostUrlConf> toHostUrlsListModel(List<HostUrl> hostUrlsList) {
        return hostUrlsList.stream().map(hostUrl -> ClusterHostUrlConf.of(
                hostUrl.getUrl(),
                fromLabelsMap(hostUrl.getLabelsMap()),
                hostUrl.getIgnorePorts()
        )).collect(Collectors.toList());
    }

    private static List<ClusterHostListConf> toHostsListModel(List<PatternHost> hostsList) {
        return hostsList.stream().map(host -> ClusterHostListConf.of(
                host.getUrlPattern(),
                host.getDc(),
                host.getRanges(),
                fromLabelsMap(host.getLabelsMap())))
                .collect(Collectors.toList());
    }

    private static Map<String, String> toLabelsMap(String[] labels) {
        Map<String, String> labelsMap = new HashMap<>(labels.length);
        for (String label : labels) {
            String[] labelParts = label.split("=");
            if (labelParts.length == 2) {
                labelsMap.put(labelParts[0], labelParts[1]);
            }
        }
        return labelsMap;
    }

    private static String[] fromLabelsMap(Map<String, String> labels) {
        return labels.entrySet().stream()
                .map(entry -> entry.getKey() + "=" + entry.getValue())
                .toArray(String[]::new);
    }
}
