package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Int64Value;
import com.google.protobuf.Timestamp;
import com.google.protobuf.util.Timestamps;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monitoring.api.v3.AlertWidget;
import ru.yandex.monitoring.api.v3.ChartWidget;
import ru.yandex.monitoring.api.v3.CreateDashboardRequest;
import ru.yandex.monitoring.api.v3.CustomParameter;
import ru.yandex.monitoring.api.v3.Dashboard;
import ru.yandex.monitoring.api.v3.Downsampling;
import ru.yandex.monitoring.api.v3.EvaluationStatus;
import ru.yandex.monitoring.api.v3.IframeWidget;
import ru.yandex.monitoring.api.v3.LabelValuesParameter;
import ru.yandex.monitoring.api.v3.ListDashboardsResponse;
import ru.yandex.monitoring.api.v3.Parameter;
import ru.yandex.monitoring.api.v3.Parametrization;
import ru.yandex.monitoring.api.v3.TextParameter;
import ru.yandex.monitoring.api.v3.TextWidget;
import ru.yandex.monitoring.api.v3.TitleWidget;
import ru.yandex.monitoring.api.v3.UpdateDashboardRequest;
import ru.yandex.monitoring.api.v3.Widget;
import ru.yandex.solomon.conf.db3.ydb.Entity;
import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.util.Proto;
import ru.yandex.solomon.ydb.page.TokenBasePage;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class DashboardDtoConverter {

    public static Dashboard fromEntity(Entity entity) {
        if (!entity.hasProto()) {
            throw new UnsupportedOperationException("Can't use old format, tru to convert it");
        }
        return Proto.unpack(entity.getProto(), Dashboard.class).toBuilder()
                .setEtag(String.valueOf(entity.getVersion())) // fix problem from alpha update
                .build();
    }

    public static Entity toEntity(Dashboard dashboard, int version) {
        String parentId = switch (dashboard.getContainerCase()) {
            case PROJECT_ID -> dashboard.getProjectId();
            case FOLDER_ID -> dashboard.getFolderId();
            default -> throw new UnsupportedOperationException("Unsupported container type");
        };
        return Entity.newBuilder()
                .setParentId(parentId)
                .setId(dashboard.getId())
                .setContainerType(dashboard.getContainerCase() == Dashboard.ContainerCase.PROJECT_ID
                        ? ContainerType.PROJECT
                        : ContainerType.FOLDER)
                .setName(dashboard.getTitle())
                .setLocalId(dashboard.getName())
                .setDescription(dashboard.getDescription())
                .setProto(Proto.pack(dashboard))
                .setCreatedAt(Timestamps.toMillis(dashboard.getCreatedAt()))
                .setUpdatedAt(Timestamps.toMillis(dashboard.getModifiedAt()))
                .setCreatedBy(dashboard.getCreatedBy())
                .setUpdatedBy(dashboard.getModifiedBy())
                .setVersion(version)
                .build();
    }

    public static Dashboard toDashboard(UpdateDashboardRequest request, String subjectId, Dashboard previousVersion, String newEtag) {
        Timestamp nowTs = Timestamps.fromMillis(System.currentTimeMillis());
        return previousVersion.toBuilder()
                .clearWidgets()
                .clearLabels()
                .setName(request.getName())
                .setDescription(request.getDescription())
                .addAllWidgets(request.getWidgetsList())
                .setParametrization(request.getParametrization())
                .setModifiedAt(nowTs)
                .setModifiedBy(subjectId)
                .putAllLabels(request.getLabelsMap())
                .setTitle(request.getTitle())
                .setEtag(newEtag)
                .build();
    }

    public static Dashboard toDashboard(CreateDashboardRequest request, String id, String subjectId) {
        var builder = Dashboard.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            case FOLDER_ID -> builder.setFolderId(request.getFolderId());
            default -> throw new UnsupportedOperationException("Unsupported container type");
        }
        Timestamp nowTs = Timestamps.fromMillis(System.currentTimeMillis());

        return builder
                .setId(id)
                .setName(request.getName())
                .setDescription(request.getDescription())
                .addAllWidgets(request.getWidgetsList())
                .setParametrization(request.getParametrization())
                .setCreatedAt(nowTs)
                .setModifiedAt(nowTs)
                .setCreatedBy(subjectId)
                .setModifiedBy(subjectId)
                .putAllLabels(request.getLabelsMap())
                .setTitle(request.getTitle())
                .setEtag("0")
                .build();
    }

    public static ListDashboardsResponse fromEntity(TokenBasePage<Entity> pagedResult) {
        var dashboards = pagedResult.getItems().stream()
                .map(DashboardDtoConverter::fromEntity)
                .collect(Collectors.toList());

        return ListDashboardsResponse.newBuilder()
                .setNextPageToken(pagedResult.getNextPageToken())
                .addAllDashboards(dashboards)
                .build();
    }

    public static Dashboard fromAlphaDashboardModel(ru.yandex.monitoring.v3.Dashboard oldDashboard, String localId) {
        var widgets = oldDashboard.getWidgetsList().stream()
                .map(DashboardDtoConverter::fromWidgetModel)
                .collect(Collectors.toList());

        var parametrization = fromParametrizationModel(oldDashboard.getParametrization());
        return Dashboard.newBuilder()
                .setProjectId(oldDashboard.getProjectId())
                .setId(oldDashboard.getId())
                .setName(localId)
                .setDescription(oldDashboard.getDescription())
                .addAllWidgets(widgets)
                .setParametrization(parametrization)
                .setCreatedAt(oldDashboard.getCreatedAt())
                .setModifiedAt(oldDashboard.getUpdatedAt())
                .setCreatedBy(oldDashboard.getCreatedBy())
                .setModifiedBy(oldDashboard.getUpdatedBy())
                .putAllLabels(oldDashboard.getLabelsMap())
                .setTitle(oldDashboard.getName())
                .setEtag(String.valueOf(oldDashboard.getVersion()))
                .build();
    }

    public static ru.yandex.monitoring.v3.Dashboard toAlphaDashboardModel(Dashboard newDashboard) {
        var widgets = newDashboard.getWidgetsList().stream()
                .map(DashboardDtoConverter::toWidgetModel)
                .collect(Collectors.toList());

        long version = parseVersion(newDashboard.getEtag());

        var parametrization = toParametrizationModel(newDashboard.getParametrization());
        return ru.yandex.monitoring.v3.Dashboard.newBuilder()
                .setProjectId(newDashboard.getProjectId())
                .setId(newDashboard.getId())
                .setName(newDashboard.getTitle())
                .setDescription(newDashboard.getDescription())
                .addAllWidgets(widgets)
                .setParametrization(parametrization)
                .setCreatedAt(newDashboard.getCreatedAt())
                .setUpdatedAt(newDashboard.getModifiedAt())
                .setCreatedBy(newDashboard.getCreatedBy())
                .setUpdatedBy(newDashboard.getModifiedBy())
                .putAllLabels(newDashboard.getLabelsMap())
                .setVersion(version)
                .build();
    }

    public static int parseVersion(String etag) {
        int version = -1;
        try {
            if (!StringUtils.isEmpty(etag)) {
                version = Integer.parseInt(etag);
            }
        } catch (NumberFormatException ignore) {
            throw new BadRequestException("Etag wrong format");
        }
        return version;
    }

    private static Widget fromWidgetModel(ru.yandex.monitoring.v3.Widget widget) {
        var builder = Widget.newBuilder();
        builder.setPosition(fromPositionModel(widget.getPosition()));
        switch (widget.getWidgetCase()) {
            case TEXT -> {
                var text = widget.getText();
                builder.setText(TextWidget.newBuilder()
                        .setText(text.getText()));
            }
            case TITLE -> {
                var title = widget.getTitle();
                builder.setTitle(TitleWidget.newBuilder()
                        .setText(title.getText())
                        .setSizeValue((title.getSizeValue())));
            }
            case CHART -> {
                var chart = widget.getChart();
                builder.setChart(fromChartWidgetModel(chart));
            }
            case ALERT -> {
                var alert = widget.getAlert();
                builder.setAlert(fromAlertWidgetModel(alert));
            }
            case IFRAME -> {
                var iframe = widget.getIframe();
                builder.setIframe(fromIframeWidgetModel(iframe));
            }
            case WIDGET_NOT_SET -> {
            }
        }
        return builder.build();
    }

    private static IframeWidget fromIframeWidgetModel(ru.yandex.monitoring.v3.IframeWidget iframe) {
        return IframeWidget.newBuilder()
                .setUrl(iframe.getUrl())
                .setDescription(iframe.getDescription())
                .setTitle(iframe.getTitle())
                .build();
    }

    private static AlertWidget fromAlertWidgetModel(ru.yandex.monitoring.v3.AlertWidget alert) {
        var evalStatuses =
                alert.getEvalStatusFiltersList().stream()
                        .map(status -> EvaluationStatus.forNumber(status.getNumber()))
                        .collect(Collectors.toList());

        return AlertWidget.newBuilder()
                .setProjectId(alert.getProjectId())
                .setAlertId(alert.getAlertId())
                .setLabelsFilter(alert.getLabelsFilter())
                .addAllEvalStatusFilters(evalStatuses)
                .addAllAnnotationKeys(alert.getAnnotationKeysList())
                .setDescription(alert.getDescription())
                .setTitle(alert.getTitle())
                .build();
    }

    private static ru.yandex.monitoring.v3.Widget toWidgetModel(Widget widget) {
        var builder = ru.yandex.monitoring.v3.Widget.newBuilder();
        builder.setPosition(toPositionModel(widget.getPosition()));
        switch (widget.getWidgetCase()) {
            case TEXT -> {
                var text = widget.getText();
                builder.setText(ru.yandex.monitoring.v3.TextWidget.newBuilder()
                        .setText(text.getText()));
            }
            case TITLE -> {
                var title = widget.getTitle();
                builder.setTitle(ru.yandex.monitoring.v3.TitleWidget.newBuilder()
                        .setText(title.getText())
                        .setSizeValue((title.getSizeValue())));
            }
            case CHART -> {
                var chart = widget.getChart();
                builder.setChart(toChartWidgetModel(chart));
            }
            case ALERT -> {
                var alert = widget.getAlert();
                builder.setAlert(toAlertWidgetModel(alert));
            }
            case IFRAME -> {
                var iframe = widget.getIframe();
                builder.setIframe(toIframeWidgetModel(iframe));
            }
            case WIDGET_NOT_SET -> {
            }
        }
        return builder.build();
    }

    private static ru.yandex.monitoring.v3.IframeWidget toIframeWidgetModel(IframeWidget iframe) {
        return ru.yandex.monitoring.v3.IframeWidget.newBuilder()
                .setUrl(iframe.getUrl())
                .setDescription(iframe.getDescription())
                .setTitle(iframe.getTitle())
                .build();
    }

    private static ru.yandex.monitoring.v3.AlertWidget toAlertWidgetModel(AlertWidget alert) {
        var evalStatuses =
                alert.getEvalStatusFiltersList().stream()
                        .map(status -> ru.yandex.monitoring.v3.AlertEvalStatus.forNumber(status.getNumber()))
                        .collect(Collectors.toList());

        return ru.yandex.monitoring.v3.AlertWidget.newBuilder()
                .setProjectId(alert.getProjectId())
                .setAlertId(alert.getAlertId())
                .setLabelsFilter(alert.getLabelsFilter())
                .addAllEvalStatusFilters(evalStatuses)
                .addAllAnnotationKeys(alert.getAnnotationKeysList())
                .setDescription(alert.getDescription())
                .setTitle(alert.getTitle())
                .build();
    }

    private static Widget.LayoutPosition fromPositionModel(ru.yandex.monitoring.v3.Widget.LayoutPosition position) {
        return Widget.LayoutPosition.newBuilder()
                .setX(position.getX())
                .setY(position.getY())
                .setW(position.getW())
                .setH(position.getH())
                .build();
    }

    private static ru.yandex.monitoring.v3.Widget.LayoutPosition toPositionModel(Widget.LayoutPosition position) {
        return ru.yandex.monitoring.v3.Widget.LayoutPosition.newBuilder()
                .setX(position.getX())
                .setY(position.getY())
                .setW(position.getW())
                .setH(position.getH())
                .build();
    }

    private static ChartWidget fromChartWidgetModel(ru.yandex.monitoring.v3.ChartWidget chart) {
        var builder = ChartWidget.newBuilder()
                .setId(chart.getId())
                .setTitle(chart.getTitle())
                .setDescription(chart.getDescription())
                .setDisplayLegend(chart.getDisplayLegend())
                .setFreezeValue(chart.getFreezeValue())
                .addAllSeriesOverrides(fromSeriesOverridesModelList(chart.getSeriesOverridesList()));

        if (chart.hasQueries()) {
                builder.setQueries(fromQueriesModel(chart.getQueries()));
        }
        if (chart.hasVisualizationSettings()) {
                builder.setVisualizationSettings(fromVisualizationSettingsModel(chart.getVisualizationSettings()));
        }
        if (chart.hasNameHidingSettings()) {
                builder.setNameHidingSettings(fromNameHidingSettingsModel(chart.getNameHidingSettings()));
        }

        return builder.build();
    }

    private static ru.yandex.monitoring.v3.ChartWidget toChartWidgetModel(ChartWidget chart) {
        var builder = ru.yandex.monitoring.v3.ChartWidget.newBuilder()
                .setId(chart.getId())
                .setTitle(chart.getTitle())
                .setDescription(chart.getDescription())
                .setDisplayLegend(chart.getDisplayLegend())
                .setFreezeValue(chart.getFreezeValue())
                .addAllSeriesOverrides(toSeriesOverridesModelList(chart.getSeriesOverridesList()));

        if (chart.hasQueries()) {
            builder.setQueries(toQueriesModel(chart.getQueries()));
        }
        if (chart.hasVisualizationSettings()) {
            builder.setVisualizationSettings(toVisualizationSettingsModel(chart.getVisualizationSettings()));
        }
        if (chart.hasNameHidingSettings()) {
            builder.setNameHidingSettings(toNameHidingSettingsModel(chart.getNameHidingSettings()));
        }

        return builder.build();
    }

    private static List<ChartWidget.SeriesOverrides> fromSeriesOverridesModelList(List<ru.yandex.monitoring.v3.ChartWidget.SeriesOverrides> seriesOverridesList) {
        return seriesOverridesList.stream()
                .map(DashboardDtoConverter::fromSeriesOverridesModel)
                .collect(Collectors.toList());
    }

    private static List<ru.yandex.monitoring.v3.ChartWidget.SeriesOverrides> toSeriesOverridesModelList(List<ChartWidget.SeriesOverrides> seriesOverridesList) {
        return seriesOverridesList.stream()
                .map(DashboardDtoConverter::toSeriesOverridesModel)
                .collect(Collectors.toList());
    }

    private static ChartWidget.SeriesOverrides fromSeriesOverridesModel(ru.yandex.monitoring.v3.ChartWidget.SeriesOverrides seriesOverrides) {
        var builder =  ChartWidget.SeriesOverrides.newBuilder()
                .setSettings(fromSeriesOverridesSettingsModel(seriesOverrides.getSettings()));
        switch (seriesOverrides.getTypeCase()) {
            case NAME -> builder.setName(seriesOverrides.getName());
            case TARGET_INDEX -> builder.setTargetIndex(seriesOverrides.getTargetIndex());
        }
        return builder.build();
    }

    private static ru.yandex.monitoring.v3.ChartWidget.SeriesOverrides toSeriesOverridesModel(ChartWidget.SeriesOverrides seriesOverrides) {
        var builder =  ru.yandex.monitoring.v3.ChartWidget.SeriesOverrides.newBuilder()
                .setSettings(toSeriesOverridesSettingsModel(seriesOverrides.getSettings()));
        switch (seriesOverrides.getTypeCase()) {
            case NAME -> builder.setName(seriesOverrides.getName());
            case TARGET_INDEX -> builder.setTargetIndex(seriesOverrides.getTargetIndex());
        }
        return builder.build();
    }

    private static ChartWidget.SeriesOverrides.SeriesOverrideSettings fromSeriesOverridesSettingsModel(ru.yandex.monitoring.v3.ChartWidget.SeriesOverrides.SeriesOverrideSettings settings) {
        return ChartWidget.SeriesOverrides.SeriesOverrideSettings.newBuilder()
                .setName(settings.getName())
                .setColor(settings.getColor())
                .setTypeValue(settings.getTypeValue())
                .setStackName(settings.getStackName())
                .setGrowDown(settings.getGrowDown())
                .setYaxisPositionValue(settings.getYaxisPositionValue())
                .build();
    }

    private static ru.yandex.monitoring.v3.ChartWidget.SeriesOverrides.SeriesOverrideSettings toSeriesOverridesSettingsModel(ChartWidget.SeriesOverrides.SeriesOverrideSettings settings) {
        return ru.yandex.monitoring.v3.ChartWidget.SeriesOverrides.SeriesOverrideSettings.newBuilder()
                .setName(settings.getName())
                .setColor(settings.getColor())
                .setTypeValue(settings.getTypeValue())
                .setStackName(settings.getStackName())
                .setGrowDown(settings.getGrowDown())
                .setYaxisPositionValue(settings.getYaxisPositionValue())
                .build();
    }

    private static ChartWidget.NameHidingSettings fromNameHidingSettingsModel(ru.yandex.monitoring.v3.ChartWidget.NameHidingSettings nameHidingSettings) {
        return ChartWidget.NameHidingSettings.newBuilder()
                .addAllNames(nameHidingSettings.getNamesList())
                .setPositive(nameHidingSettings.getPositive())
                .build();
    }

    private static ru.yandex.monitoring.v3.ChartWidget.NameHidingSettings toNameHidingSettingsModel(ChartWidget.NameHidingSettings nameHidingSettings) {
        return ru.yandex.monitoring.v3.ChartWidget.NameHidingSettings.newBuilder()
                .addAllNames(nameHidingSettings.getNamesList())
                .setPositive(nameHidingSettings.getPositive())
                .build();
    }

    private static ChartWidget.VisualizationSettings fromVisualizationSettingsModel(ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings visualizationSettings) {
        var builder = ChartWidget.VisualizationSettings.newBuilder()
                .setTypeValue(visualizationSettings.getTypeValue())
                .setNormalize(visualizationSettings.getNormalize())
                .setInterpolateValue(visualizationSettings.getInterpolateValue())
                .setAggregationValue(visualizationSettings.getAggregationValue())
                .setShowLabels(visualizationSettings.getShowLabels())
                .setTitle(visualizationSettings.getTitle());

        if (visualizationSettings.hasColorSchemeSettings()) {
            builder.setColorSchemeSettings(fromColorSchemeSettingsModel(visualizationSettings.getColorSchemeSettings()));
        }
        if (visualizationSettings.hasHeatmapSettings()) {
            builder.setHeatmapSettings(fromHeatmapSettingsModel(visualizationSettings.getHeatmapSettings()));
        }
        if (visualizationSettings.hasYaxisSettings()) {
            builder.setYaxisSettings(fromYaxisSettingsModel(visualizationSettings.getYaxisSettings()));
        }

        return builder.build();
    }

    private static ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings toVisualizationSettingsModel(ChartWidget.VisualizationSettings visualizationSettings) {
        var builder = ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.newBuilder()
                .setTypeValue(visualizationSettings.getTypeValue())
                .setNormalize(visualizationSettings.getNormalize())
                .setInterpolateValue(visualizationSettings.getInterpolateValue())
                .setAggregationValue(visualizationSettings.getAggregationValue())
                .setShowLabels(visualizationSettings.getShowLabels())
                .setTitle(visualizationSettings.getTitle());

        if (visualizationSettings.hasColorSchemeSettings()) {
            builder.setColorSchemeSettings(toColorSchemeSettingsModel(visualizationSettings.getColorSchemeSettings()));
        }

        if (visualizationSettings.hasHeatmapSettings()) {
            builder.setHeatmapSettings(toHeatmapSettingsModel(visualizationSettings.getHeatmapSettings()));
        }

        if (visualizationSettings.hasYaxisSettings()) {
            builder.setYaxisSettings(toYaxisSettingsModel(visualizationSettings.getYaxisSettings()));
        }

        return builder.build();
    }

    private static ChartWidget.VisualizationSettings.YaxisSettings fromYaxisSettingsModel(ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.YaxisSettings yaxisSettings) {
        var builder = ChartWidget.VisualizationSettings.YaxisSettings.newBuilder();
        if (yaxisSettings.hasLeft()) {
            builder.setLeft(fromYaxisModel(yaxisSettings.getLeft()));
        }
        if (yaxisSettings.hasRight()) {
            builder.setRight(fromYaxisModel(yaxisSettings.getRight()));
        }
        return builder.build();
    }

    private static ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.YaxisSettings toYaxisSettingsModel(ChartWidget.VisualizationSettings.YaxisSettings yaxisSettings) {
        var builder = ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.YaxisSettings.newBuilder();
        if (yaxisSettings.hasLeft()) {
            builder.setLeft(toYaxisModel(yaxisSettings.getLeft()));
        }
        if (yaxisSettings.hasRight()) {
            builder.setRight(toYaxisModel(yaxisSettings.getRight()));
        }
        return builder.build();
    }

    private static ChartWidget.VisualizationSettings.Yaxis fromYaxisModel(ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.Yaxis yaxis) {
        var builder = ChartWidget.VisualizationSettings.Yaxis.newBuilder()
                .setTypeValue(yaxis.getTypeValue())
                .setTitle(yaxis.getTitle())
                .setMin(yaxis.getMin())
                .setMax(yaxis.getMax())
                .setUnitFormatValue(yaxis.getUnitFormatValue());

        if (!yaxis.getZeroPrecision()) {
            builder.setPrecision(Int64Value.newBuilder().setValue(yaxis.getPrecision()).build());
        }

        return builder.build();
    }

    private static ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.Yaxis toYaxisModel(ChartWidget.VisualizationSettings.Yaxis yaxis) {
        var builder = ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.Yaxis.newBuilder()
                .setTypeValue(yaxis.getTypeValue())
                .setTitle(yaxis.getTitle())
                .setMin(yaxis.getMin())
                .setMax(yaxis.getMax())
                .setUnitFormatValue(yaxis.getUnitFormatValue());

        if (yaxis.hasPrecision()) {
            builder.setPrecision((int) yaxis.getPrecision().getValue());
        } else {
            builder.setPrecision(0);
            builder.setZeroPrecision(true);
        }
        return builder.build();
    }

    private static ChartWidget.VisualizationSettings.HeatmapSettings fromHeatmapSettingsModel(ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.HeatmapSettings heatmapSettings) {
        return ChartWidget.VisualizationSettings.HeatmapSettings.newBuilder()
                .setGreenValue(heatmapSettings.getGreenValue())
                .setYellowValue(heatmapSettings.getYellowValue())
                .setRedValue(heatmapSettings.getRedValue())
                .setVioletValue(heatmapSettings.getVioletValue())
                .build();
    }

    private static ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.HeatmapSettings toHeatmapSettingsModel(ChartWidget.VisualizationSettings.HeatmapSettings heatmapSettings) {
        return ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.HeatmapSettings.newBuilder()
                .setGreenValue(heatmapSettings.getGreenValue())
                .setYellowValue(heatmapSettings.getYellowValue())
                .setRedValue(heatmapSettings.getRedValue())
                .setVioletValue(heatmapSettings.getVioletValue())
                .build();
    }

    private static ChartWidget.VisualizationSettings.ColorSchemeSettings fromColorSchemeSettingsModel(ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.ColorSchemeSettings colorSchemeSettings) {
        var builder = ChartWidget.VisualizationSettings.ColorSchemeSettings.newBuilder();
        switch (colorSchemeSettings.getSchemeCase()) {
            case AUTO -> builder.setAutomatic(ChartWidget.VisualizationSettings.ColorSchemeSettings.AutomaticColorScheme.getDefaultInstance());
            case STANDARD -> builder.setStandard(ChartWidget.VisualizationSettings.ColorSchemeSettings.StandardColorScheme.getDefaultInstance());
            case GRADIENT -> {
                var gradient = colorSchemeSettings.getGradient();
                builder.setGradient(ChartWidget.VisualizationSettings.ColorSchemeSettings.GradientColorScheme.newBuilder()
                        .setGreenValue(gradient.getGreenValue())
                        .setYellowValue(gradient.getYellowValue())
                        .setRedValue(gradient.getRedValue())
                        .setVioletValue(gradient.getVioletValue()));
            }
        }
        return builder.build();
    }

    private static ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.ColorSchemeSettings toColorSchemeSettingsModel(ChartWidget.VisualizationSettings.ColorSchemeSettings colorSchemeSettings) {
        var builder = ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.ColorSchemeSettings.newBuilder();
        switch (colorSchemeSettings.getSchemeCase()) {
            case AUTOMATIC -> builder.setAuto(ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.ColorSchemeSettings.AutoColorScheme.getDefaultInstance());
            case STANDARD -> builder.setStandard(ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.ColorSchemeSettings.StandardColorScheme.getDefaultInstance());
            case GRADIENT -> {
                var gradient = colorSchemeSettings.getGradient();
                builder.setGradient(ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.ColorSchemeSettings.GradientColorScheme.newBuilder()
                        .setGreenValue(gradient.getGreenValue())
                        .setYellowValue(gradient.getYellowValue())
                        .setRedValue(gradient.getRedValue())
                        .setVioletValue(gradient.getVioletValue()));
            }
        }
        return builder.build();
    }

    private static ChartWidget.Queries fromQueriesModel(ru.yandex.monitoring.v3.ChartWidget.Queries queries) {
        var builder = ChartWidget.Queries.newBuilder()
                .addAllTargets(fromTargetsModel(queries.getTargetsList()));

        if (queries.hasDownsampling()) {
            builder.setDownsampling(fromDownsamplingModel(queries.getDownsampling()));
        }

        return builder.build();
    }

    private static ru.yandex.monitoring.v3.ChartWidget.Queries toQueriesModel(ChartWidget.Queries queries) {
        var builder = ru.yandex.monitoring.v3.ChartWidget.Queries.newBuilder()
                .addAllTargets(toTargetsModel(queries.getTargetsList()));
        if (queries.hasDownsampling()) {
            builder.setDownsampling(toDownsamplingModel(queries.getDownsampling()));
        }
        return builder.build();
    }

    private static Downsampling fromDownsamplingModel(ru.yandex.monitoring.v3.Downsampling downsampling) {
        var builder = Downsampling.newBuilder();
        switch (downsampling.getModeCase()) {
            case MAX_POINTS -> builder.setMaxPoints(downsampling.getMaxPoints());
            case GRID_INTERVAL -> builder.setGridInterval(downsampling.getGridInterval());
            case DISABLED -> builder.setDisabled(downsampling.getDisabled());
        }
        builder.setGridAggregationValue(downsampling.getGridAggregationValue());
        builder.setGapFillingValue(downsampling.getGapFillingValue());
        return builder.build();
    }

    private static ru.yandex.monitoring.v3.Downsampling toDownsamplingModel(Downsampling downsampling) {
        var builder = ru.yandex.monitoring.v3.Downsampling.newBuilder();
        switch (downsampling.getModeCase()) {
            case MAX_POINTS -> builder.setMaxPoints(downsampling.getMaxPoints());
            case GRID_INTERVAL -> builder.setGridInterval(downsampling.getGridInterval());
            case DISABLED -> builder.setDisabled(downsampling.getDisabled());
        }
        builder.setGridAggregationValue(downsampling.getGridAggregationValue());
        builder.setGapFillingValue(downsampling.getGapFillingValue());
        return builder.build();
    }

    private static List<ChartWidget.Queries.Target> fromTargetsModel(List<ru.yandex.monitoring.v3.ChartWidget.Queries.Target> targetsList) {
        return targetsList.stream()
                .map(DashboardDtoConverter::fromTargetModel)
                .collect(Collectors.toList());
    }

    private static List<ru.yandex.monitoring.v3.ChartWidget.Queries.Target> toTargetsModel(List<ChartWidget.Queries.Target> targetsList) {
        return targetsList.stream()
                .map(DashboardDtoConverter::toTargetModel)
                .collect(Collectors.toList());
    }

    private static ChartWidget.Queries.Target fromTargetModel(ru.yandex.monitoring.v3.ChartWidget.Queries.Target target) {
        return ChartWidget.Queries.Target.newBuilder()
                .setQuery(target.getQuery())
                .setTextMode(target.getTextMode())
                .setHidden(target.getHidden())
                .build();
    }

    private static ru.yandex.monitoring.v3.ChartWidget.Queries.Target toTargetModel(ChartWidget.Queries.Target target) {
        return ru.yandex.monitoring.v3.ChartWidget.Queries.Target.newBuilder()
                .setQuery(target.getQuery())
                .setTextMode(target.getTextMode())
                .setHidden(target.getHidden())
                .build();
    }

    private static Parametrization fromParametrizationModel(ru.yandex.monitoring.v3.Parametrization oldParametrization) {
        var parameters = oldParametrization.getParametersList().stream()
                .map(DashboardDtoConverter::fromParameterModel)
                .collect(Collectors.toList());
        return Parametrization.newBuilder()
                .addAllParameters(parameters)
                .setSelectors(oldParametrization.getSelectors())
                .build();
    }

    private static ru.yandex.monitoring.v3.Parametrization toParametrizationModel(Parametrization oldParametrization) {
        var parameters = oldParametrization.getParametersList().stream()
                .map(DashboardDtoConverter::toParameterModel)
                .collect(Collectors.toList());
        return ru.yandex.monitoring.v3.Parametrization.newBuilder()
                .addAllParameters(parameters)
                .setSelectors(oldParametrization.getSelectors())
                .build();
    }

    private static Parameter fromParameterModel(ru.yandex.monitoring.v3.Parameter oldParameter) {
        var builder = Parameter.newBuilder()
                .setName(oldParameter.getName())
                .setHidden(oldParameter.getHidden())
                .setTitle(oldParameter.getTitle());

        switch (oldParameter.getDataCase()) {
            case LABEL_VALUES -> {
                var labelValues = oldParameter.getLabelValues();
                builder.setLabelValues(LabelValuesParameter.newBuilder()
                        .setProjectId(labelValues.getProjectId())
                        .setSelectors(labelValues.getSelectors())
                        .setLabelKey(labelValues.getLabelKey())
                        .setMultiselectable(labelValues.getMultiselectable())
                        .addAllDefaultValues(labelValues.getDefaultValuesList()));
            }
            case CUSTOM -> {
                var custom = oldParameter.getCustom();
                builder.setCustom(CustomParameter.newBuilder()
                        .addAllValues(custom.getValuesList())
                        .addAllDefaultValues(custom.getDefaultValuesList())
                        .setMultiselectable(custom.getMultiselectable())
                        .build());
            }
            case TEXT -> {
                var text = oldParameter.getText();
                builder.setText(TextParameter.newBuilder()
                        .setDefaultValue(text.getDefaultValue())
                        .build());
            }
            default -> throw new IllegalStateException("unknown parameter data type: " + oldParameter.getDataCase());
        }

        return builder.build();
    }

    private static ru.yandex.monitoring.v3.Parameter toParameterModel(Parameter oldParameter) {
        var builder = ru.yandex.monitoring.v3.Parameter.newBuilder()
                .setName(oldParameter.getName())
                .setHidden(oldParameter.getHidden())
                .setTitle(oldParameter.getTitle());

        switch (oldParameter.getDataCase()) {
            case LABEL_VALUES -> {
                var labelValues = oldParameter.getLabelValues();
                builder.setLabelValues(ru.yandex.monitoring.v3.LabelValuesParameter.newBuilder()
                        .setProjectId(labelValues.getProjectId())
                        .setSelectors(labelValues.getSelectors())
                        .setLabelKey(labelValues.getLabelKey())
                        .setMultiselectable(labelValues.getMultiselectable())
                        .addAllDefaultValues(labelValues.getDefaultValuesList()));
            }
            case CUSTOM -> {
                var custom = oldParameter.getCustom();
                builder.setCustom(ru.yandex.monitoring.v3.CustomParameter.newBuilder()
                        .addAllValues(custom.getValuesList())
                        .addAllDefaultValues(custom.getDefaultValuesList())
                        .setMultiselectable(custom.getMultiselectable())
                        .build());
            }
            case TEXT -> {
                var text = oldParameter.getText();
                builder.setText(ru.yandex.monitoring.v3.TextParameter.newBuilder()
                        .setDefaultValue(text.getDefaultValue())
                        .build());
            }
            default -> throw new IllegalStateException("unknown parameter data type: " + oldParameter.getDataCase());
        }

        return builder.build();
    }
}
