package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Timestamp;
import com.google.protobuf.util.Durations;
import com.google.protobuf.util.Timestamps;

import ru.yandex.monitoring.api.v3.DoubleValues;
import ru.yandex.monitoring.api.v3.Downsampling;
import ru.yandex.monitoring.api.v3.Int64Values;
import ru.yandex.monitoring.api.v3.MetricType;
import ru.yandex.monitoring.api.v3.ReadMetricsDataRequest;
import ru.yandex.monitoring.api.v3.ReadMetricsDataResponse;
import ru.yandex.monitoring.api.v3.Timeseries;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.compile.DeprOpts;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.gateway.data.DataRequest;
import ru.yandex.solomon.gateway.data.DataResponse;
import ru.yandex.solomon.gateway.data.DownsamplingOptions;
import ru.yandex.solomon.gateway.data.DownsamplingType;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayListOrView;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class DataReadDtoConverter {

    public static DataRequest toModel(ReadMetricsDataRequest request, Instant deadline, String subjectId) {
        var builder = DataRequest.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        DownsamplingOptions downsamplingOptions = toDownsamplingOptions(request.getDownsampling());

        Interval interval = Interval.seconds(
                Timestamps.toSeconds(request.getFromTime()),
                Timestamps.toSeconds(request.getToTime()));

        boolean useNewFormat = !request.getUseOldFormat();

        return builder.setProgram(request.getQuery())
                .setUseNewFormat(useNewFormat)
                .setDownsampling(downsamplingOptions)
                .setInterval(interval)
                .setForceCluster(request.getForceZoneId())
                .setDeadline(deadline)
                .setProducer(RequestProducer.STAFF)
                .setDeprOpts(DeprOpts.DATA_API_V3)
                .setSubjectId(subjectId)
                .build();
    }

    @Nonnull
    private static DownsamplingOptions toDownsamplingOptions(Downsampling downsampling) {
        var builder = DownsamplingOptions.newBuilder();

        switch (downsampling.getModeCase()) {
            case MAX_POINTS -> {
                builder.setDownsamplingType(DownsamplingType.BY_POINTS);
                builder.setPoints((int) downsampling.getMaxPoints());
            }
            case GRID_INTERVAL -> {
                builder.setDownsamplingType(DownsamplingType.BY_INTERVAL);
                builder.setGridMillis((int) downsampling.getGridInterval());
            }
            case DISABLED -> builder.setDownsamplingType(DownsamplingType.OFF);
            case MODE_NOT_SET -> {
            }
        }

        builder.setDownsamplingAggr(toDownsamplingAggrModel(downsampling.getGridAggregation()));
        builder.setDownsamplingFill(toDownsamplingFillModel(downsampling.getGapFilling()));

        return builder.build();
    }

    private static Aggregation toDownsamplingAggrModel(Downsampling.GridAggregation gridAggregation) {
        return switch (gridAggregation) {
            case GRID_AGGREGATION_UNSPECIFIED, UNRECOGNIZED -> Aggregation.DEFAULT_AGGREGATION;
            case GRID_AGGREGATION_MAX -> Aggregation.MAX;
            case GRID_AGGREGATION_MIN -> Aggregation.MIN;
            case GRID_AGGREGATION_SUM -> Aggregation.SUM;
            case GRID_AGGREGATION_AVG -> Aggregation.AVG;
            case GRID_AGGREGATION_LAST -> Aggregation.LAST;
            case GRID_AGGREGATION_COUNT -> Aggregation.COUNT;
        };
    }

    private static OperationDownsampling.FillOption toDownsamplingFillModel(Downsampling.GapFilling gapFilling) {
        return switch (gapFilling) {
            case GAP_FILLING_UNSPECIFIED, GAP_FILLING_NULL, UNRECOGNIZED -> OperationDownsampling.FillOption.NULL;
            case GAP_FILLING_NONE -> OperationDownsampling.FillOption.NONE;
            case GAP_FILLING_PREVIOUS -> OperationDownsampling.FillOption.PREVIOUS;
        };
    }

    public static ReadMetricsDataResponse fromModel(DataResponse response) {
        ReadMetricsDataResponse.Builder builder = ReadMetricsDataResponse.newBuilder();

        SelValue selValue = response.getEvalResult();

        SelType type = selValue.type();

        if (type.isDouble()) {
            builder.setScalar(selValue.castToScalar().getValue());
        } else if (type.isBoolean()) {
            builder.setBool(selValue.castToBoolean().getValue());
        } else if (type.isString()) {
            builder.setString(selValue.castToString().getValue());
        } else if (type.isDuration()) {
            Duration duration = selValue.castToDuration().getDuration();
            builder.setDuration(Durations.fromMillis(duration.toMillis()));
        } else if (type.isGraphData()) {
            builder.setTimeseries(fromGraphDataModel(selValue));
        } else if (type.isGraphDataVector()) {
            List<Timeseries> timeseriesVector = Arrays.stream(selValue.castToVector().valueArray())
                    .map(DataReadDtoConverter::fromGraphDataModel)
                    .collect(Collectors.toList());
            builder.setTimeseriesVector(ReadMetricsDataResponse.TimeseriesVector.newBuilder()
                    .addAllValues(timeseriesVector)
                    .build());
        } else {
            // Interval, lambdas, objects and complex vectors isn't supported
            throw new BadRequestException("unsupported result type: " + type);
        }

        return builder.build();
    }

    private static Timeseries fromGraphDataModel(SelValue selValue) {
        Timeseries.Builder builder = Timeseries.newBuilder();

        NamedGraphData namedGraphData = selValue.castToGraphData().getNamedGraphData();
        MetricType type = fromMetricType(namedGraphData.getType());
        Map<String, String> labelsWithoutProject = namedGraphData.getLabels().removeByKey(LabelKeys.PROJECT).toMap();

        builder.setAlias(namedGraphData.getAlias());
        builder.setName(namedGraphData.getMetricName());
        builder.putAllLabels(labelsWithoutProject);
        builder.setType(type);

        var graphData = namedGraphData.getAggrGraphDataArrayList();

        switch (type) {
            case METRIC_TYPE_UNSPECIFIED, UNRECOGNIZED -> {
            }
            case DGAUGE, RATE -> fillDoubleTimeseries(builder, graphData);
            case IGAUGE, COUNTER -> fillInt64Timeseries(builder, graphData);
        }

        return builder.build();
    }

    private static void fillInt64Timeseries(
            Timeseries.Builder timeseriesBuilder,
            AggrGraphDataArrayListOrView graphData) {
        AggrGraphDataListIterator iterator = graphData.iterator();

        RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();
        Int64Values.Builder valuesBuilder = Int64Values.newBuilder();

        try {
            while (iterator.next(point)) {
                timeseriesBuilder.addTimestamps(toTimestamp(point.getTsMillis()));
                valuesBuilder.addValues(point.longValue);
            }
        } finally {
            point.recycle();
        }

        timeseriesBuilder.setInt64Values(valuesBuilder);
    }

    private static void fillDoubleTimeseries(
            Timeseries.Builder timeseriesBuilder,
            AggrGraphDataArrayListOrView graphData) {
        AggrGraphDataListIterator iterator = graphData.iterator();

        RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();

        DoubleValues.Builder valuesBuilder = DoubleValues.newBuilder();


        try {
            while (iterator.next(point)) {
                timeseriesBuilder.addTimestamps(toTimestamp(point.tsMillis));
                valuesBuilder.addValues(point.getValueDivided());
            }
        } finally {
            point.recycle();
        }

        timeseriesBuilder.setDoubleValues(valuesBuilder);
    }

    private static Timestamp toTimestamp(long tsMillis) {
        return Timestamps.fromSeconds(tsMillis / 1000);
    }

    private static MetricType fromMetricType(ru.yandex.monlib.metrics.MetricType type) {
        switch (type) {
            case UNKNOWN, LOG_HISTOGRAM, DSUMMARY, ISUMMARY, HIST_RATE, HIST -> {
                return MetricType.METRIC_TYPE_UNSPECIFIED;
            }
            case DGAUGE -> {
                return MetricType.DGAUGE;
            }
            case IGAUGE -> {
                return MetricType.IGAUGE;
            }
            case COUNTER -> {
                return MetricType.COUNTER;
            }
            case RATE -> {
                return MetricType.RATE;
            }
            default -> throw new IllegalStateException("unknown metric type: " + type);
        }
    }
}
