package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Any;
import com.google.protobuf.Empty;
import com.google.protobuf.util.Timestamps;
import io.grpc.Status;
import io.grpc.protobuf.StatusProto;

import ru.yandex.monitoring.api.v3.DeleteMetricsMetadata;
import ru.yandex.monitoring.api.v3.DeleteMetricsOperation;
import ru.yandex.monitoring.api.v3.Operation;
import ru.yandex.solomon.gateway.operations.LongRunningOperation;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationStatus;

import static ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationManager.unpackData;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public final class DeleteMetricsOperationConverter {

    private DeleteMetricsOperationConverter() {
    }

    public static DeleteMetricsOperation toProto(LongRunningOperation operation) {
        var data = unpackData(operation);

        var status = DeleteMetricsOperationStatus.fromValue(operation.status());
        var statusMessage = data.getStatusMessage();
        if (status == DeleteMetricsOperationStatus.METRICS_HAVE_RECENT_WRITES) {
            statusMessage = failedDueToRecentWritesDescription(statusMessage);
        }

        return DeleteMetricsOperation.newBuilder()
            .setId(operation.operationId())
            .setProjectId(operation.containerId())
            .setDescription(operation.description())
            .setCreatedAt(Timestamps.fromMillis(operation.createdAt()))
            .setCreatedBy(operation.createdBy())
            .setModifiedAt(Timestamps.fromMillis(operation.updatedAt()))
            .setCancelledAt(Timestamps.fromMillis(data.getCancelledAt()))
            .setCancelledBy(data.getCancelledBy())
            .setStatus(toProtoStatus(status))
            .setSelectors(data.getSelectors())
            .setPermanentDeletionAt(Timestamps.fromMillis(data.getPermanentDeletionAt()))
            .setProgressPercentage(data.getProgressPercentage())
            .setEstimatedMetricsCount(data.getEstimatedMetricsCount())
            .setStatusMessage(statusMessage)
            .setPermanentlyDeletedMetricsCount(data.getPermanentlyDeletedMetricsCount())
            .build();
    }

    public static Operation toOperation(LongRunningOperation operation) {
        var data = unpackData(operation);

        var meta = DeleteMetricsMetadata.newBuilder()
            .setProjectId(operation.containerId())
            .setSelectors(data.getSelectors())
            .setPermanentDeletionAt(Timestamps.fromMillis(data.getPermanentDeletionAt()))
            .setProgressPercentage(data.getProgressPercentage())
            .build();

        var status = DeleteMetricsOperationStatus.fromValue(operation.status());
        var done = status.isTerminal();

        var builder = Operation.newBuilder()
            .setId(operation.operationId())
            .setDescription(operation.description())
            .setCreatedAt(Timestamps.fromNanos(operation.createdAt()))
            .setCreatedBy(operation.createdBy())
            .setModifiedAt(Timestamps.fromNanos(operation.updatedAt()))
            .setDone(done)
            .setMetadata(Any.pack(meta));

        switch (status) {
            case COMPLETED -> builder.setResponse(Any.pack(Empty.getDefaultInstance()));
            case CANCELLED -> builder.setError(StatusProto.fromStatusAndTrailers(Status.CANCELLED, null));
            case FAILED -> builder.setError(StatusProto.fromStatusAndTrailers(Status.INTERNAL, null));
            case METRICS_HAVE_RECENT_WRITES -> builder.setError(
                StatusProto.fromStatusAndTrailers(failedDueToRecentWrites(data.getStatusMessage()), null));
            default -> { /* no-op */ }
        }

        return builder.build();
    }

    private static DeleteMetricsOperation.Status toProtoStatus(DeleteMetricsOperationStatus status) {
        return switch (status) {
            case DELETING -> DeleteMetricsOperation.Status.DELETING;
            case WAITING_FOR_PERMANENT_DELETION -> DeleteMetricsOperation.Status.WAITING_FOR_PERMANENT_DELETION;
            case DELETING_PERMANENTLY -> DeleteMetricsOperation.Status.DELETING_PERMANENTLY;
            case COMPLETED -> DeleteMetricsOperation.Status.COMPLETED;
            case CANCELLING -> DeleteMetricsOperation.Status.CANCELLING;
            case CANCELLED -> DeleteMetricsOperation.Status.CANCELLED;
            case FAILED -> DeleteMetricsOperation.Status.FAILED;
            case METRICS_HAVE_RECENT_WRITES -> DeleteMetricsOperation.Status.METRICS_HAVE_RECENT_WRITES;
            default -> DeleteMetricsOperation.Status.STATUS_UNSPECIFIED;
        };
    }

    private static Status failedDueToRecentWrites(String reason) {
        return Status.FAILED_PRECONDITION.withDescription(failedDueToRecentWritesDescription(reason));
    }

    private static String failedDueToRecentWritesDescription(String reason) {
        return "Metrics deletion was rejected because some of the selected metrics "
            + "contain data points written during the last hour. "
            + "If you would like to proceed, you may stop the writing and retry after one hour. "
            + "Or you may exclude actively used metrics from the provided selectors and retry immediately. "
            + reason;
    }
}
