package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import ru.yandex.monitoring.api.v3.DataType;
import ru.yandex.monitoring.api.v3.ExpressionObject;
import ru.yandex.monitoring.api.v3.FunctionCategory;
import ru.yandex.monitoring.api.v3.ListFunctionsResponse;
import ru.yandex.monitoring.api.v3.ParseExpressionResponse;
import ru.yandex.solomon.expression.analytics.ProgramWithReturn;
import ru.yandex.solomon.expression.ast.Ast;
import ru.yandex.solomon.expression.ast.AstStatement;
import ru.yandex.solomon.expression.ast.serialization.AstMappingContext;
import ru.yandex.solomon.expression.exceptions.ParserException;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncArgument;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ExpressionDtoConverter {
    private static final Map<SelType, DataType> TYPE_MAPPING =
            Map.ofEntries(
                    Map.entry(SelTypes.BOOLEAN, DataType.BOOL),
                    Map.entry(SelTypes.DOUBLE, DataType.SCALAR),
                    Map.entry(SelTypes.STRING, DataType.STRING),
                    Map.entry(SelTypes.GRAPH_DATA, DataType.TIMESERIES),
                    Map.entry(SelTypes.DURATION, DataType.DURATION),

                    Map.entry(SelTypes.DOUBLE_VECTOR, DataType.SCALAR_VECTOR),
                    Map.entry(SelTypes.STRING_VECTOR, DataType.STRING_VECTOR),
                    Map.entry(SelTypes.GRAPH_DATA_VECTOR, DataType.TIMESERIES_VECTOR));

    public static SelVersion parseVersion(long number) {
        if (number == 0) {
            return SelVersion.CURRENT;
        }

        for (SelVersion version : SelVersion.values()) {
            if (version.getNumber() == number) {
                return version;
            }
        }

        return SelVersion.CURRENT;
    }

    @Nullable
    public static SelFuncCategory parseFuncCategory(FunctionCategory category) {
        return switch (category) {
            case FUNCTION_CATEGORY_UNSPECIFIED, UNRECOGNIZED -> null;
            case AGGREGATION -> SelFuncCategory.AGGREGATION;
            case COMBINE -> SelFuncCategory.COMBINE;
            case RANK -> SelFuncCategory.RANK;
            case TRANSFORMATION -> SelFuncCategory.TRANSFORMATION;
            case PREDICT -> SelFuncCategory.PREDICT;
            case OTHER -> SelFuncCategory.OTHER;
            case DEPRECATED -> SelFuncCategory.DEPRECATED;
            case INTERNAL -> SelFuncCategory.INTERNAL;
        };
    }

    @Nullable
    public static ListFunctionsResponse.Function fromModel(SelFunc func) {
        List<ListFunctionsResponse.Argument> args = func.getArgs().stream()
                .map(ExpressionDtoConverter::toModel)
                .collect(Collectors.toList());

        DataType returnType = toModel(func.getReturnType());

        if (returnType == null || args.stream().anyMatch(Objects::isNull)) {
            return null;
        }

        return ListFunctionsResponse.Function.newBuilder()
                .setName(func.getName())
                .setHelp(Nullables.orEmpty(func.getHelp()))
                .setCategory(toModel(func.getCategory()))
                .addAllArgs(args)
                .setReturnType(returnType)
                .build();
    }


    @Nullable
    private static DataType toModel(SelType returnType) {
        return TYPE_MAPPING.get(returnType);
    }

    @Nullable
    private static ListFunctionsResponse.Argument toModel(SelFuncArgument arg) {
        DataType type = toModel(arg.getType());

        if (type == null) {
            return null;
        }

        return ListFunctionsResponse.Argument.newBuilder()
                .setName(arg.getName())
                .setType(type)
                .setHelp(Nullables.orEmpty(arg.getHelp()))
                .addAllAvailableValues(arg.getAvailableValues())
                .build();
    }

    private static FunctionCategory toModel(SelFuncCategory category) {
        return switch (category) {
            case AGGREGATION -> FunctionCategory.AGGREGATION;
            case COMBINE -> FunctionCategory.COMBINE;
            case RANK -> FunctionCategory.RANK;
            case TRANSFORMATION -> FunctionCategory.TRANSFORMATION;
            case PREDICT -> FunctionCategory.PREDICT;
            case OTHER -> FunctionCategory.OTHER;
            case DEPRECATED -> FunctionCategory.DEPRECATED;
            case INTERNAL -> FunctionCategory.INTERNAL;
        };
    }

    public static ParseExpressionResponse fromModel(ProgramWithReturn programWithReturn, boolean needPosition) {
        List<ExpressionObject> variables = new ArrayList<>(programWithReturn.getStatements().size());
        for (AstStatement statement : programWithReturn.getStatements()) {
            var objectNode = render(statement, needPosition);
            variables.add(fromModel(objectNode));
        }
        return ParseExpressionResponse.newBuilder()
                .setProgramAst(ParseExpressionResponse.ProgramAst.newBuilder()
                        .setExpression(fromModel(render(programWithReturn.getExpr(), needPosition)))
                        .addAllVariables(variables)
                        .build())
                .build();
    }

    private static ExpressionObject fromModel(ObjectNode objectNode) {
        var builder = ExpressionObject.newBuilder();
        var iterator = objectNode.fieldNames();
        while (iterator.hasNext()) {
            String next = iterator.next();
            JsonNode jsonNode = objectNode.get(next);
            ExpressionObject.Node node = fromModel(jsonNode);
            if (node == null) {
                continue;
            }
            builder.putProperties(next, node);
        }
        return builder.build();
    }

    private static ExpressionObject.Node fromModel(JsonNode jsonNode) {
        if (jsonNode.isNull()) {
            return null;
        } else if (jsonNode.isTextual()) {
            return ExpressionObject.Node.newBuilder()
                    .setStringValue(jsonNode.asText())
                    .build();
        } else if (jsonNode.isDouble()) {
            return ExpressionObject.Node.newBuilder()
                    .setDoubleValue(jsonNode.asDouble())
                    .build();
        } else if (jsonNode.isObject()) {
            return ExpressionObject.Node.newBuilder()
                    .setExpressionObject(fromModel((ObjectNode) jsonNode))
                    .build();
        } else if (jsonNode.isArray()) {
            if (jsonNode.size() == 0) {
                return null;
            }
            if (jsonNode.get(0).isInt()) {
                var nodeBuilder = ExpressionObject.Node.IntArrayNode.newBuilder();
                for (JsonNode item : jsonNode) {
                    nodeBuilder.addValues(item.asInt());
                }
                return ExpressionObject.Node.newBuilder()
                        .setIntArrayNode(nodeBuilder.build())
                        .build();
            } else if (jsonNode.get(0).isObject()) {
                var nodeBuilder = ExpressionObject.Node.ExpressionObjectArrayNode.newBuilder();
                for (JsonNode item : jsonNode) {
                    nodeBuilder.addValues(fromModel((ObjectNode) item));
                }
                return ExpressionObject.Node.newBuilder()
                        .setExpressionObjectArrayNode(nodeBuilder.build())
                        .build();
            } else if (jsonNode.get(0).isTextual()) {
                var nodeBuilder = ExpressionObject.Node.StringArrayNode.newBuilder();
                for (JsonNode item : jsonNode) {
                    nodeBuilder.addValues(item.asText());
                }
                return ExpressionObject.Node.newBuilder()
                        .setStringArrayNode(nodeBuilder.build())
                        .build();
            }
        }
        return ExpressionObject.Node.newBuilder()
                .setStringValue(jsonNode.asText())
                .build();
    }

    private static ObjectNode render(@Nonnull Ast ast, boolean includeRanges) {
        return new AstMappingContext(includeRanges).render(ast);
    }

    public static ParseExpressionResponse fromModel(ParserException e) {
        return ParseExpressionResponse.newBuilder()
                .setError(ParseExpressionResponse.Error.newBuilder()
                        .setMessage(e.getErrorMessage())
                        .setStartLine(e.getRange().getBegin().getLine())
                        .setStartColumn(e.getRange().getBegin().getColumn())
                        .setStartOffset(e.getRange().getBegin().getOffset())
                        .setEndLine(e.getRange().getEnd().getLine())
                        .setEndColumn(e.getRange().getEnd().getColumn())
                        .setEndOffset(e.getRange().getEnd().getOffset())
                        .build())
                .build();
    }
}
