package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Timestamps;

import ru.yandex.monitoring.api.v3.ListAllLabelValuesResponse;
import ru.yandex.monitoring.api.v3.ListLabelKeysResponse;
import ru.yandex.monitoring.api.v3.ListLabelValuesResponse;
import ru.yandex.monitoring.api.v3.ListMetricNamesResponse;
import ru.yandex.monitoring.api.v3.ListMetricsResponse;
import ru.yandex.monitoring.api.v3.MetricType;
import ru.yandex.solomon.core.label.LabelConf;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metrics.client.LabelNamesResponse;
import ru.yandex.solomon.metrics.client.LabelValuesResponse;
import ru.yandex.solomon.metrics.client.MetricNamesResponse;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.util.labelStats.LabelStats;
import ru.yandex.solomon.util.labelStats.LabelValuesStats;
import ru.yandex.solomon.util.text.TextWithNumbersComparator;
import ru.yandex.solomon.ydb.page.TokenBasePage;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class MetricsMetaDtoConverter {
    public static ListMetricsResponse toListMetricsResponse(TokenBasePage<MetricKey> response) {
        var metrics = response.getItems().stream()
                .map(MetricsMetaDtoConverter::fromMetricKey)
                .collect(Collectors.toList());

        return ListMetricsResponse.newBuilder()
                .addAllMetrics(metrics)
                .setNextPageToken(response.getNextPageToken())
                .build();
    }

    private static ListMetricsResponse.MetricMeta fromMetricKey(MetricKey metricKey) {
        Map<String, String> labelsWithoutProject = metricKey.getLabels().removeByKey(LabelKeys.PROJECT).toMap();
        MetricType type = fromMetricTypeModel(metricKey.getType());

        return ListMetricsResponse.MetricMeta.newBuilder()
                .setName(metricKey.getName())
                .putAllLabels(labelsWithoutProject)
                .setType(type)
                .setCreatedAt(Timestamps.fromMillis(metricKey.getCreatedAtMillis()))
                .build();
    }

    private static MetricType fromMetricTypeModel(ru.yandex.monlib.metrics.MetricType type) {
        return switch (type) {
            case DGAUGE -> MetricType.DGAUGE;
            case IGAUGE -> MetricType.IGAUGE;
            case COUNTER -> MetricType.COUNTER;
            case RATE -> MetricType.RATE;
            default -> MetricType.METRIC_TYPE_UNSPECIFIED;
        };
    }

    public static ListMetricNamesResponse toListMetricNamesResponse(MetricNamesResponse response) {
        return ListMetricNamesResponse.newBuilder()
                .addAllNames(response.getNames())
                .setTruncated(response.isTruncated())
                .build();
    }

    public static ListLabelKeysResponse toListLabelKeysResponse(LabelNamesResponse response) {
        List<String> labelKeys = response.getNames().stream()
                .sorted(LabelConf.getPartialSorterComparator())
                .collect(Collectors.toList());

        return ListLabelKeysResponse.newBuilder()
                .addAllKeys(labelKeys)
                .build();
    }

    public static ListLabelValuesResponse toListLabelValuesResponse(LabelValuesResponse response, boolean sharded) {
        LabelValuesStats labelValuesStats = response.getLabelValuesStats();

        if (labelValuesStats.getStatsByLabelKey().isEmpty()) {
            return ListLabelValuesResponse.newBuilder()
                    .setSharded(sharded)
                    .build();
        }

        LabelStats labelStats = labelValuesStats.getStatsByLabelKey().values().iterator().next();

        List<String> filteredAndSortedValues =
                labelStats.getValues().stream()
                        .sorted(TextWithNumbersComparator.instance)
                        .collect(Collectors.toList());

        int maxMetricsCount = response.getMetricsCountByDestination().isEmpty()
                ? 0
                : Collections.max(response.getMetricsCountByDestination().values());

        boolean absent = labelStats.getCount() < labelValuesStats.getMetricsCount();

        return ListLabelValuesResponse.newBuilder()
                .addAllValues(filteredAndSortedValues)
                .setMaxMetricsCount(maxMetricsCount)
                .setTruncated(labelStats.isTruncated())
                .setAbsent(absent)
                .setSharded(sharded)
                .build();
    }

    public static ListAllLabelValuesResponse toListAllLabelValuesResponse(
            Selectors selectors,
            LabelValuesResponse response,
            boolean sharded) {
        LabelValuesStats labelValuesStats = response.getLabelValuesStats();

        Set<String> exactKeys = selectors.stream()
                .filter(Selector::isExact)
                .map(Selector::getKey)
                .collect(Collectors.toSet());

        if (labelValuesStats.getStatsByLabelKey().isEmpty()) {
            return ListAllLabelValuesResponse.newBuilder()
                    .setSharded(sharded)
                    .build();
        }

        Map<String, Integer> metricsCountByDestination = response.getMetricsCountByDestination();

        int maxMetricsCount = metricsCountByDestination.isEmpty()
                ? 0
                : Collections.max(metricsCountByDestination.values());

        int totalMetricsCount = labelValuesStats.getMetricsCount();

        Map<String, Long> metricsCountByZoneId = new HashMap<>(metricsCountByDestination.size());
        for (var entry : metricsCountByDestination.entrySet()) {
            metricsCountByZoneId.put(entry.getKey(), (long) entry.getValue());
        }

        var builder = ListAllLabelValuesResponse.newBuilder()
                .setMaxMetricsCount(maxMetricsCount)
                .setSharded(sharded);

        var labels = labelValuesStats.getStatsByLabelKey().entrySet()
                .stream()
                .filter(entry -> !LabelKeys.PROJECT.equals(entry.getKey()))
                .filter(entry -> !entry.getValue().getValues().isEmpty())
                .filter(entry -> !exactKeys.contains(entry.getKey()))
                .sorted(Map.Entry.comparingByKey(LabelConf.getPartialSorterComparator()))
                .map(entry -> createLabelValues(entry.getKey(), entry.getValue(), totalMetricsCount))
                .collect(Collectors.toList());

        builder.addAllLabels(labels);
        builder.putAllMetricsCountByZoneId(metricsCountByZoneId);

        return builder.build();
    }

    private static ListAllLabelValuesResponse.LabelValues createLabelValues(String labelKey, LabelStats labelStats, int totalMetricsCount) {
        boolean absent = labelStats.getCount() < totalMetricsCount;

        List<String> values = labelStats.getValues().stream()
                .sorted(TextWithNumbersComparator.instance)
                .collect(Collectors.toList());

        return ListAllLabelValuesResponse.LabelValues.newBuilder()
                .setKey(labelKey)
                .addAllValues(values)
                .setAbsent(absent)
                .setTruncated(labelStats.isTruncated())
                .build();
    }
}
