package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Timestamps;

import ru.yandex.monitoring.api.v3.QuickLinks;
import ru.yandex.monitoring.api.v3.QuickLinksData;
import ru.yandex.solomon.conf.db3.QuickLinksRecord;
import ru.yandex.solomon.core.exceptions.BadRequestException;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class QuickLinksDtoConverter {

    public static QuickLinksRecord encode(QuickLinks proto, int version) {
        return QuickLinksRecord.newBuilder()
                .setProjectId(proto.getProjectId())
                .setItems(toLinkItems(proto.getData().getItemsList()))
                .setCreatedAtMillis(Timestamps.toMillis(proto.getCreatedAt()))
                .setUpdatedAtMillis(Timestamps.toMillis(proto.getModifiedAt()))
                .setCreatedBy(proto.getCreatedBy())
                .setUpdatedBy(proto.getModifiedBy())
                .setVersion(version)
                .build();
    }

    public static QuickLinks decode(QuickLinksRecord model) {
        var data = QuickLinksData.newBuilder()
                .addAllItems(fromMenuItems(model.getItems()))
                .build();

        return QuickLinks.newBuilder()
                .setProjectId(model.getProjectId())
                .setData(data)
                .setCreatedAt(Timestamps.fromMillis(model.getCreatedAtMillis()))
                .setModifiedAt(Timestamps.fromMillis(model.getUpdatedAtMillis()))
                .setCreatedBy(model.getCreatedBy())
                .setModifiedBy(model.getUpdatedBy())
                .build();
    }

    private static QuickLinksRecord.LinkItem[] toLinkItems(List<QuickLinksData.Item> itemsList) {
        return itemsList.stream()
                .map(QuickLinksDtoConverter::toMenuItem)
                .toArray(QuickLinksRecord.LinkItem[]::new);
    }

    private static QuickLinksRecord.LinkItem toMenuItem(QuickLinksData.Item proto) {
        switch (proto.getTypeCase()) {
            case LINK -> {
                QuickLinksData.LinkItem item = proto.getLink();
                return new QuickLinksRecord.LinkItem(item.getTitle(), item.getUrl(), new QuickLinksRecord.LinkItem[0]);
            }
            case FOLDER -> {
                QuickLinksData.FolderItem item = proto.getFolder();
                return new QuickLinksRecord.LinkItem(item.getTitle(), "", toLinkItems(item.getItemsList()));
            }
            case TYPE_NOT_SET -> throw new BadRequestException("item type not set");
            default -> throw new IllegalStateException("unknown item type: " + proto.getTypeCase());
        }
    }

    private static List<QuickLinksData.Item> fromMenuItems(QuickLinksRecord.LinkItem[] items) {
        return Arrays.stream(items)
                .map(QuickLinksDtoConverter::fromMenuItem)
                .collect(Collectors.toList());
    }

    private static QuickLinksData.Item fromMenuItem(QuickLinksRecord.LinkItem item) {
        if (item.getUrl().isEmpty()) {
            return QuickLinksData.Item.newBuilder()
                    .setFolder(QuickLinksData.FolderItem.newBuilder()
                            .setTitle(item.getTitle())
                            .addAllItems(fromMenuItems(item.getItems()))
                            .build())
                    .build();
        }

        return QuickLinksData.Item.newBuilder()
                .setLink(QuickLinksData.LinkItem.newBuilder()
                        .setTitle(item.getTitle())
                        .setUrl(item.getUrl())
                        .build())
                .build();
    }
}
