package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monitoring.api.v3.CreateResourceRequest;
import ru.yandex.monitoring.api.v3.DeleteResourceRequest;
import ru.yandex.monitoring.api.v3.GetResourceRequest;
import ru.yandex.monitoring.api.v3.ListResourcesRequest;
import ru.yandex.monitoring.api.v3.ListResourcesResponse;
import ru.yandex.monitoring.api.v3.ServiceProviderResource;
import ru.yandex.monitoring.api.v3.ServiceProviderResourceId;
import ru.yandex.monitoring.api.v3.ServiceProviderResourceSeverity;
import ru.yandex.monitoring.api.v3.UpdateResourceRequest;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.api.v3.intranet.impl.ResourcesYtDataSource;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.name.resolver.client.FindRequest;
import ru.yandex.solomon.name.resolver.client.FindResponse;
import ru.yandex.solomon.name.resolver.client.ResolveRequest;
import ru.yandex.solomon.name.resolver.client.ResolveResponse;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.client.UpdateRequest;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class ServiceProviderResourceDtoConverter {
    public static String toResourceId(ServiceProviderResourceId id) {
        switch (id.getIdCase()) {
            case COMPLEX_ID -> {
                var complexId = id.getComplexId();
                var map = new HashMap<>(complexId.getResourceComplexIdMap());
                map.put("serviceProviderId", complexId.getServiceProviderId());
                return map.entrySet().stream()
                        .sorted(Map.Entry.comparingByKey())
                        .collect(Collectors.toList()).hashCode() + "";
            }
            case RESOURCE_ID -> {
                return id.getResourceId();
            }
        }
        throw new BadRequestException("Id should be specified");
    }

    public static ResolveRequest toResolveRequest(GetResourceRequest request) {
        String id = ServiceProviderResourceDtoConverter.toResourceId(request.getId());
        return ResolveRequest.newBuilder()
                .cloudId(request.getAbcSlug())
                .resourceIds(List.of(id))
                .build();
    }

    public static ServiceProviderResource toSingleResource(List<Resource> resources) {
        if (resources.size() != 1) {
            throw new BadRequestException("No single resource");
        }
        final Resource resource = resources.get(0);
        return toServiceProviderResource(resource);
    }

    private static ServiceProviderResource toServiceProviderResource(Resource resource) {
        return ServiceProviderResource.newBuilder()
                .setAbcSlug(resource.cloudId)
                .setProjectId(resource.folderId)
                .setServiceProviderId(resource.service)
                .setType(resource.type)
                .putAllResourceComplexId(Nullables.orEmpty(resource.resourceComplexId))
                .setName(resource.name)
                .setResourceId(resource.resourceId)
                .setDeletedAt(resource.deletedAt)
                .setSeverity(severity(resource.severity))
                .setResponsible(resource.responsible)
                .setEnvironment(resource.environment)
                .build();
    }

    private static ServiceProviderResourceSeverity severity(Resource.Severity severity) {
        switch (severity) {
            case HIGHLY_CRITICAL -> {
                return ServiceProviderResourceSeverity.SERVICE_PROVIDER_RESOURCE_SEVERITY_HIGHLY_CRITICAL;
            }
            case CRITICAL -> {
                return ServiceProviderResourceSeverity.SERVICE_PROVIDER_RESOURCE_SEVERITY_CRITICAL;
            }
            case NORMAL, NON_CRITICAL -> {
                return ServiceProviderResourceSeverity.SERVICE_PROVIDER_RESOURCE_SEVERITY_NON_CRITICAL;
            }
            default -> {
                return ServiceProviderResourceSeverity.SERVICE_PROVIDER_RESOURCE_SEVERITY_UNSPECIFIED;
            }
        }
    }

    public static UpdateRequest toUpdateResourcesRequest(CreateResourceRequest request) {
        return new UpdateRequest(request.getAbcSlug(), List.of(toResource(request)));
    }

    private static Resource toResource(CreateResourceRequest request) {
        String resourceId = getId(request);
        return new Resource()
                .setCloudId(request.getAbcSlug())
                .setFolderId(request.getProjectId())
                .setService(request.getServiceProviderId())
                .setType(request.getType())
                .setName(request.getName())
                .setResourceId(resourceId)
                .setSeverity(severity(request.getSeverityValue() > 0
                        ? request.getSeverity()
                        : ServiceProviderResourceSeverity.SERVICE_PROVIDER_RESOURCE_SEVERITY_HIGHLY_CRITICAL))
                .setResourceComplexId(request.getResourceComplexIdMap())
                .setResponsible(request.getResponsible())
                .setEnvironment(request.getEnvironment())
                .setUpdatedAt(System.currentTimeMillis());
    }

    private static Resource.Severity severity(ServiceProviderResourceSeverity severity) {
        switch (severity) {
            case SERVICE_PROVIDER_RESOURCE_SEVERITY_HIGHLY_CRITICAL -> {
                return Resource.Severity.HIGHLY_CRITICAL;
            }
            case SERVICE_PROVIDER_RESOURCE_SEVERITY_CRITICAL -> {
                return Resource.Severity.CRITICAL;
            }
            case SERVICE_PROVIDER_RESOURCE_SEVERITY_NON_CRITICAL -> {
                return Resource.Severity.NON_CRITICAL;
            }
            default -> {
                return Resource.Severity.UNKNOWN;
            }
        }
    }

    private static String getId(CreateResourceRequest request) {
        String resourceId;
        if (request.getResourceId().isBlank()) {
            var map = new HashMap<>(request.getResourceComplexIdMap());
            map.put("serviceProviderId", request.getServiceProviderId());
            resourceId = map.entrySet().stream()
                    .sorted(Map.Entry.comparingByKey())
                    .collect(Collectors.toList()).hashCode() + "";
        } else {
            resourceId = request.getResourceId();
        }
        return resourceId;
    }

    public static UpdateRequest toUpdateResourcesRequest(UpdateResourceRequest request, ResolveResponse prev) {
        if (prev.resources.size() != 1) {
            throw new BadRequestException("No single resource");
        }
        final Resource resource = prev.resources.get(0);
        return new UpdateRequest(request.getAbcSlug(), List.of(resource
                .setFolderId(request.getProjectId())
                .setType(request.getType())
                .setName(request.getName())
                .setSeverity(severity(request.getSeverity()))
                .setResponsible(request.getResponsible())
                .setEnvironment(request.getEnvironment())
                .setUpdatedAt(System.currentTimeMillis())
        ));
    }

    public static UpdateRequest toUpdateResourcesRequest(DeleteResourceRequest request, ResolveResponse prev) {
        if (prev.resources.size() != 1) {
            throw new BadRequestException("No single resource");
        }
        final Resource resource = prev.resources.get(0);
        var millis = System.currentTimeMillis();
        return new UpdateRequest(request.getAbcSlug(), List.of(resource
                .setUpdatedAt(millis)
                .setDeletedAt(millis)
        ));
    }

    public static FindRequest toUpdateResourcesRequest(ListResourcesRequest request) {
        return FindRequest.newBuilder()
                .cloudId(request.getAbcSlug())
                .limit((int) request.getPageSize())
                .selectors(Selectors.parse(request.getSelectors()))
                .build();
    }

    public static ListResourcesResponse toResources(FindResponse findResponse) {
        return ListResourcesResponse.newBuilder()
                .setNextPageToken(findResponse.nextPageToken)
                .addAllResources(findResponse.resources.stream()
                        .map(ServiceProviderResourceDtoConverter::toServiceProviderResource)
                        .collect(Collectors.toList()))
                .build();
    }

    public static ServiceProviderResource toServiceProviderResource(CreateResourceRequest request) {
        String resourceId = getId(request);
        return ServiceProviderResource.newBuilder()
                .setResourceId(resourceId)
                .setAbcSlug(request.getAbcSlug())
                .setProjectId(request.getProjectId())
                .setServiceProviderId(request.getServiceProviderId())
                .setType(request.getType())
                .putAllResourceComplexId(request.getResourceComplexIdMap())
                .setName(request.getName())
                .setSeverity(request.getSeverity())
                .build();
    }

    public static ResolveRequest toResolveRequest(List<CreateResourceRequest> requests) {
        return ResolveRequest.newBuilder()
                .cloudId(requests.get(0).getAbcSlug())
                .resourceIds(requests.stream()
                        .map(ServiceProviderResourceDtoConverter::getId)
                        .collect(Collectors.toList()))
                .build();
    }

    public static UpdateRequest toUpdateResourcesRequest(
            ResolveResponse resolveResponse,
            List<CreateResourceRequest> requests,
            boolean removeOther,
            String serviceProviderId)
    {
        var map = resolveResponse.resources.stream()
                .collect(Collectors.toMap(Resource::getResourceId, Function.identity()));
        List<Resource> resources = new ArrayList<>();
        for (var create : requests) {
            var resourceId = getId(create);
            var prevResource = map.get(resourceId);
            if (prevResource == null) {
                resources.add(toResource(create));
            } else {
                resources.add(prevResource
                        .setType(create.getType())
                        .setResponsible(create.getResponsible())
                        .setEnvironment(create.getEnvironment())
                        .setUpdatedAt(System.currentTimeMillis()));
            }
        }
        return new UpdateRequest(requests.get(0).getAbcSlug(), resources, removeOther, serviceProviderId);
    }

    public static List<CreateResourceRequest> toCreateResourceRequests(Map.Entry<String, List<ResourcesYtDataSource.TableRow>> entry) {
        List<CreateResourceRequest> requests = new ArrayList<>();
        for (var resource : entry.getValue()) {
            requests.add(CreateResourceRequest.newBuilder()
                    .setAbcSlug(entry.getKey())
                    .setServiceProviderId(resource.service_provider_id)
                    .setType(Nullables.orEmpty(resource.resource_type))
                    .putAllResourceComplexId(resource.resource_id)
                    .setResourceId(resource.global_id)
                    .setResponsible(resource.responsible)
                    .setEnvironment(resource.environment)
                    .setSeverity(ServiceProviderResourceSeverity.SERVICE_PROVIDER_RESOURCE_SEVERITY_HIGHLY_CRITICAL)
                    .build());
        }
        return requests;
    }
}
