package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monitoring.api.v3.ListClusterServicesResponse;
import ru.yandex.monitoring.api.v3.ListServiceClustersResponse;
import ru.yandex.monitoring.api.v3.Shard;
import ru.yandex.solomon.core.db.model.ClusterServiceAssociation;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.ydb.page.TokenBasePage;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ShardAssociationDtoConverter {

    public static ListClusterServicesResponse fromClusterServicesModel(
            String projectId,
            TokenBasePage<ClusterServiceAssociation> response) {
        var associations = response.getItems().stream()
                .map(item -> fromClusterServiceModel(projectId, item))
                .collect(Collectors.toList());

        return ListClusterServicesResponse.newBuilder()
                .addAllAssociations(associations)
                .setNextPageToken(response.getNextPageToken())
                .build();
    }

    public static ListServiceClustersResponse fromServiceClustersModel(
            String projectId,
            TokenBasePage<ClusterServiceAssociation> response) {
        var associations = response.getItems().stream()
                .map(item -> fromServiceClusterModel(projectId, item))
                .collect(Collectors.toList());

        return ListServiceClustersResponse.newBuilder()
                .addAllAssociations(associations)
                .setNextPageToken(response.getNextPageToken())
                .build();
    }

    private static ListClusterServicesResponse.Association fromClusterServiceModel(
            String projectId,
            ClusterServiceAssociation association) {
        return ListClusterServicesResponse.Association.newBuilder()
                .setProjectId(projectId)
                .setShardId(association.getShardId())
                .setServiceId(association.getId())
                .setServiceName(association.getName())
                .setMode(ShardStateConverter.fromStateModel(association.getShardState()))
                .build();
    }

    private static ListServiceClustersResponse.Association fromServiceClusterModel(
            String projectId,
            ClusterServiceAssociation association) {
        return ListServiceClustersResponse.Association.newBuilder()
                .setProjectId(projectId)
                .setShardId(association.getShardId())
                .setClusterId(association.getId())
                .setClusterName(association.getName())
                .setMode(ShardStateConverter.fromStateModel(association.getShardState()))
                .build();
    }

    public static Set<ShardState> toModel(List<Shard.Mode> filterByModes) {
        if (filterByModes.isEmpty()) {
            return Set.of();
        }
        Set<ShardState> states = new HashSet<>(filterByModes.size());
        for (Shard.Mode mode : filterByModes) {
            switch (mode) {
                case MODE_UNSPECIFIED, UNRECOGNIZED -> {
                }
                case ACTIVE -> {
                    states.add(ShardState.RW);
                    // Deprecated.
                    states.add(ShardState.ACTIVE);
                }
                case READ_ONLY -> {
                    states.add(ShardState.READ_ONLY);
                }
                case WRITE_ONLY -> {
                    states.add(ShardState.WRITE_ONLY);
                }
                case INACTIVE -> {
                    states.add(ShardState.INACTIVE);
                }
            }
        }
        return states;
    }
}
