package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.time.Instant;
import java.util.Collections;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Durations;
import com.google.protobuf.util.Timestamps;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monitoring.api.v3.CreateShardRequest;
import ru.yandex.monitoring.api.v3.ListShardTargetsResponse;
import ru.yandex.monitoring.api.v3.Shard;
import ru.yandex.monitoring.api.v3.ShardQuotas;
import ru.yandex.monitoring.api.v3.ShardSettings;
import ru.yandex.monitoring.api.v3.UpdateShardRequest;
import ru.yandex.monitoring.api.v3.ValidationMode;
import ru.yandex.salmon.fetcher.proto.FetcherApiProto;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.coremon.client.CoremonClient;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.solomon.util.UnknownShardLocation;
import ru.yandex.solomon.util.net.KnownDc;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static ru.yandex.solomon.core.db.model.ValidationMode.LEGACY_SKIP;
import static ru.yandex.solomon.core.db.model.ValidationMode.STRICT_FAIL;
import static ru.yandex.solomon.core.db.model.ValidationMode.STRICT_SKIP;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public final class ShardDtoConverter {

    private final CoremonClient coremonClient;

    public ShardDtoConverter(CoremonClient coremonClient) {
        this.coremonClient = coremonClient;
    }

    public Pair<Shard, Integer> fromModelWithVersion(ru.yandex.solomon.core.db.model.Shard shard) {
        var dto = fromModel(shard);
        var version = shard.getVersion();
        return Pair.of(dto, version);
    }

    public Shard fromModel(ru.yandex.solomon.core.db.model.Shard shard) {
        var hosts = Collections.<String>emptyList();
        try {
            hosts = coremonClient.shardHosts(shard.getNumId());
        } catch (UnknownShardLocation ignore) {
        }
        return Shard.newBuilder()
                .setProjectId(shard.getProjectId())
                .setId(shard.getId())
                .setClusterId(shard.getClusterId())
                .setServiceId(shard.getServiceId())
                .setClusterLabelName(shard.getClusterName())
                .setServiceLabelName(shard.getServiceName())
                .setCreatedAt(Timestamps.fromMillis(shard.getCreatedAtMillis()))
                .setModifiedAt(Timestamps.fromMillis(shard.getUpdatedAtMillis()))
                .setCreatedBy(shard.getCreatedBy())
                .setModifiedBy(shard.getUpdatedBy())
                .setDescription(shard.getDescription())
                .setShardSettings(fromShardSettingsModel(shard))
                .setMode(ShardStateConverter.fromStateModel(shard.getState()))
                .setMetricNameLabel(shard.getMetricNameLabel())
                .setQuotas(ShardQuotas.newBuilder()
                        .setMaxMetricsPerDocument(shard.getMaxMetricsPerUrl())
                        .setMaxStoredMetrics(shard.getMaxFileMetrics())
                        .setMaxMemonlyMetrics(shard.getMaxMemMetrics())
                        .setMaxResponseSizeMb(shard.getMaxResponseSizeBytes() / (1 << 20))
                        .build())
                .setNumId(shard.getNumId())
                .setValidationMode(fromValidationModeModel(shard.getValidationMode()))
                .addAllHosts(hosts)
                .putAllLabels(shard.getLabels())
                .build();
    }

    public ru.yandex.solomon.core.db.model.Shard toModel(CreateShardRequest request, String login, Instant now) {
        var builder = ru.yandex.solomon.core.db.model.Shard.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        builder.setId(request.getShardId())
                .setClusterId(request.getClusterId())
                .setServiceId(request.getServiceId())
                .setCreatedAt(now)
                .setUpdatedAt(now)
                .setCreatedBy(login)
                .setUpdatedBy(login)
                .setDescription(request.getDescription())
                .setState(ShardStateConverter.toStateModel(request.getMode()))
                .setMetricNameLabel(request.getMetricNameLabel())
                .setValidationMode(toStateModel(request.getValidationMode()))
                .setLabels(request.getLabelsMap());

        if (request.hasShardSettings()) {
            builder.setShardSettings(ShardSettingsDtoConverter.toModel(request.getShardSettings()));
        }
        return builder.build();
    }

    public ru.yandex.solomon.core.db.model.Shard toModel(UpdateShardRequest request, String login, Instant now, int etag) {
        var builder = ru.yandex.solomon.core.db.model.Shard.newBuilder();
        switch (request.getContainerCase()) {
            case PROJECT_ID -> builder.setProjectId(request.getProjectId());
            default -> throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        builder.setId(request.getShardId())
                .setUpdatedAt(now)
                .setUpdatedBy(login)
                .setDescription(request.getDescription())
                .setState(ShardStateConverter.toStateModel(request.getMode()))
                .setMetricNameLabel(request.getMetricNameLabel())
                .setVersion(etag)
                .setValidationMode(toStateModel(request.getValidationMode()))
                .setLabels(request.getLabelsMap());

        if (request.hasShardSettings()) {
            builder.setShardSettings(ShardSettingsDtoConverter.toModel(request.getShardSettings()));
        }
        return builder.build();
    }

    private ShardSettings fromShardSettingsModel(ru.yandex.solomon.core.db.model.Shard shard) {
        if (shard.getShardSettings() == null) {
            return ShardSettings.newBuilder()
                    .setRetentionPolicy(ShardSettingsDtoConverter.fromRetentionPolicyModel(shard.getShardSettings().getRetentionPolicy()))
                    .setMetricsTtlDays(shard.getShardSettings().getMetricsTtl())
                    .build();
        }
        return ShardSettingsDtoConverter.fromModel(shard.getShardSettings());
    }

    @Nullable
    public UrlStatusType parseFilterByStatus(String filterByStatus, boolean notOkStatus) {
        try {
            return (notOkStatus || StringUtils.isEmpty(filterByStatus))
                    ? null
                    : UrlStatusType.valueOf(filterByStatus);
        } catch (Exception e) {
            throw new BadRequestException("unknown status: " + filterByStatus);
        }
    }

    @Nullable
    public KnownDc parseFilterByDc(String filterByDc) {
        try {
            return filterByDc.isEmpty() ? null : KnownDc.valueOf(filterByDc);
        } catch (Exception e) {
            throw new BadRequestException("unknown DC: " + filterByDc);
        }
    }

    public ListShardTargetsResponse fromModel(TokenBasePage<FetcherApiProto.TargetStatus> page) {
        var targets = page.getItems().stream()
                .map(this::fromModel)
                .collect(Collectors.toList());

        return ListShardTargetsResponse.newBuilder()
                .addAllTargets(targets)
                .build();
    }

    private ListShardTargetsResponse.ShardTarget fromModel(FetcherApiProto.TargetStatus item) {
        var r = ListShardTargetsResponse.ShardTarget.newBuilder()
                .setUrl(item.getUrl())
                .setHost(item.getHost())
                .setDc(item.getDc())
                .setLastFetchTime(Timestamps.fromMillis(item.getLastFetchTime()))
                .setLastFetchDuration(Durations.fromMillis(item.getLastFetchDurationMillis()))
                .setLastResponseSizeBytes(item.getLastResponseBytes())
                .setLastStatus(item.getLastUrlStatus().name())
                .setLastError(item.getLastUrlError())
                .setLastContentType(item.getLastContentType().name());

        for (var ss: item.getShardStatusesList()) {
            var shardStatus = ListShardTargetsResponse.ShardStatus.newBuilder()
                    .setShardId(ss.getShardId())
                    .setLastStatus(ss.getLastStatus().name())
                    .setLastError(ss.getLastError())
                    .setLastParsedMetrics(ss.getLastMetricsParsed())
                    .setLastOverflowMetrics(ss.getLastMetricsOverflow());
            r.addShardStatuses(shardStatus);
        }

        return r.build();
    }

    private ValidationMode fromValidationModeModel(ru.yandex.solomon.core.db.model.ValidationMode mode) {
        return switch (mode) {
            case LEGACY_SKIP -> ValidationMode.VALIDATION_MODE_LEGACY_SKIP;
            case STRICT_SKIP -> ValidationMode.VALIDATION_MODE_STRICT_SKIP;
            case STRICT_FAIL -> ValidationMode.VALIDATION_MODE_STRICT_FAIL;
        };
    }

    private ru.yandex.solomon.core.db.model.ValidationMode toStateModel(ValidationMode mode) {
        return switch (mode) {
            case VALIDATION_MODE_LEGACY_SKIP, VALIDATION_MODE_UNSPECIFIED, UNRECOGNIZED -> LEGACY_SKIP;
            case VALIDATION_MODE_STRICT_SKIP -> STRICT_SKIP;
            case VALIDATION_MODE_STRICT_FAIL -> STRICT_FAIL;
        };
    }
}
