package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.util.Arrays;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Durations;
import org.jetbrains.annotations.NotNull;

import ru.yandex.monitoring.api.v3.ShardSettings;
import ru.yandex.monitoring.api.v3.ShardSettings.AggregationRule;
import ru.yandex.monitoring.api.v3.ShardSettings.AggregationSettings;
import ru.yandex.monitoring.api.v3.ShardSettings.HostLabelPolicy;
import ru.yandex.monitoring.api.v3.ShardSettings.PullSettings;
import ru.yandex.monitoring.api.v3.ShardSettings.PushSettings;
import ru.yandex.monitoring.api.v3.ShardSettings.RetentionPolicy;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.MetricAggregation;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.ServiceMetricConf.AggrRule;
import ru.yandex.solomon.core.db.model.ShardSettings.PullProtocol;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ShardSettingsDtoConverter {
    public static ShardSettings fromModel(ru.yandex.solomon.core.db.model.ShardSettings shardSettings) {
        var builder = ShardSettings.newBuilder();

        switch (shardSettings.getType()) {
            case PULL -> {
                var pullSettings = shardSettings.getPullSettings();
                if (pullSettings != null) {
                    builder.setPullSettings(PullSettings.newBuilder()
                            .setPath(pullSettings.getPath())
                            .setPort(pullSettings.getPort())
                            .setAddTimestampArgs(pullSettings.isAddTsArgs())
                            .setProtocol(fromPullProtocolModel(pullSettings.getProtocol()))
                            .setTvmDestinationId(pullSettings.getTvmDestinationId())
                            .setHostLabelPolicy(fromHostLabelPolicyModel(pullSettings.getHostLabelPolicy()))
                            .build());
                } else {
                    builder.setPullSettings(PullSettings.getDefaultInstance());
                }
            }
            case PUSH -> builder.setPushSettings(PushSettings.getDefaultInstance());
        }

        builder.setAggregationSettings(fromAggregationSettingsModel(shardSettings.getAggregationSettings()));

        builder.setGrid(Durations.fromSeconds(shardSettings.getGrid()));
        builder.setMetricsTtlDays(shardSettings.getMetricsTtl());
        builder.setRetentionPolicy(fromRetentionPolicyModel(shardSettings.getRetentionPolicy()));
        builder.setInterval(Durations.fromSeconds(shardSettings.getInterval()));

        return builder.build();
    }

    public static ru.yandex.solomon.core.db.model.ShardSettings toModel(ShardSettings shardSettings) {
        final ru.yandex.solomon.core.db.model.ShardSettings.Type type;
        final ru.yandex.solomon.core.db.model.ShardSettings.PullSettings pullSettings;
        var pullSettingsDto = shardSettings.getPullSettings();
        if (shardSettings.getTypeCase() == ShardSettings.TypeCase.PULL_SETTINGS) {
            type = ru.yandex.solomon.core.db.model.ShardSettings.Type.PULL;
            pullSettings = ru.yandex.solomon.core.db.model.ShardSettings.PullSettings.newBuilder()
                    .setPath(pullSettingsDto.getPath())
                    .setPort((int) pullSettingsDto.getPort())
                    .setAddTsArgs(pullSettingsDto.getAddTimestampArgs())
                    .setProtocol(toPullProtocolModel(pullSettingsDto.getProtocol()))
                    .setTvmDestinationId(pullSettingsDto.getTvmDestinationId())
                    .setHostLabelPolicy(fromHostLabelPolicyModel(pullSettingsDto.getHostLabelPolicy()))
                    .build();
        } else {
            type = ru.yandex.solomon.core.db.model.ShardSettings.Type.PUSH;
            pullSettings = null;
        }

        var aggregationSettings = toAggregationSettingsModel(shardSettings.getAggregationSettings());

        return ru.yandex.solomon.core.db.model.ShardSettings.of(
                type,
                pullSettings,
                (int) Durations.toSeconds(shardSettings.getGrid()),
                (int) shardSettings.getMetricsTtlDays(),
                toRetentionPolicyModel(shardSettings.getRetentionPolicy()),
                aggregationSettings,
                (int) Durations.toSeconds(shardSettings.getInterval())
        );
    }

    private static AggregationSettings fromAggregationSettingsModel(ru.yandex.solomon.core.db.model.ShardSettings.AggregationSettings aggregationSettings) {
        return AggregationSettings.newBuilder()
                .setEnabled(aggregationSettings.isEnabled())
                .addAllRules(Arrays.stream(aggregationSettings.getAggrRules())
                        .map(ShardSettingsDtoConverter::fromAggrRuleModel)
                        .collect(Collectors.toList()))
                .setMemOnly(aggregationSettings.isMemOnly())
                .build();
    }

    private static ru.yandex.solomon.core.db.model.ShardSettings.AggregationSettings toAggregationSettingsModel(AggregationSettings aggregationSettings) {
        var aggrRules = aggregationSettings.getRulesList().stream()
                .map(ShardSettingsDtoConverter::toAggrRuleModel)
                .toArray(ServiceMetricConf.AggrRule[]::new);
        return ru.yandex.solomon.core.db.model.ShardSettings.AggregationSettings.of(
                aggregationSettings.getEnabled(),
                aggrRules,
                aggregationSettings.getMemOnly()
        );
    }

    private static ru.yandex.monitoring.api.v3.ShardSettings.AggregationRule fromAggrRuleModel(AggrRule model) {
        return AggregationRule.newBuilder()
                .addAllConditions(Arrays.asList(model.getCond()))
                .addAllTargets(Arrays.asList(model.getTarget()))
                .setFunction(fromMetricAggregationModel(model.getAggr()))
                .build();
    }

    @NotNull
    private static AggrRule toAggrRuleModel(ru.yandex.monitoring.api.v3.ShardSettings.AggregationRule aggrRule) {
        var conditions = aggrRule.getConditionsList().toArray(String[]::new);
        var targets = aggrRule.getTargetsList().toArray(String[]::new);
        var function = toMetricAggregationModel(aggrRule.getFunction());
        return new AggrRule(conditions, targets, function);
    }

    public static PullProtocol toPullProtocolModel(ShardSettings.PullProtocol protocol) {
        return switch (protocol) {
            case HTTP -> PullProtocol.HTTP;
            case HTTPS -> PullProtocol.HTTPS;
            default -> PullProtocol.UNKNOWN;
        };
    }

    public static ShardSettings.PullProtocol fromPullProtocolModel(@Nullable PullProtocol protocol) {
        if (protocol == null) {
            return ShardSettings.PullProtocol.PULL_PROTOCOL_UNSPECIFIED;
        }
        return switch (protocol) {
            case HTTP -> ShardSettings.PullProtocol.HTTP;
            case HTTPS -> ShardSettings.PullProtocol.HTTPS;
            case UNKNOWN -> ShardSettings.PullProtocol.PULL_PROTOCOL_UNSPECIFIED;
        };
    }

    private static MetricAggregation toMetricAggregationModel(AggregationRule.Function function) {
        return switch (function) {
            case SUM -> MetricAggregation.SUM;
            case LAST -> MetricAggregation.LAST;
            default -> null;
        };
    }

    public static AggregationRule.Function fromMetricAggregationModel(@Nullable MetricAggregation aggregation) {
        if (aggregation == null) {
            return AggregationRule.Function.FUNCTION_UNSPECIFIED;
        }

        return switch (aggregation) {
            case SUM -> AggregationRule.Function.SUM;
            case LAST -> AggregationRule.Function.LAST;
        };
    }

    public static RetentionPolicy fromRetentionPolicyModel(DecimPolicy retentionPolicy) {
        return switch (retentionPolicy) {
            case UNDEFINED -> RetentionPolicy.RETENTION_POLICY_UNSPECIFIED;
            case POLICY_KEEP_FOREVER -> RetentionPolicy.KEEP_FOREVER;
            case POLICY_5_MIN_AFTER_7_DAYS -> RetentionPolicy.FIVE_MIN_AFTER_WEEK;
            case POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_3_MONTHS -> RetentionPolicy.ONE_MIN_AFTER_MONTH_FIVE_MIN_AFTER_THREE_MONTHS;
            case POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_2_MONTHS -> RetentionPolicy.ONE_MIN_AFTER_MONTH_FIVE_MIN_AFTER_TWO_MONTHS;
            case POLICY_5_MIN_AFTER_2_MONTHS -> RetentionPolicy.FIVE_MIN_AFTER_TWO_MONTHS;
            case POLICY_5_MIN_AFTER_8_DAYS -> RetentionPolicy.FIVE_MIN_AFTER_8_DAYS;
        };
    }

    private static DecimPolicy toRetentionPolicyModel(RetentionPolicy retentionPolicy) {
        return switch (retentionPolicy) {
            case RETENTION_POLICY_UNSPECIFIED -> DecimPolicy.UNDEFINED;
            case KEEP_FOREVER -> DecimPolicy.POLICY_KEEP_FOREVER;
            case FIVE_MIN_AFTER_WEEK -> DecimPolicy.POLICY_5_MIN_AFTER_7_DAYS;
            case ONE_MIN_AFTER_MONTH_FIVE_MIN_AFTER_THREE_MONTHS -> DecimPolicy.POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_3_MONTHS;
            case ONE_MIN_AFTER_MONTH_FIVE_MIN_AFTER_TWO_MONTHS -> DecimPolicy.POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_2_MONTHS;
            case FIVE_MIN_AFTER_TWO_MONTHS -> DecimPolicy.POLICY_5_MIN_AFTER_2_MONTHS;
            case FIVE_MIN_AFTER_8_DAYS -> DecimPolicy.POLICY_5_MIN_AFTER_8_DAYS;
            default -> throw new IllegalStateException("unknown retention policy: " + retentionPolicy);
        };
    }

    private static HostLabelPolicy fromHostLabelPolicyModel(ru.yandex.solomon.core.db.model.ShardSettings.HostLabelPolicy hostLabelPolicy) {
        return switch (hostLabelPolicy) {
            case SHORT_HOSTNAME -> HostLabelPolicy.SHORT_HOSTNAME;
            case NO_HOSTNAME -> HostLabelPolicy.NO_HOSTNAME;
            case FULL_HOSTNAME -> HostLabelPolicy.FULL_HOSTNAME;
            case UNKNOWN -> HostLabelPolicy.HOST_LABEL_POLICY_UNSPECIFIED;
        };
    }

    private static ru.yandex.solomon.core.db.model.ShardSettings.HostLabelPolicy fromHostLabelPolicyModel(HostLabelPolicy hostLabelPolicy) {
        return switch (hostLabelPolicy) {
            case NO_HOSTNAME -> ru.yandex.solomon.core.db.model.ShardSettings.HostLabelPolicy.NO_HOSTNAME;
            case FULL_HOSTNAME -> ru.yandex.solomon.core.db.model.ShardSettings.HostLabelPolicy.FULL_HOSTNAME;
            case SHORT_HOSTNAME -> ru.yandex.solomon.core.db.model.ShardSettings.HostLabelPolicy.SHORT_HOSTNAME;
            default -> ru.yandex.solomon.core.db.model.ShardSettings.HostLabelPolicy.UNKNOWN;
        };
    }

    public static com.google.protobuf.Duration fromMetricsTtlModel(int metricsTtlDays) {
        return Durations.fromSeconds(TimeUnit.DAYS.toSeconds(metricsTtlDays));
    }
}
