package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import com.google.protobuf.Any;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.monitoring.api.v3.AlertingStubRequest;
import ru.yandex.monitoring.api.v3.CompleteStubRequest;
import ru.yandex.monitoring.api.v3.CompleteStubRequestResponse;
import ru.yandex.monitoring.api.v3.CreateResourceRequest;
import ru.yandex.monitoring.api.v3.CreateStubRequest;
import ru.yandex.monitoring.api.v3.CreateStubRequestResponse;
import ru.yandex.monitoring.api.v3.EvaluationStatus;
import ru.yandex.monitoring.api.v3.MonitoringConfig;
import ru.yandex.monitoring.api.v3.NotificationChannel;
import ru.yandex.monitoring.api.v3.ServiceProviderResourceSeverity;
import ru.yandex.solomon.alert.protobuf.AlertParameter;
import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateRequest;
import ru.yandex.solomon.alert.protobuf.TChannelConfig;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.alert.protobuf.TNotificationChannelOptions;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.gateway.stub.StubRequest;
import ru.yandex.solomon.util.Proto;

import static ru.yandex.solomon.gateway.api.v3.utils.StubRequestIdempotency.CREATE_STUB_REQUEST_OPERATION;

/**
 * @author Nuradil Zhambyl
 */
public class StubRequestConverter {
    public static StubRequest extractStubRequest(CreateStubRequest request) {
        return new StubRequest(
                request.getId(),
                request.getServiceProviderId(),
                CREATE_STUB_REQUEST_OPERATION,
                Any.pack(request.getStub()),
                Instant.now().toEpochMilli()
        );
    }

    public static AlertingStubRequest toAlertingStubRequest(StubRequest request) {
        return AlertingStubRequest
                .newBuilder()
                .setId(request.id())
                .setServiceProviderId(request.serviceProviderId())
                .setType(request.type())
                .setStub(Proto.unpack(request.stub(), MonitoringConfig.class))
                .build();
    }

    public static CreateStubRequestResponse toCreateStubRequestResponse(String id) {
        return CreateStubRequestResponse
                .newBuilder()
                .setId(id)
                .build();
    }

    public static CompleteStubRequestResponse toCompleteStubRequestResponse(String id) {
        return CompleteStubRequestResponse
                .newBuilder()
                .setId(id)
                .build();
    }

    private static List<CreateAlertsFromTemplateRequest.Resource> convertResources(List<ru.yandex.monitoring.api.v3.Resource> resources) {
        List<CreateAlertsFromTemplateRequest.Resource> list = new LinkedList<>();
        for (var resource : resources) {
            var map = new HashMap<>(resource.getResourceParametersMap());
            if (!StringUtils.isEmpty(resource.getResourceType())) {
                map.put("resourceType", resource.getResourceType());
            }
            var convertedResource = CreateAlertsFromTemplateRequest.Resource
                    .newBuilder()
                    .putAllResourceParameters(map)
                    .build();
            list.add(convertedResource);
        }
        return list;
    }

    private static List<String> convertIds(MonitoringConfig config) {
        List<String> list = new LinkedList<>();
        for (var alertConfig : config.getAlertConfigsList()) {
            list.add(alertConfig.getAlertTemplateId());
        }
        return list;
    }

    private static List<TEvaluationStatus.ECode> toECodeList(List<EvaluationStatus> statuses) {
        List<TEvaluationStatus.ECode> list = new LinkedList<>();
        for (var status : statuses) {
            var eCode = switch (status) {
                case EVALUATION_STATUS_OK -> TEvaluationStatus.ECode.OK;
                case EVALUATION_STATUS_NO_DATA -> TEvaluationStatus.ECode.NO_DATA;
                case EVALUATION_STATUS_ERROR -> TEvaluationStatus.ECode.ERROR;
                case EVALUATION_STATUS_ALARM -> TEvaluationStatus.ECode.ALARM;
                case EVALUATION_STATUS_WARN -> TEvaluationStatus.ECode.WARN;
                default -> null;
            };
            if (eCode != null) {
                list.add(eCode);
            }
        }
        return list;
    }

    private static TNotificationChannelOptions toTNotificationChannelOptions(NotificationChannel channel) {
        return TNotificationChannelOptions
                .newBuilder()
                .setChannelConfig(TChannelConfig
                        .newBuilder()
                        .setRepeatNotifyDelayMillis(channel.getRepeatNotifyDelayMs())
                        .addAllNotifyAboutStatuses(toECodeList(channel.getNotifyAboutStatusesList())))
                .build();
    }

    private static Map<String, TNotificationChannelOptions> channelsMapping(List<NotificationChannel> channels) {
        Map<String, TNotificationChannelOptions> mapping = new HashMap<>();
        for (var channel : channels) {
            mapping.put(channel.getNotificationChannelId(), toTNotificationChannelOptions(channel));
        }
        return mapping;
    }

    private static List<AlertParameter> formatThresholds(List<ru.yandex.monitoring.api.v3.AlertParameter> thresholds) {
        List<AlertParameter> list = new LinkedList<>();
        for (var threshold : thresholds) {
            var number = threshold.getParameterCase();
            var builder = AlertParameter.newBuilder();
            switch (number) {
                case DOUBLE_PARAMETER_VALUE -> builder.setDoubleParameterValue(AlertParameter
                        .DoubleParameterValue
                        .newBuilder()
                        .setName(threshold.getDoubleParameterValue().getName())
                        .setValue(threshold.getDoubleParameterValue().getValue())
                        .build());
                case INTEGER_PARAMETER_VALUE -> builder.setIntegerParameterValue(AlertParameter
                        .IntegerParameterValue
                        .newBuilder()
                        .setName(threshold.getIntegerParameterValue().getName())
                        .setValue(threshold.getIntegerParameterValue().getValue())
                        .build());
                case TEXT_PARAMETER_VALUE -> builder.setTextParameterValue(AlertParameter
                        .TextParameterValue
                        .newBuilder()
                        .setName(threshold.getTextParameterValue().getName())
                        .setValue(threshold.getTextParameterValue().getValue())
                        .build());
                case TEXT_LIST_PARAMETER_VALUE -> builder.setTextListParameterValue(AlertParameter
                        .TextListParameterValue
                        .newBuilder()
                        .setName(threshold.getTextListParameterValue().getName())
                        .addAllValues(threshold.getTextListParameterValue().getValuesList())
                        .build());
                case LABEL_LIST_PARAMETER_VALUE -> builder.setLabelListParameterValue(AlertParameter
                        .LabelListParameterValue
                        .newBuilder()
                        .setName(threshold.getLabelListParameterValue().getName())
                        .addAllValues(threshold.getLabelListParameterValue().getValuesList())
                        .build());
            }
            var result = builder.build();
            list.add(result);
        }
        return list;
    }

    private static List<CreateAlertsFromTemplateRequest.TemplateConfigs> convertTemplateConfigs(MonitoringConfig config) {
        List<CreateAlertsFromTemplateRequest.TemplateConfigs> list = new LinkedList<>();
        for (var alertConfig : config.getAlertConfigsList()) {
            var templateConfigs = CreateAlertsFromTemplateRequest.TemplateConfigs
                    .newBuilder()
                    .setTemplateId(alertConfig.getAlertTemplateId())
                    .putAllChannels(channelsMapping(alertConfig.getNotificationChannelsList()))
                    .addAllEscalations(alertConfig.getEscalationsList())
                    .addAllAlertThresholds(formatThresholds(alertConfig.getAlertThresholdsList()))
                    .build();
            list.add(templateConfigs);
        }
        return list;
    }

    public static CreateAlertsFromTemplateRequest toCreateAlertsFromTemplateRequest(
            StubRequest stubRequest,
            List<ru.yandex.monitoring.api.v3.Resource> resources,
            String createdBy)
    {
        MonitoringConfig config = Proto.unpack(stubRequest.stub(), MonitoringConfig.class);
        var convertedResources = convertResources(resources);
        var convertedIds = convertIds(config);
        var templateConfigs = convertTemplateConfigs(config);
        return CreateAlertsFromTemplateRequest
                .newBuilder()
                .addAllTemplateIds(convertedIds)
                .setServiceProviderId(stubRequest.serviceProviderId())
                .addAllResources(convertedResources)
                .setProjectId(config.getProjectId())
                .setCreatedBy(createdBy)
                .addAllTemplateConfigs(templateConfigs)
                .build();
    }

    public static List<CreateResourceRequest> toCreateResourceRequests(
            CompleteStubRequest completeStubRequest,
            ServiceProvider serviceProvider,
            String abcService,
            String projectId,
            ServiceProviderResourceSeverity severity)
    {
        List<CreateResourceRequest> requests = new ArrayList<>();
        for (var resource : completeStubRequest.getResourcesList()) {
            requests.add(CreateResourceRequest.newBuilder()
                    .setAbcSlug(abcService)
                    .setProjectId(projectId)
                    .setServiceProviderId(serviceProvider.getId())
                    .setType(resource.getResourceType())
                    .setResourceId(serviceProvider.isHasGlobalId() ? resource.getResourceId() : "")
                    .setSeverity(severity)
                    .putAllResourceComplexId(resource.getResourceParametersMap())
                    .build());
        }
        return requests;
    }
}
