package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import com.google.protobuf.util.Timestamps;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monitoring.api.v3.Alert;
import ru.yandex.monitoring.api.v3.CreateAlertRequest;
import ru.yandex.monitoring.api.v3.DeleteAlertRequest;
import ru.yandex.monitoring.api.v3.ExplainAlertEvaluationRequest;
import ru.yandex.monitoring.api.v3.ExplainAlertEvaluationResponse;
import ru.yandex.monitoring.api.v3.ExplainNewAlertEvaluationRequest;
import ru.yandex.monitoring.api.v3.ExplainNewAlertEvaluationResponse;
import ru.yandex.monitoring.api.v3.GetAlertEvaluationStateRequest;
import ru.yandex.monitoring.api.v3.GetAlertEvaluationStateResponse;
import ru.yandex.monitoring.api.v3.GetAlertEvaluationStatsRequest;
import ru.yandex.monitoring.api.v3.GetAlertEvaluationStatsResponse;
import ru.yandex.monitoring.api.v3.GetAlertNotificationStateRequest;
import ru.yandex.monitoring.api.v3.GetAlertNotificationStateResponse;
import ru.yandex.monitoring.api.v3.GetAlertNotificationStatsRequest;
import ru.yandex.monitoring.api.v3.GetAlertNotificationStatsResponse;
import ru.yandex.monitoring.api.v3.GetAlertRequest;
import ru.yandex.monitoring.api.v3.GetAlertStatsRequest;
import ru.yandex.monitoring.api.v3.GetAlertStatsResponse;
import ru.yandex.monitoring.api.v3.ListAlertsRequest;
import ru.yandex.monitoring.api.v3.ListAlertsResponse;
import ru.yandex.monitoring.api.v3.ListFullAlertsRequest;
import ru.yandex.monitoring.api.v3.ListFullAlertsResponse;
import ru.yandex.monitoring.api.v3.MuteAlertRequest;
import ru.yandex.monitoring.api.v3.UnmuteAlertRequest;
import ru.yandex.monitoring.api.v3.UpdateAlertRequest;
import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.gateway.endpoint.AlertServiceException;
import ru.yandex.solomon.alert.protobuf.EAlertState;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.alert.protobuf.TCreateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertRequest;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TListAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadProjectStatsRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.gateway.api.v3.intranet.AlertService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.AlertDtoConverter;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class AlertServiceImpl implements AlertService {
    private final Authorizer authorizer;
    private final AlertApi alertingClient;
    private final ProjectsDao projectsDao;

    @Autowired
    public AlertServiceImpl(Authorizer authorizer, AlertApi alertingClient, ProjectsDao projectsDao) {
        this.authorizer = authorizer;
        this.alertingClient = alertingClient;
        this.projectsDao = projectsDao;
    }

    @Override
    public CompletableFuture<Pair<Alert, Integer>> get(GetAlertRequest request, AuthSubject subject) {
        if (request.getContainerCase() != GetAlertRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(aVoid -> doGet(request));
    }

    private CompletableFuture<Pair<Alert, Integer>> doGet(GetAlertRequest request) {
        TReadAlertRequest readRequest = TReadAlertRequest.newBuilder()
                .setAlertId(request.getAlertId())
                .setProjectId(request.getProjectId())
                .build();

        return alertingClient.readAlert(readRequest)
                .thenApply(response -> {
                    ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                    TAlert alert = response.getAlert();
                    return Pair.of(AlertDtoConverter.fromProto(alert), alert.getVersion());
                });
    }

    @Override
    public CompletableFuture<ListAlertsResponse> list(ListAlertsRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_LIST)
                .thenCompose(aVoid -> doList(request));
    }

    private CompletableFuture<ListAlertsResponse> doList(ListAlertsRequest request) {
        TListAlertRequest listRequest = AlertDtoConverter.toProto(request);

        return alertingClient.listAlerts(listRequest).thenApply(response -> {
            ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
            return AlertDtoConverter.fromProto(response);
        });
    }

    @Override
    public CompletableFuture<ListFullAlertsResponse> listFull(ListFullAlertsRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_LIST)
                .thenCompose(aVoid -> doListFull(request));
    }

    private CompletableFuture<ListFullAlertsResponse> doListFull(ListFullAlertsRequest request) {
        TListAlertRequest listRequest = AlertDtoConverter.toProto(request);

        return alertingClient.listAlerts(listRequest).thenApply(response -> {
            ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
            return AlertDtoConverter.fromProtoFull(response);
        });
    }

    @Override
    public CompletableFuture<Alert> create(CreateAlertRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_CREATE)
                .thenCompose(aVoid -> doCreateAlert(request, subject));
    }

    private CompletableFuture<Alert> doCreateAlert(CreateAlertRequest request, AuthSubject subject) {
        Instant now = Instant.now();
        String login = subject.getUniqueId();

        TCreateAlertRequest createRequest = AlertDtoConverter.toProto(request, now, login);

        return checkProjectExistence(request.getProjectId())
                .thenCompose(aVoid -> alertingClient.createAlert(createRequest))
                .thenApply(response -> {
                    ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                    return AlertDtoConverter.fromProto(response);
                });
    }

    @Override
    public CompletableFuture<Alert> update(UpdateAlertRequest request, AuthSubject subject, int etag) {
        Instant now = Instant.now();
        String login = subject.getUniqueId();
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_CREATE)
                .thenCompose(aVoid -> doUpdateAlert(request, etag, login, now));
    }

    private CompletableFuture<Alert> doUpdateAlert(UpdateAlertRequest request, int etag, String login, Instant now) {
        TUpdateAlertRequest updateRequest = TUpdateAlertRequest.newBuilder()
                .setAlert(AlertDtoConverter.toProto(request, etag, login, now))
                .build();

        return checkProjectExistence(request.getProjectId())
                .thenCompose(aVoid -> alertingClient.updateAlert(updateRequest))
                .thenApply(response -> {
                    ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                    return AlertDtoConverter.fromProto(response);
                });
    }

    @Override
    public CompletableFuture<Empty> delete(DeleteAlertRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_CREATE)
                .thenCompose(aVoid -> doDeleteAlert(request));
    }

    private CompletableFuture<Empty> doDeleteAlert(DeleteAlertRequest request) {
        TDeleteAlertRequest deleteRequest = AlertDtoConverter.toProto(request);
        return alertingClient.deleteAlert(deleteRequest).thenApply(response -> {
            ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
            return Empty.getDefaultInstance();
        });
    }

    @Override
    public CompletableFuture<GetAlertEvaluationStatsResponse> getEvaluationStats(GetAlertEvaluationStatsRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(account -> doGetEvaluationStats(request));
    }

    private CompletableFuture<GetAlertEvaluationStatsResponse> doGetEvaluationStats(GetAlertEvaluationStatsRequest request) {
        TReadEvaluationStatsRequest readRequest = AlertDtoConverter.toProto(request);
        return alertingClient.readEvaluationStats(readRequest).thenApply(response -> {
            ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
            return AlertDtoConverter.fromProto(response);
        });
    }

    @Override
    public CompletableFuture<GetAlertNotificationStatsResponse> getNotificationStats(GetAlertNotificationStatsRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(account -> doGetNotificationStats(request));
    }

    private CompletableFuture<GetAlertNotificationStatsResponse> doGetNotificationStats(GetAlertNotificationStatsRequest request) {
        TReadNotificationStatsRequest readRequest = AlertDtoConverter.toProto(request);
        return alertingClient.readNotificationStats(readRequest).thenApply(response -> {
            ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
            return AlertDtoConverter.fromProto(response);
        });
    }

    @Override
    public CompletableFuture<GetAlertEvaluationStateResponse> getEvaluationState(GetAlertEvaluationStateRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(account -> doGetEvaluationState(request));
    }

    private CompletableFuture<GetAlertEvaluationStateResponse> doGetEvaluationState(GetAlertEvaluationStateRequest request) {
        TReadEvaluationStateRequest readRequest = AlertDtoConverter.toProto(request);

        return alertingClient.readEvaluationState(readRequest).thenApply(response -> {
            ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
            return AlertDtoConverter.fromProto(response);
        });
    }

    @Override
    public CompletableFuture<GetAlertNotificationStateResponse> getNotificationState(GetAlertNotificationStateRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(account -> doGetNotificationState(request));
    }

    private CompletableFuture<GetAlertNotificationStateResponse> doGetNotificationState(GetAlertNotificationStateRequest request) {
        TReadNotificationStateRequest readRequest = AlertDtoConverter.toProto(request);

        return alertingClient.readNotificationState(readRequest).thenApply(response -> {
            ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
            return AlertDtoConverter.fromAlertProto(response);
        });
    }

    @Override
    public CompletableFuture<ExplainAlertEvaluationResponse> explainEvaluation(ExplainAlertEvaluationRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(account -> doExplainEvaluation(request));
    }

    private CompletableFuture<ExplainAlertEvaluationResponse> doExplainEvaluation(ExplainAlertEvaluationRequest request) {
        long evaluationTimeMillis = Timestamps.toMillis(request.getTime());
        if (evaluationTimeMillis == 0) {
            evaluationTimeMillis = System.currentTimeMillis();
        }
        long deadlineMillis = System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30L);
        TExplainEvaluationRequest explainRequest = AlertDtoConverter.toProto(request, evaluationTimeMillis, deadlineMillis);

        return alertingClient.explainEvaluation(explainRequest)
                .thenApply(response -> {
                    ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                    return AlertDtoConverter.fromAlertProto(response);
                });
    }

    @Override
    public CompletableFuture<ExplainNewAlertEvaluationResponse> explainNewEvaluation(ExplainNewAlertEvaluationRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(account -> doExplainNewEvaluation(request));
    }

    private CompletableFuture<ExplainNewAlertEvaluationResponse> doExplainNewEvaluation(ExplainNewAlertEvaluationRequest request) {
        long evaluationTimeMillis = Timestamps.toMillis(request.getTime());
        if (evaluationTimeMillis == 0) {
            evaluationTimeMillis = System.currentTimeMillis();
        }

        long deadlineMillis = System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30L);

        TExplainEvaluationRequest explainEvaluationRequest = AlertDtoConverter.toProto(request, evaluationTimeMillis, deadlineMillis);

        return alertingClient.explainEvaluation(explainEvaluationRequest)
                .thenApply(response -> {
                    ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                    return AlertDtoConverter.fromProto(response);
                });
    }

    @Override
    public CompletableFuture<GetAlertStatsResponse> getStats(GetAlertStatsRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(account -> doGetStats(request));
    }

    private CompletableFuture<GetAlertStatsResponse> doGetStats(GetAlertStatsRequest request) {
        TReadProjectStatsRequest readRequest = AlertDtoConverter.toProto(request);

        return alertingClient.readProjectStats(readRequest).thenApply(response -> {
           ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
           return AlertDtoConverter.fromProto(response);
        });
    }

    @Override
    public CompletableFuture<Alert> mute(MuteAlertRequest request, AuthSubject subject, int etag) {
        if (request.getContainerCase() != MuteAlertRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_UPDATE)
                .thenCompose(account -> changeAlertState(request.getProjectId(), request.getAlertId(), etag, EAlertState.MUTED));
    }

    @Override
    public CompletableFuture<Alert> unmute(UnmuteAlertRequest request, AuthSubject subject, int etag) {
        if (request.getContainerCase() != UnmuteAlertRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_UPDATE)
                .thenCompose(account -> changeAlertState(request.getProjectId(), request.getAlertId(), etag, EAlertState.ACTIVE));
    }

    private CompletableFuture<Alert> changeAlertState(String projectId, String alertId, int etag, EAlertState state) {
        TReadAlertRequest readRequest = TReadAlertRequest.newBuilder()
                .setProjectId(projectId)
                .setAlertId(alertId)
                .build();

        return alertingClient.readAlert(readRequest)
                .thenCompose(response -> {
                    ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);

                    TAlert alert = response.getAlert();
                    if (etag != -1 && alert.getVersion() != etag) {
                        return CompletableFuture.failedFuture(new ConflictException("alert with version " + etag + " is out of date"));
                    }

                    TUpdateAlertRequest updateRequest = TUpdateAlertRequest.newBuilder()
                            .setAlert(alert.toBuilder().setState(state).build())
                            .build();

                    return alertingClient.updateAlert(updateRequest).thenApply(updateResponse -> {
                        ensureStatusValid(response.getRequestStatus(), response::getStatusMessage);
                        return AlertDtoConverter.fromProto(updateResponse);
                    });
                });
    }

    private CompletableFuture<Void> checkProjectExistence(String projectId) {
        return projectsDao.exists(projectId)
                .thenAccept(exists -> {
                    if (!exists) {
                        throw new BadRequestException(String.format("project %s does not exist", projectId));
                    }
                });
    }

    private void ensureStatusValid(ERequestStatusCode statusCode, Supplier<String> messageFn) {
        if (statusCode != ERequestStatusCode.OK) {
            throw new AlertServiceException(statusCode, messageFn.get());
        }
    }
}
