package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.monitoring.api.v3.AlertTemplate;
import ru.yandex.monitoring.api.v3.CreateAlertTemplateRequest;
import ru.yandex.monitoring.api.v3.DeployAlertTemplateRequest;
import ru.yandex.monitoring.api.v3.ListAlertTemplateRequest;
import ru.yandex.monitoring.api.v3.ListAlertTemplateResponse;
import ru.yandex.monitoring.api.v3.ListAlertTemplateVersionsRequest;
import ru.yandex.monitoring.api.v3.ListAlertTemplateVersionsResponse;
import ru.yandex.monitoring.api.v3.PublishAlertTemplateRequest;
import ru.yandex.monitoring.api.v3.ReadAlertTemplateRequest;
import ru.yandex.monitoring.api.v3.UnpublishAlertTemplateRequest;
import ru.yandex.solomon.alert.client.AlertTemplateApi;
import ru.yandex.solomon.alert.protobuf.DeployAlertTemplateResponse;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.SolomonTeam;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.gateway.api.v3.intranet.AlertTemplateService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.AlertTemplateDtoConverter;

/**
 * @author Alexey Trushkin
 */
@Component
@ParametersAreNonnullByDefault
public class AlertTemplateServiceImpl implements AlertTemplateService {

    private final Authorizer authorizer;
    private final AlertTemplateApi alertTemplateApi;
    private final AlertTemplateDtoConverter converter;

    public AlertTemplateServiceImpl(Authorizer authorizer, AlertTemplateApi alertTemplateApi) {
        this.authorizer = authorizer;
        this.alertTemplateApi = alertTemplateApi;
        this.converter = new AlertTemplateDtoConverter();
    }

    @Override
    public CompletableFuture<AlertTemplate> create(CreateAlertTemplateRequest request, AuthSubject subject) {
        return authorize(subject, request.getServiceProviderId(), Permission.ALERT_TEMPLATE_MANAGEMENT)
                .thenCompose(account -> alertTemplateApi.create(converter.toInternalProto(request))
                        .thenApply(response -> converter.fromInternalProto(response.getAlertTemplate())));
    }

    @Override
    public CompletableFuture<AlertTemplate> read(ReadAlertTemplateRequest request, AuthSubject subject) {
        return alertTemplateApi.read(converter.toInternalProto(request))
                .thenApply(response -> converter.fromInternalProto(response.getAlertTemplate()));
    }

    @Override
    public CompletableFuture<AlertTemplate> publish(PublishAlertTemplateRequest request, AuthSubject subject) {
        if (!SolomonTeam.isMember(subject)) {
            return CompletableFuture.failedFuture(new AuthorizationException("Haven't permission"));
        }
        return authorize(subject, request.getServiceProviderId(), Permission.ALERT_TEMPLATE_MANAGEMENT)
                .thenCompose(account -> alertTemplateApi.publish(converter.toInternalProto(request))
                        .thenApply(response -> converter.fromInternalProto(response.getAlertTemplate())));
    }

    @Override
    public CompletableFuture<Void> unpublish(UnpublishAlertTemplateRequest request, AuthSubject subject) {
        if (!SolomonTeam.isMember(subject)) {
            return CompletableFuture.failedFuture(new AuthorizationException("Haven't permission"));
        }
        return authorize(subject, request.getServiceProviderId(), Permission.ALERT_TEMPLATE_MANAGEMENT)
                .thenCompose(account -> alertTemplateApi.unpublish(converter.toInternalProto(request))
                        .thenApply(response -> null));
    }

    @Override
    public CompletableFuture<String> deploy(DeployAlertTemplateRequest request, AuthSubject subject) {
        if (!SolomonTeam.isMember(subject)) {
            return CompletableFuture.failedFuture(new AuthorizationException("Haven't permission"));
        }
        return authorize(subject, request.getServiceProviderId(), Permission.ALERT_TEMPLATE_MANAGEMENT)
                .thenCompose(account -> alertTemplateApi.deploy(converter.toInternalProto(request))
                        .thenApply(DeployAlertTemplateResponse::getTaskId));
    }

    @Override
    public CompletableFuture<ListAlertTemplateVersionsResponse> listTemplateVersions(
            ListAlertTemplateVersionsRequest request,
            AuthSubject subject)
    {
        return alertTemplateApi.listTemplateVersions(converter.toInternalProto(request))
                .thenApply(converter::fromInternalProto);
    }

    @Override
    public CompletableFuture<ListAlertTemplateResponse> listTemplates(
            ListAlertTemplateRequest request,
            AuthSubject subject)
    {
        return alertTemplateApi.list(converter.toInternalProto(request))
                .thenApply(converter::fromInternalProto);
    }

    private CompletableFuture<Account> authorize(AuthSubject subject, String serviceProviderId, Permission permission) {
        if (SolomonTeam.isMember(subject)) {
            return CompletableFuture.completedFuture(null);
        }
        return authorizer.authorize(subject, AuthorizationObject.serviceProvider(serviceProviderId, null), permission);
    }
}
