package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.util.concurrent.CompletableFuture;

import com.google.common.base.Throwables;

import ru.yandex.monitoring.api.v3.AlertingStubRequest;
import ru.yandex.monitoring.api.v3.CompleteStubRequest;
import ru.yandex.monitoring.api.v3.CompleteStubRequestResponse;
import ru.yandex.monitoring.api.v3.CreateStubRequest;
import ru.yandex.monitoring.api.v3.CreateStubRequestResponse;
import ru.yandex.monitoring.api.v3.GetStubRequest;
import ru.yandex.monitoring.api.v3.MonitoringConfig;
import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.gateway.endpoint.AlertServiceException;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.gateway.api.v3.intranet.AlertingStubRequestService;
import ru.yandex.solomon.gateway.api.v3.intranet.ResourceService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.StubRequestConverter;
import ru.yandex.solomon.gateway.api.v3.utils.StubRequestIdempotency;
import ru.yandex.solomon.gateway.stub.StubRequest;
import ru.yandex.solomon.gateway.stub.dao.StubRequestDao;
import ru.yandex.solomon.idempotency.IdempotentOperationExistException;
import ru.yandex.solomon.idempotency.IdempotentOperationService;
import ru.yandex.solomon.util.Proto;

import static ru.yandex.solomon.gateway.api.v3.intranet.dto.StubRequestConverter.extractStubRequest;
import static ru.yandex.solomon.gateway.api.v3.intranet.dto.StubRequestConverter.toAlertingStubRequest;
import static ru.yandex.solomon.gateway.api.v3.intranet.dto.StubRequestConverter.toCompleteStubRequestResponse;
import static ru.yandex.solomon.gateway.api.v3.intranet.dto.StubRequestConverter.toCreateAlertsFromTemplateRequest;
import static ru.yandex.solomon.gateway.api.v3.intranet.dto.StubRequestConverter.toCreateStubRequestResponse;
import static ru.yandex.solomon.gateway.api.v3.intranet.validators.AlertingStubRequestValidator.validate;
import static ru.yandex.solomon.gateway.api.v3.utils.StubRequestIdempotency.CREATE_STUB_REQUEST_OPERATION;
import static ru.yandex.solomon.gateway.api.v3.utils.StubRequestIdempotency.DELETE_STUB_REQUEST_OPERATION;

/**
 * @author Nuradil Zhambyl
 */
public class AlertingStubRequestServiceImpl implements AlertingStubRequestService {
    private final StubRequestDao dao;
    private final IdempotentOperationService idempotentOperationService;
    private final SolomonConfHolder confHolder;
    private final Authorizer authorizer;
    private final AlertApi alertApi;
    private final ResourceService resourceService;

    public AlertingStubRequestServiceImpl(
            StubRequestDao dao,
            IdempotentOperationService idempotentOperationService,
            SolomonConfHolder confHolder,
            Authorizer authorizer,
            AlertApi alertApi,
            ResourceService resourceService)
    {
        this.dao = dao;
        this.idempotentOperationService = idempotentOperationService;
        this.confHolder = confHolder;
        this.authorizer = authorizer;
        this.alertApi = alertApi;
        this.resourceService = resourceService;
    }

    @Override
    public CompletableFuture<AlertingStubRequest> get(GetStubRequest request, AuthSubject authSubject) {
        validate(request);
        return dao.get(request.getId())
                .thenApply(entityOpt -> entityOpt.orElseThrow(
                        () -> stubRequestNotFound(request.getId())))
                .thenApply(StubRequestConverter::toAlertingStubRequest)
                .thenCompose(alertingStubRequest -> authorizer.authorize(authSubject, alertingStubRequest.getStub().getProjectId(), Permission.CONFIGS_GET)
                        .thenApply(unused -> alertingStubRequest));
    }

    @Override
    public CompletableFuture<CompleteStubRequestResponse> complete(CompleteStubRequest request, AuthSubject authSubject) {
        validate(request);
        return idempotentOperationService.get(request.getId(), request.getId(), ContainerType.SERVICE_PROVIDER, DELETE_STUB_REQUEST_OPERATION)
                .thenCompose(operationOptional -> {
                    if (operationOptional.isPresent()) {
                        // operation's presence implies earlier deletion
                        return CompletableFuture.completedFuture(toCompleteStubRequestResponse(request.getId()));
                    }
                    return dao.get(request.getId())
                            .thenApply(entityOpt -> entityOpt.orElseThrow(
                                    () -> stubRequestNotFound(request.getId())))
                            .thenCompose(stubRequest -> {
                                var alertingStubRequest = toAlertingStubRequest(stubRequest);
                                return authorizer.authorize(authSubject, AuthorizationObject.serviceProvider(alertingStubRequest.getServiceProviderId(), alertingStubRequest.getStub().getProjectId()), Permission.ALERT_MANAGEMENT)
                                        .thenCompose(unused -> doComplete(stubRequest, request, authSubject.getUniqueId()));
                            });
                });
    }

    private CompletableFuture<CompleteStubRequestResponse> doComplete(StubRequest stubRequest, CompleteStubRequest completeStubRequest, String createdBy) {
        var request = toCreateAlertsFromTemplateRequest(stubRequest, completeStubRequest.getResourcesList(), createdBy);
        MonitoringConfig config = Proto.unpack(stubRequest.stub(), MonitoringConfig.class);
        Project project = confHolder.getConfOrThrow().getProject(request.getProjectId());
        if (project == null) {
            return CompletableFuture.failedFuture(new BadRequestException("No project '" + request.getProjectId() + "'"));
        }
        return resourceService.upsert(StubRequestConverter.toCreateResourceRequests(completeStubRequest, confHolder.getConf().getServiceProvider(request.getServiceProviderId()), project.getAbcService(), request.getProjectId(), config.getSeverity()), false, "")
                .thenCompose(unused1 -> alertApi.createAlerts(request)
                .thenCompose(response -> {
                    if (response.getRequestStatusCode() != ERequestStatusCode.OK) {
                        return CompletableFuture.failedFuture(new AlertServiceException(response.getRequestStatusCode(), "alerts creation failed: " + response.getStatusMessage()));
                    }
                    var op = StubRequestIdempotency.operationDelete(stubRequest);
                    return idempotentOperationService.delete(stubRequest.id(), stubRequest.serviceProviderId(), ContainerType.SERVICE_PROVIDER, CREATE_STUB_REQUEST_OPERATION)
                            .thenCompose(negligible -> dao.deleteOne(stubRequest.id(), op)
                                    .handle((unused, throwable) -> handleDeleteOneResult(throwable, stubRequest.id())));
                }));
    }

    @Override
    public CompletableFuture<CreateStubRequestResponse> create(CreateStubRequest request, AuthSubject authSubject) {
        validate(request, confHolder);
        return authorizer.authorize(authSubject, request.getStub().getProjectId(), Permission.CONFIGS_CREATE)
                .thenCompose(ignore -> idempotentOperationService.get(request.getId(), request.getServiceProviderId(), ContainerType.SERVICE_PROVIDER, CREATE_STUB_REQUEST_OPERATION)
                        .thenCompose(operationOptional -> {
                            if (operationOptional.isPresent()) {
                                // operation's presence implies earlier creation
                                return CompletableFuture.completedFuture(toCreateStubRequestResponse(request.getId()));
                            }
                            var op = StubRequestIdempotency.operationCreate(request);
                            var stubRequest = extractStubRequest(request);
                            return idempotentOperationService.delete(stubRequest.id(), stubRequest.id(), ContainerType.SERVICE_PROVIDER, DELETE_STUB_REQUEST_OPERATION)
                                    .thenCompose(negligible -> dao.insert(stubRequest, op)
                                            .handle((unused, throwable) -> handleInsertResult(throwable, request.getId())));
                        }));
    }

    private CreateStubRequestResponse handleInsertResult(Throwable throwable, String id) {
        if (throwable != null) {
            var cause = Throwables.getRootCause(throwable);
            if (cause instanceof IdempotentOperationExistException) {
                return toCreateStubRequestResponse(id);
            }
            Throwables.propagate(throwable);
        }
        return toCreateStubRequestResponse(id);
    }

    private CompleteStubRequestResponse handleDeleteOneResult(Throwable throwable, String id) {
        if (throwable != null) {
            var cause = Throwables.getRootCause(throwable);
            if (cause instanceof IdempotentOperationExistException) {
                return toCompleteStubRequestResponse(id);
            }
            Throwables.propagate(throwable);
        }
        return toCompleteStubRequestResponse(id);
    }

    private static NotFoundException stubRequestNotFound(String stubRequestId) {
        return new NotFoundException(String.format("no stub request with id %s found", stubRequestId));
    }
}
