package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monitoring.api.v3.Channel;
import ru.yandex.monitoring.api.v3.CreateChannelRequest;
import ru.yandex.monitoring.api.v3.DeleteChannelRequest;
import ru.yandex.monitoring.api.v3.GetChannelRequest;
import ru.yandex.monitoring.api.v3.ListChannelsRequest;
import ru.yandex.monitoring.api.v3.ListChannelsResponse;
import ru.yandex.monitoring.api.v3.UpdateChannelRequest;
import ru.yandex.solomon.alert.client.AlertingClient;
import ru.yandex.solomon.alert.protobuf.TCreateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TListNotificationsRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.notification.TNotification;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.gateway.api.v3.intranet.ChannelService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.ChannelDtoConverter;

import static ru.yandex.solomon.gateway.api.cloud.v1.RequestStatusToAlertingException.throwIfNotOk;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class ChannelServiceImpl implements ChannelService {
    private final Authorizer authorizer;
    private final AlertingClient alertingClient;
    private final ProjectsDao projectsDao;

    @Autowired
    public ChannelServiceImpl(Authorizer authorizer, AlertingClient alertingClient, ProjectsDao projectsDao) {
        this.authorizer = authorizer;
        this.alertingClient = alertingClient;
        this.projectsDao = projectsDao;
    }

    @Override
    public CompletableFuture<Pair<Channel, Integer>> get(GetChannelRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(account -> doGet(request));
    }

    private CompletableFuture<Pair<Channel, Integer>> doGet(GetChannelRequest request) {
        TReadNotificationRequest readRequest = ChannelDtoConverter.toProto(request);

        return alertingClient.readNotification(readRequest).thenApply(response -> {
            throwIfNotOk(response.getRequestStatus(), response::getStatusMessage);
            TNotification channel = response.getNotification();
            return Pair.of(ChannelDtoConverter.fromProto(channel), channel.getVersion());
        });
    }

    @Override
    public CompletableFuture<ListChannelsResponse> list(ListChannelsRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_LIST)
                .thenCompose(account -> checkProjectExistence(request.getProjectId()))
                .thenCompose(aVoid -> doList(request));
    }

    private CompletableFuture<ListChannelsResponse> doList(ListChannelsRequest request) {
        TListNotificationsRequest listRequest = ChannelDtoConverter.toProto(request);

        return alertingClient.listNotification(listRequest)
                .thenApply(response -> {
                    throwIfNotOk(response.getRequestStatus(), response::getStatusMessage);
                    return ChannelDtoConverter.fromProto(response);
                });
    }

    @Override
    public CompletableFuture<Channel> create(CreateChannelRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_CREATE)
                .thenCompose(account -> checkProjectExistence(request.getProjectId()))
                .thenCompose(aVoid -> doCreate(request, subject));
    }

    private CompletableFuture<Channel> doCreate(CreateChannelRequest request, AuthSubject subject) {
        Instant now = Instant.now();
        String login = subject.getUniqueId();

        TCreateNotificationRequest createRequest = ChannelDtoConverter.toProto(request, now, login);

        return alertingClient.createNotification(createRequest).thenApply(response -> {
          throwIfNotOk(response.getRequestStatus(), response::getStatusMessage);
          return ChannelDtoConverter.fromProto(response.getNotification());
       });
    }

    @Override
    public CompletableFuture<Channel> update(UpdateChannelRequest request, AuthSubject subject, int etag) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_UPDATE)
                .thenCompose(account -> checkProjectExistence(request.getProjectId()))
                .thenCompose(aVoid -> doUpdate(request, subject, etag));
    }

    private CompletableFuture<Channel> doUpdate(UpdateChannelRequest request, AuthSubject subject, int etag) {
        Instant now = Instant.now();
        String login = subject.getUniqueId();

        TUpdateNotificationRequest updateRequest = ChannelDtoConverter.toProto(request, etag, now, login);

        return alertingClient.updateNotification(updateRequest).thenApply(response -> {
            throwIfNotOk(response.getRequestStatus(), response::getStatusMessage);
            return ChannelDtoConverter.fromProto(response.getNotification());
        });
    }

    @Override
    public CompletableFuture<Empty> delete(DeleteChannelRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_DELETE)
                .thenCompose(account -> doDelete(request));
    }

    private CompletableFuture<Empty> doDelete(DeleteChannelRequest request) {
        TDeleteNotificationRequest deleteRequest = ChannelDtoConverter.toProto(request);

        return alertingClient.deleteNotification(deleteRequest).thenApply(response -> {
            throwIfNotOk(response.getRequestStatus(), response::getStatusMessage);
            return Empty.getDefaultInstance();
        });
    }

    private CompletableFuture<Void> checkProjectExistence(String projectId) {
        return projectsDao.exists(projectId)
                .thenAccept(exists -> {
                    if (!exists) {
                        throw new BadRequestException(String.format("project %s does not exist", projectId));
                    }
                });
    }
}
