package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monitoring.api.v3.Cluster;
import ru.yandex.monitoring.api.v3.CreateClusterRequest;
import ru.yandex.monitoring.api.v3.DeleteClusterRequest;
import ru.yandex.monitoring.api.v3.GetClusterRequest;
import ru.yandex.monitoring.api.v3.ListClusterServicesRequest;
import ru.yandex.monitoring.api.v3.ListClusterServicesResponse;
import ru.yandex.monitoring.api.v3.ListClustersRequest;
import ru.yandex.monitoring.api.v3.ListClustersResponse;
import ru.yandex.monitoring.api.v3.UpdateClusterRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.cloud.resource.resolver.FolderResolver;
import ru.yandex.solomon.cloud.resource.resolver.FolderResolverNoOp;
import ru.yandex.solomon.core.conf.ClustersManager;
import ru.yandex.solomon.gateway.api.v3.intranet.ClusterService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.ClusterDtoConverter;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.ShardAssociationDtoConverter;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class ClusterServiceImpl implements ClusterService {
    private final Authorizer authorizer;
    private final ClustersManager clustersManager;
    private final FolderResolver folderResolver;

    @Autowired
    public ClusterServiceImpl(
        Authorizer authorizer,
        ClustersManager clustersManager,
        Optional<FolderResolver> folderResolver)
    {
        this.authorizer = authorizer;
        this.clustersManager = clustersManager;
        this.folderResolver = folderResolver.orElseGet(FolderResolverNoOp::new);
    }

    @Override
    public CompletableFuture<Pair<Cluster, Integer>> get(GetClusterRequest request, AuthSubject subject) {
        if (request.getContainerCase() != GetClusterRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                .thenCompose(account -> doGet(request));
    }

    private CompletableFuture<Pair<Cluster, Integer>> doGet(GetClusterRequest request) {
        return clustersManager.getCluster(request.getProjectId(), "", request.getClusterId())
                .thenApply(ClusterDtoConverter::fromModelWithVersion);
    }

    @Override
    public CompletableFuture<ListClustersResponse> list(ListClustersRequest request, AuthSubject subject) {
        if (request.getContainerCase() != ListClustersRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_LIST)
                .thenCompose(account -> doList(request));
    }

    private CompletableFuture<ListClustersResponse> doList(ListClustersRequest request) {
        return clustersManager.getPagedClusters(
                request.getProjectId(),
                "",
                (int) request.getPageSize(),
                request.getPageToken(),
                request.getFilter())
                .thenApply(response -> {
                    var pagedResult = response.map(ClusterDtoConverter::fromModel);
                    return ListClustersResponse.newBuilder()
                            .addAllClusters(pagedResult.getItems())
                            .setNextPageToken(pagedResult.getNextPageToken())
                            .build();
                });
    }

    @Override
    public CompletableFuture<Cluster> create(CreateClusterRequest request, AuthSubject subject) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_CREATE)
            .thenCombine(resolveFolderId(request.getProjectId(), request.getLabelName()), (acc, folderId) -> folderId)
            .thenCompose(folderId -> doCreate(request, subject, folderId));
    }

    private CompletableFuture<Cluster> doCreate(
        CreateClusterRequest request,
        AuthSubject subject,
        String folderId)
    {
        Instant now = Instant.now();
        var cluster = ClusterDtoConverter.toModel(request, subject.getUniqueId(), now, folderId);
        return clustersManager.createCluster(cluster)
                .thenApply(ClusterDtoConverter::fromModel);
    }

    @Override
    public CompletableFuture<Cluster> update(UpdateClusterRequest request, AuthSubject subject, int etag) {
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_UPDATE)
            .thenCombine(resolveFolderId(request.getProjectId(), request.getLabelName()), (acc, folderId) -> folderId)
            .thenCompose(folderId -> doUpdate(request, subject.getUniqueId(), etag, folderId));
    }

    private CompletableFuture<Cluster> doUpdate(
        UpdateClusterRequest request,
        String login,
        int etag,
        String folderId)
    {
        Instant now = Instant.now();
        var cluster = ClusterDtoConverter.toModel(request, login, now, etag, folderId);
        return clustersManager.updateCluster(cluster)
                .thenApply(ClusterDtoConverter::fromModel);
    }

    @Override
    public CompletableFuture<Empty> delete(DeleteClusterRequest request, AuthSubject subject) {
        if (request.getContainerCase() != DeleteClusterRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_DELETE)
                .thenCompose(account -> doDelete(request));
    }

    private CompletableFuture<Empty> doDelete(DeleteClusterRequest request) {
        return clustersManager.deleteCluster(request.getProjectId(), "", request.getClusterId())
                .thenApply(aVoid -> Empty.getDefaultInstance());
    }

    @Override
    public CompletableFuture<ListClusterServicesResponse> listServices(
            ListClusterServicesRequest request,
            AuthSubject subject)
    {
        if (request.getContainerCase() != ListClusterServicesRequest.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type " + request.getContainerCase());
        }
        return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_LIST)
                .thenCompose(account -> doListServices(request));
    }

    private CompletableFuture<ListClusterServicesResponse> doListServices(ListClusterServicesRequest request) {
        String projectId = request.getProjectId();

        var future = clustersManager.getClusterAssociations(
                projectId,
                "",
                request.getClusterId(),
                ShardAssociationDtoConverter.toModel(request.getFilterByModesList()),
                request.getFilterByName(),
                (int) request.getPageSize(),
                request.getPageToken());

        return future.thenApply(response -> ShardAssociationDtoConverter.fromClusterServicesModel(projectId, response));
    }

    private CompletableFuture<String> resolveFolderId(String projectId, String clusterName) {
        return folderResolver.resolveFolderId(projectId, clusterName);
    }
}
