package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monitoring.api.v3.CreateDashboardRequest;
import ru.yandex.monitoring.api.v3.Dashboard;
import ru.yandex.monitoring.api.v3.DeleteDashboardRequest;
import ru.yandex.monitoring.api.v3.GetDashboardRequest;
import ru.yandex.monitoring.api.v3.ListDashboardsRequest;
import ru.yandex.monitoring.api.v3.ListDashboardsResponse;
import ru.yandex.monitoring.api.v3.UpdateDashboardRequest;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolver;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolverNoOp;
import ru.yandex.solomon.conf.db3.MonitoringDashboardsDao;
import ru.yandex.solomon.conf.db3.ydb.Entity;
import ru.yandex.solomon.config.gateway.TGatewayCloudConfig;
import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.gateway.api.utils.IdGenerator;
import ru.yandex.solomon.gateway.api.v3.intranet.DashboardService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.DashboardDtoConverter;
import ru.yandex.solomon.gateway.api.v3.intranet.validators.DashboardValidator;
import ru.yandex.solomon.gateway.cloud.search.NoopSearchEventSink;
import ru.yandex.solomon.gateway.cloud.search.SearchEvent;
import ru.yandex.solomon.gateway.cloud.search.SearchEventSink;
import ru.yandex.solomon.labels.query.Selectors;

import static ru.yandex.solomon.gateway.api.v3.intranet.dto.DashboardDtoConverter.parseVersion;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class DashboardServiceImpl implements DashboardService {
    private final Authorizer authorizer;
    private final MonitoringDashboardsDao dashboardsDao;
    private final CloudByFolderResolver cloudByFolderResolver;
    private final String entityIdPrefix;
    private final SearchEventSink searchEventSink;

    @Autowired
    public DashboardServiceImpl(
            Authorizer authorizer,
            MonitoringDashboardsDao dashboardsDao,
            Optional<CloudByFolderResolver> cloudByFolderResolverOptional,
            Optional<TGatewayCloudConfig> config,
            Optional<SearchEventSink> searchEventSink)
    {
        this.authorizer = authorizer;
        this.dashboardsDao = dashboardsDao;
        this.cloudByFolderResolver = cloudByFolderResolverOptional.orElseGet(CloudByFolderResolverNoOp::new);
        this.entityIdPrefix = config.map(TGatewayCloudConfig::getEntityIdPrefix).orElse("");
        this.searchEventSink = searchEventSink.orElseGet(NoopSearchEventSink::new);
    }

    @Override
    public CompletableFuture<Dashboard> get(GetDashboardRequest request, AuthSubject subject) {
        DashboardValidator.validate(request);
        return dashboardsDao.readById(request.getDashboardId())
                .thenApply(entityOpt -> entityOpt.orElseThrow(() -> dashboardNotFound(request.getDashboardId())))
                .thenApply(DashboardDtoConverter::fromEntity)
                .thenCompose(dashboard -> authorize(subject, AuthInfo.of(dashboard), Permission.CONFIGS_GET)
                        .thenApply(unused -> dashboard));
    }

    @Override
    public CompletableFuture<ListDashboardsResponse> list(ListDashboardsRequest request, AuthSubject subject) {
        DashboardValidator.validate(request);
        var nameFilter = request.getFilter().isEmpty()
                ? ""
                : Selectors.parse(request.getFilter()).findByKey("name").getValue();
        String parentId = switch (request.getContainerCase()) {
            case PROJECT_ID -> request.getProjectId();
            case FOLDER_ID -> request.getFolderId();
            default -> throw new UnsupportedOperationException("Unsupported container type");
        };
        return authorize(subject, AuthInfo.of(request), Permission.CONFIGS_LIST)
                .thenCompose(unused -> dashboardsDao.listByLocalId(parentId, nameFilter, (int) request.getPageSize(), request.getPageToken())
                        .thenApply(DashboardDtoConverter::fromEntity));
    }

    @Override
    public CompletableFuture<Dashboard> create(CreateDashboardRequest request, AuthSubject subject) {
        DashboardValidator.validate(request);
        return authorize(subject, AuthInfo.of(request), Permission.CONFIGS_UPDATE)
                .thenCompose(authResult -> doCreate(request, authResult));
    }

    private CompletableFuture<Dashboard> doCreate(CreateDashboardRequest request, AuthResult authResult) {
        var dashboard = DashboardDtoConverter.toDashboard(request, generate(), authResult.account.getId());
        var entity = DashboardDtoConverter.toEntity(dashboard, 0);
        return dashboardsDao.insert(entity)
                .thenApply(inserted -> {
                    if (!inserted) {
                        throw new ConflictException("dashboard '" + dashboard.getId() + "' already exists");
                    }
                    searchEventSink.accept(SearchEvent.dashboard(authResult.cloudId, entity));
                    return dashboard;
                });
    }

    @Override
    public CompletableFuture<Dashboard> update(UpdateDashboardRequest request, AuthSubject subject) {
        DashboardValidator.validate(request);
        return dashboardsDao.readById(request.getDashboardId())
                .thenApply(entityOpt -> entityOpt.orElseThrow(() -> dashboardNotFound(request.getDashboardId())))
                .thenApply(DashboardDtoConverter::fromEntity)
                .thenCompose(dashboard -> authorize(subject, AuthInfo.of(dashboard), Permission.CONFIGS_UPDATE)
                        .thenCompose(authResult -> doUpdate(request, authResult, dashboard)));
    }

    private CompletableFuture<Dashboard> doUpdate(
            UpdateDashboardRequest request,
            AuthResult authResult,
            Dashboard previousVersion)
    {
        int currentVersion = parseVersion(request.getEtag());
        int newVersion = parseVersion(previousVersion.getEtag()) + 1;
        String newEtag = String.valueOf(newVersion);
        var dashboard = DashboardDtoConverter.toDashboard(request, authResult.account.getId(), previousVersion, newEtag);
        var entity = DashboardDtoConverter.toEntity(dashboard, currentVersion);
        return dashboardsDao.update(entity)
                .thenCompose(updatedDashboardOpt -> checkExistence(entity, updatedDashboardOpt))
                .thenApply(updated -> {
                    searchEventSink.accept(SearchEvent.dashboard(authResult.cloudId, updated));
                    return updated;
                })
                .thenApply(DashboardDtoConverter::fromEntity);
    }

    @Override
    public CompletableFuture<Empty> delete(DeleteDashboardRequest request, AuthSubject subject) {
        DashboardValidator.validate(request);
        return dashboardsDao.readById(request.getDashboardId())
                .thenApply(entityOpt -> entityOpt.orElseThrow(() -> dashboardNotFound(request.getDashboardId())))
                .thenCompose(entity -> {
                    var dashboard = DashboardDtoConverter.fromEntity(entity);
                    return authorize(subject, AuthInfo.of(dashboard), Permission.CONFIGS_DELETE)
                            .thenCompose(authResult -> dashboardsDao.deleteWithVersion(entity.getParentId(), request.getDashboardId(), parseVersion(request.getEtag()))
                                    .thenCompose(deleted -> {
                                        if (!deleted) {
                                            return checkExistence(entity, Optional.empty())
                                                    .thenApply(unused -> Empty.getDefaultInstance());
                                        }
                                        searchEventSink.accept(SearchEvent.dashboard(authResult.cloudId, entity).deletedAt(System.currentTimeMillis()));
                                        return CompletableFuture.completedFuture(Empty.getDefaultInstance());
                                    }));
                });
    }

    private CompletableFuture<AuthResult> authorize(AuthSubject subject, AuthInfo authInfo, Permission permission) {
        switch (authInfo.containerType) {
            case PROJECT -> {
                return authorizer.authorize(subject, authInfo.containerId(), permission)
                        .thenApply(account -> new AuthResult(account, ""));
            }
            case FOLDER -> {
                return cloudByFolderResolver.resolveCloudId(authInfo.containerId())
                        .thenCompose(cloudId -> authorizer.authorize(subject, cloudId, authInfo.containerId(), permission)
                                .thenApply(account -> new AuthResult(account, cloudId)));
            }
        }
        return CompletableFuture.failedFuture(new UnsupportedOperationException("Unsupported container type '" + authInfo.containerType + "'"));
    }

    private CompletableFuture<Entity> checkExistence(
            Entity dashboard,
            Optional<Entity> updatedDashboardOpt)
    {
        if (updatedDashboardOpt.isPresent()) {
            return CompletableFuture.completedFuture(updatedDashboardOpt.get());
        }

        String parentId = dashboard.getParentId();
        String dashboardId = dashboard.getId();

        return dashboardsDao.exists(parentId, dashboardId)
                .thenApply(exists -> {
                    if (exists) {
                        String message = String.format(
                                "dashboard '%s' with etag '%s' is out of date",
                                dashboardId,
                                dashboard.getVersion()
                        );
                        throw new ConflictException(message);
                    }
                    throw dashboardNotFound(dashboardId);
                });
    }

    private static NotFoundException dashboardNotFound(String dashboardId) {
        return new NotFoundException(String.format("no dashboard with id %s found", dashboardId));
    }

    private String generate() {
        return entityIdPrefix.isEmpty()
                ? IdGenerator.generateInternalId()
                : IdGenerator.generateId(entityIdPrefix);
    }

    static record AuthResult(Account account, String cloudId) {

    }

    static record AuthInfo(String containerId, ContainerType containerType) {

        public static AuthInfo of(Dashboard dashboard) {
            switch (dashboard.getContainerCase()) {
                case PROJECT_ID -> {
                    return new AuthInfo(dashboard.getProjectId(), ContainerType.PROJECT);
                }
                case FOLDER_ID -> {
                    return new AuthInfo(dashboard.getFolderId(), ContainerType.FOLDER);
                }
            }
            throw new UnsupportedOperationException("Unsupported container type");
        }

        public static AuthInfo of(ListDashboardsRequest request) {
            switch (request.getContainerCase()) {
                case PROJECT_ID -> {
                    return new AuthInfo(request.getProjectId(), ContainerType.PROJECT);
                }
                case FOLDER_ID -> {
                    return new AuthInfo(request.getFolderId(), ContainerType.FOLDER);
                }
            }
            throw new UnsupportedOperationException("Unsupported container type");
        }

        public static AuthInfo of(CreateDashboardRequest request) {
            switch (request.getContainerCase()) {
                case PROJECT_ID -> {
                    return new AuthInfo(request.getProjectId(), ContainerType.PROJECT);
                }
                case FOLDER_ID -> {
                    return new AuthInfo(request.getFolderId(), ContainerType.FOLDER);
                }
            }
            throw new UnsupportedOperationException("Unsupported container type");
        }
    }
}
