package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.util.Comparator;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.monitoring.api.v3.DataType;
import ru.yandex.monitoring.api.v3.ListFunctionsRequest;
import ru.yandex.monitoring.api.v3.ListFunctionsResponse;
import ru.yandex.monitoring.api.v3.ParseExpressionRequest;
import ru.yandex.monitoring.api.v3.ParseExpressionResponse;
import ru.yandex.solomon.expression.SelParser;
import ru.yandex.solomon.expression.analytics.ProgramWithReturn;
import ru.yandex.solomon.expression.exceptions.ParserException;
import ru.yandex.solomon.expression.expr.SelFunctions;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.gateway.api.v3.intranet.ExpressionService;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.ExpressionDtoConverter;

/**
 * @author Oleg Baryshnikov
 */
@Component
@ParametersAreNonnullByDefault
public class ExpressionServiceImpl implements ExpressionService {

    @Override
    public ListFunctionsResponse listFunctions(ListFunctionsRequest request) {
        SelVersion version = ExpressionDtoConverter.parseVersion(request.getVersion());
        DataType input = request.getFilterByInputType();
        DataType output = request.getFilterByInputType();
        SelFuncCategory category = ExpressionDtoConverter.parseFuncCategory(request.getFilterByCategory());

        var functions = SelFunctions.REGISTRY.stream(version)
                .sorted(Comparator.comparing(SelFunc::getName))
                .filter(func -> category == null || category == func.getCategory())
                .map(ExpressionDtoConverter::fromModel)
                .filter(Objects::nonNull)
                .filter(func -> output == DataType.DATA_TYPE_UNSPECIFIED || func.getReturnType() == request.getFilterByOutputType())
                .filter(func -> input == DataType.DATA_TYPE_UNSPECIFIED || func.getArgsList().stream().anyMatch(arg -> arg.getType() == input))
                .collect(Collectors.toList());

        return ListFunctionsResponse.newBuilder()
                .addAllFunctions(functions)
                .build();
    }

    @Override
    public ParseExpressionResponse parse(ParseExpressionRequest request) {
        try {
            ProgramWithReturn programWithReturn = new SelParser(request.getProgramCode(), true)
                    .parseProgramWithReturn();
            return ExpressionDtoConverter.fromModel(programWithReturn, request.getNeedPosition());
        } catch (ParserException e) {
            return ExpressionDtoConverter.fromModel(e);
        }
    }

}
